/*******************************************************************************
 * Copyright (c) 2012, 2014 Pivotal Software, Inc. 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Apache License v2.0 which accompanies this distribution. 
 * 
 * The Eclipse Public License is available at 
 * 
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * and the Apache License v2.0 is available at 
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * You may elect to redistribute this code under either of these licenses.
 *  
 *  Contributors:
 *     Pivotal Software, Inc. - initial API and implementation
 ********************************************************************************/
package org.eclipse.cft.server.ui.internal.wizards;

import java.util.List;

import org.eclipse.cft.server.core.internal.CloudFoundryPlugin;
import org.eclipse.cft.server.core.internal.CloudFoundryServer;
import org.eclipse.cft.server.core.internal.client.CloudFoundryServerBehaviour;
import org.eclipse.cft.server.core.internal.spaces.CloudFoundrySpace;
import org.eclipse.cft.server.ui.internal.CloudFoundryCredentialsPart;
import org.eclipse.cft.server.ui.internal.CloudFoundryServerUiPlugin;
import org.eclipse.cft.server.ui.internal.CloudServerSpacesDelegate;
import org.eclipse.cft.server.ui.internal.Messages;
import org.eclipse.cft.server.ui.internal.ServerWizardValidator;
import org.eclipse.cft.server.ui.internal.ValidationEventHandler;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.ServerCore;
import org.eclipse.wst.server.core.TaskModel;
import org.eclipse.wst.server.core.internal.ServerWorkingCopy;
import org.eclipse.wst.server.core.util.ServerLifecycleAdapter;
import org.eclipse.wst.server.ui.wizard.IWizardHandle;
import org.eclipse.wst.server.ui.wizard.WizardFragment;

/**
 * IMPORTANT NOTE: This class can be referred by the branding extension from adopter so this class 
 * should not be moved or renamed to avoid breakage to adopters.
 * 
 * @author Christian Dupuis
 * @author Steffen Pingel
 * @author Terry Denney
 */
@SuppressWarnings("restriction")
public class CloudFoundryServerWizardFragment extends WizardFragment {

	private CloudFoundryServer cloudServer;

	private CloudFoundryCredentialsPart credentialsPart;

	private CloudFoundrySpacesWizardFragment spacesFragment;

	private ValidationEventHandler validationHandler;

	private WizardFragmentStatusHandler statusHandler;

	private IRunnableContext context;

	@Override
	public Composite createComposite(Composite parent, IWizardHandle wizardHandle) {
		initServer();
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		context = wizardHandle != null ? new WizardHandleContext(wizardHandle).getRunnableContext() : null;

		// Performs validation on credentials and cloud spaces when creating a
		// new server.
		validationHandler = new ValidationEventHandler();

		// Status handler that displays status to the wizard
		statusHandler = new WizardFragmentStatusHandler(wizardHandle);

		credentialsPart = new CloudFoundryCredentialsPart(cloudServer, new WizardHandleContext(wizardHandle));

		// Changes in the credentials part (e.g. username, password, server URL)
		// should
		// notify the validation handler so that it can perform validations
		credentialsPart.addPartChangeListener(validationHandler);

		// The credentials part should be notified when validations are
		// complete, as to enable/disable UI components accordingly
		validationHandler.addValidationListener(credentialsPart);

		// Must wire the wizard status handler to the validation handler, so
		// that the wizard
		// can be notified when to display validation status or errors.
		validationHandler.addStatusHandler(statusHandler);

		credentialsPart.createPart(composite);

		return composite;
	}

	@Override
	public void enter() {
		initServer();

		// New validator should be created as the cloud server may have changed.

		ServerWizardValidator validator = new CredentialsWizardValidator(cloudServer, new CloudServerSpacesDelegate(
				cloudServer));
		if (validationHandler != null) {
			validationHandler.updateValidator(validator);
		}

		spacesFragment = new CloudFoundrySpacesWizardFragment(cloudServer, validator.getSpaceDelegate(),
				validationHandler);

		credentialsPart.setServer(cloudServer);
	}

	public void exit() {
		// Validate credentials when exiting. This covers cases when a user:
		// 1. Clicks 'Next'
		// 2. Clicks 'Finish'
		if (validationHandler != null) {
			validationHandler.validate(context);
		}
	}

	@Override
	public boolean hasComposite() {
		return true;
	}

	@Override
	public boolean isComplete() {
		return validationHandler.isOK();
	}

	@Override
	protected void createChildFragments(List<WizardFragment> list) {
		if (spacesFragment != null) {
			list.add(spacesFragment);

		}
		super.createChildFragments(list);
	}

	protected void dispose() {
		spacesFragment = null;
	}

	@Override
	public void performCancel(IProgressMonitor monitor) throws CoreException {
		super.performCancel(monitor);
		dispose();
	}

	@Override
	public void performFinish(IProgressMonitor monitor) throws CoreException {
		// Check the current credentials without server validation first, as if
		// they are
		// valid, there is no need to send a server request.
		if (!validationHandler.isOK()) {

			IStatus status = validationHandler.getNextNonOKEvent();
			if (status == null) {
				status = CloudFoundryPlugin.getErrorStatus(Messages.ERROR_UNKNOWN_SERVER_CREATION_ERROR);
			}
			throw new CoreException(status);
		}

		ServerLifecycleAdapter listener = new ServerLifecycleAdapter() {
			@Override
			public void serverAdded(IServer server) {
				ServerCore.removeServerLifecycleListener(this);

				Job job = new ConnectJob(cloudServer, server);
				// this is getting called before
				// CloudFoundryServer.saveConfiguration() has flushed the
				// configuration therefore delay job
				job.schedule(500L);
			}
		};
		ServerCore.addServerLifecycleListener(listener);
		dispose();
	}

	private void initServer() {
		ServerWorkingCopy server = (ServerWorkingCopy) getTaskModel().getObject(TaskModel.TASK_SERVER);
		cloudServer = (CloudFoundryServer) server.loadAdapter(CloudFoundryServer.class, null);
		if (cloudServer == null) {
			CloudFoundryPlugin
					.logError("Cloud Foundry Server Framework Error: Failed to create a Cloud Foundry server working copy for: " //$NON-NLS-1$
							+ server.getId() + ". Please check if the plugin has been installed correctly."); //$NON-NLS-1$
		}
	}

	private static class ConnectJob extends Job {

		private final CloudFoundryServer originalServer;

		private final IServer server;

		public ConnectJob(CloudFoundryServer originalServer, IServer server) {
			super(Messages.CloudFoundryServerWizardFragment_TEXT_CONNECT_ACCOUNT);
			this.originalServer = originalServer;
			this.server = server;
		}

		@Override
		protected IStatus run(final IProgressMonitor monitor) {
			CloudFoundryServer cf = (CloudFoundryServer) server.loadAdapter(CloudFoundryServer.class, monitor);
			if (cf.getPassword() == null) {
				// configuration has not been saved, yet, ignore
				return Status.CANCEL_STATUS;
			}

			if (cf != null && cf.getUsername().equals(originalServer.getUsername())
					&& cf.getPassword().equals(originalServer.getPassword())
					&& cf.getUrl().equals(originalServer.getUrl())) {

				boolean connect = false;

				if (cf.hasCloudSpace() && originalServer.hasCloudSpace()) {
					CloudFoundrySpace originalSpace = originalServer.getCloudFoundrySpace();
					CloudFoundrySpace space = cf.getCloudFoundrySpace();
					connect = space.getOrgName().equals(originalSpace.getOrgName())
							&& space.getSpaceName().equals(originalSpace.getSpaceName());
				}

				if (connect) {
					CloudFoundryServerBehaviour behaviour = cf.getBehaviour();
					if (behaviour != null) {
						try {
							behaviour.connect(monitor);
						}
						catch (CoreException e) {
							CloudFoundryServerUiPlugin.getDefault().getLog().log(e.getStatus());
						}
					}
				}

			}
			return Status.OK_STATUS;
		}
	}

}
