/*******************************************************************************
 * Copyright (c) 2005 - 2006 Joel Cheuoua & others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Joel Cheuoua - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.codegen.jet.editor.util;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Stack;

import org.eclipse.emf.codegen.jet.JETCharDataGenerator;
import org.eclipse.emf.codegen.jet.JETCompiler;
import org.eclipse.emf.codegen.jet.JETConstantDataGenerator;
import org.eclipse.emf.codegen.jet.JETException;
import org.eclipse.emf.codegen.jet.JETGenerator;
import org.eclipse.emf.codegen.jet.JETMark;

/**
 */
public class JETCompilerExt extends JETCompiler {
  
  private Stack rangeStack = new Stack(); 
  
  private HashMap jetGenerators2Ranges = new HashMap();
  private HashMap ranges2JetGenerators = new HashMap();
  
  private HashMap jet2java = new HashMap();
  private HashMap java2jet = new HashMap();

  protected final String NL = System.getProperties().getProperty("line.separator");
  
  protected final String METHOD_DECLARATION = "public String generate(Object argument)";
  protected final String STRING_BUFFER_DECLARATION = "    final StringBuffer stringBuffer = new StringBuffer();" + NL;
  
  /**
   * Constructor for JETCompilerExt.
   * @param templateURI String
   * @param stream InputStream
   * @throws JETException
   */
  public JETCompilerExt(String templateURI, InputStream stream) throws JETException {
    super(templateURI, stream, "UTF-8");
  }

  /**
   * Method getJETRange.
   * @param jetOffset int
   * @param fileURI String
   * @return Range
   */  
  public Range getJETRange(int jetOffset, String fileURI) {
    for (Iterator it = rangeStack.iterator(); it.hasNext();) {
      Range range = (Range) it.next();
      if (range.start <= jetOffset && 
          jetOffset < range.stop &&
          fileURI.equals(range.fileURI))
        return range;
    }
    return null;
  }
  
  /**
   * Method getJavaRange.
   * @param javaOffset int
   * @return Range
   */
  public Range getJavaRange(int javaOffset) {
    for (Iterator it = java2jet.keySet().iterator(); it.hasNext();) {
      Range range = (Range) it.next();
      if (range.start <= javaOffset && javaOffset < range.stop)
        return range;
    }
    return null;
  }
  
  /**
   * Method getJavaRange.
   * @param jetRange Range
   * @return Range
   */
  public Range getJavaRange(Range jetRange) {
    return (Range) jet2java.get(jetRange);
  }
  
  /**
   * Method getJetRange.
   * @param javaRange Range
   * @return Range
   */
  public Range getJetRange(Range javaRange) {
    return (Range) java2jet.get(javaRange);
  }

  /*
   * (non-Javadoc)
   * 
   * @see org.eclipse.emf.codegen.jet.JETParseEventListener#handleExpression(org.eclipse.emf.codegen.jet.JETMark,
   *      org.eclipse.emf.codegen.jet.JETMark, java.util.Map)
   */
  public void handleExpression(JETMark start, JETMark stop, Map attributes) throws JETException {
    char[] chars = reader.getChars(start, stop);
    Range jetRange = new Range(start.getCursor(), stop.getCursor() + 2/* "%>"*/, chars, start.getLocalFile());
    rangeStack.push(jetRange);
    super.handleExpression(start, stop, attributes);    
  }

  /*
   * (non-Javadoc)
   * 
   * @see org.eclipse.emf.codegen.jet.JETParseEventListener#handleScriptlet(org.eclipse.emf.codegen.jet.JETMark,
   *      org.eclipse.emf.codegen.jet.JETMark, java.util.Map)
   */
  public void handleScriptlet(JETMark start, JETMark stop, Map attributes) throws JETException {
    char[] chars = reader.getChars(start, stop);
    Range jetRange = new Range(start.getCursor(), stop.getCursor() + 2/* "%>"*/, chars, start.getLocalFile());
    rangeStack.push(jetRange);
    super.handleScriptlet(start, stop, attributes);
  }

  /*
   * (non-Javadoc)
   * 
   * @see org.eclipse.emf.codegen.jet.JETCompiler#parse()
   */
  public void parse() throws JETException {
    rangeStack.clear();
    jetGenerators2Ranges.clear();
    ranges2JetGenerators.clear();
    super.parse();
  }
  
  /**
   * Method doAddCharDataGenerator.
   * @param chars char[]
   * @throws JETException
   */
  public void doAddCharDataGenerator(char[] chars) throws JETException
  {
    JETCharDataGenerator gen =null;
    if (fUseStaticFinalConstants)
    {
      gen = (JETConstantDataGenerator)constantDictionary.get(chars);
      if (gen == null)
      {
        if (constantCount == 0)
        {
          chars = stripFirstNewLineWithBlanks(chars);
        }
        ++constantCount;
        String label = CONSTANT_PREFIX + constantCount;
        gen = new JETConstantDataGenerator(chars, label);
        constantDictionary.put(chars, gen);
        constants.add(gen);
      }
      generators.add(gen);
    }
    else
    {
      generators.add(gen = new JETCharDataGenerator(chars));
    }
    JETMark current = this.reader.mark();
    Range jetRange = new Range(0, chars.length, gen.generate().toCharArray(), current.getLocalFile());
    if (!rangeStack.isEmpty()) {
      Range lastInStack = (Range) rangeStack.peek();
      int start = lastInStack.stop;
      jetRange = new Range(start, start + chars.length, gen.generate().toCharArray(), current.getLocalFile());
    }
    rangeStack.push(jetRange);
    jetGenerators2Ranges.put(gen, jetRange);
    ranges2JetGenerators.put(jetRange, gen);
  }
  
  /**
   * Method addGenerator.
   * @param gen JETEditorGenerator
   * @throws JETException
   */
  public void addGenerator(JETGenerator gen) throws JETException {
    if (!(gen instanceof JETCharDataGenerator)) {
      Range jetRange = (Range) rangeStack.peek();
      jetGenerators2Ranges.put(gen, jetRange);
      ranges2JetGenerators.put(jetRange, gen);
    }
    super.addGenerator(gen);
  }

  /**
   * Method generate.
   * @param outputStream OutputStream
   * @throws JETException
   */
  public void generate(OutputStream outputStream) throws JETException
  {
    super.generate(outputStream);
    
    java2jet.clear();
    jet2java.clear();
    
    // Compute the JET <-> Java Matching
    int currentGeneratedOffset = skeleton.getCompilationUnitContents().indexOf(METHOD_DECLARATION) + METHOD_DECLARATION.length();
    StringBuffer decl = new StringBuffer();
    decl.append(NL + "  {" + NL);
    decl.append(STRING_BUFFER_DECLARATION);
    currentGeneratedOffset += decl.toString().length();
    for (Iterator i = generators.iterator(); i.hasNext(); )
    {
      JETGenerator jetGenerator = (JETGenerator) i.next();
      Range jetRange = (Range) jetGenerators2Ranges.get(jetGenerator);      
      // Compute the body line for this generator
      String bodyLine = jetGenerator.generate();
      bodyLine = "    " + bodyLine + NL;
      // Get the jet java contents line
      String line = jetRange.getGeneratedString();

      int offsetStart = currentGeneratedOffset + bodyLine.indexOf(line);
      int offsetEnd = offsetStart + line.toString().length();
      if (jetRange != null) {
        Range javaRange = new Range(offsetStart, offsetEnd, line.toCharArray(), null);
        java2jet.put(javaRange, jetRange);
        jet2java.put(jetRange, javaRange);
      }
      currentGeneratedOffset = currentGeneratedOffset + bodyLine.toString().length();
    }    
  }
  
  /**
   */
  public static class Range {

    public int start = -1;
    public int stop = -1;
    
    private char[] chars;
    private String fileURI;
    
    /**
     * Constructor for Range.
     * @param start int
     * @param stop int
     */
    public Range(int start, int stop, char[] chars, String fileURI) {
      this.start = start;
      this.stop = stop;
      this.chars = chars;
      this.fileURI = fileURI;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object obj) {
      if (obj instanceof Range) {
        Range other = (Range) obj;
        return (start == other.start) && (stop == other.stop);
      }
      return super.equals(obj);
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
      int res = 17;
      res = res * 37 + start;
      res = res * 37 + stop;
      return res;
    }
    
    public String toString() {
      return "start : " + start + ", stop :" + stop + ", chars : " + getGeneratedString();
    }
        
    /**
     * Method getGeneratedString.
     * @return String
     */
    public String getGeneratedString() {
      return String.valueOf(chars);
    }
  }  
}