/*******************************************************************************
 * Copyright (c) 2008 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.util;

import org.eclipse.emf.mint.IItemJavaElementDescriptor;
import org.eclipse.emf.mint.IJavaTypeReference;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;

/**
 * Default implementation of {@link IItemJavaElementDescriptor}.
 * 
 * <p>
 * Clients may extend or instantiate this class.
 * </p>
 */
public class ItemJavaElementDescriptor implements IItemJavaElementDescriptor {

	private final IJavaElement javaElement;

	private final IJavaTypeReference javaTypeReference;

	private final String category;

	private final String description;

	private final String displayName;

	private final Object feature;

	/**
	 * Creates an instance with a specific Java element. All other arguments are
	 * optional.
	 * 
	 * @param javaElement
	 *            the described Java element
	 * @param category
	 *            category used to group descriptors
	 * @param description
	 *            description
	 * @param displayName
	 *            display name
	 * @param feature
	 *            related feature, if any
	 */
	public ItemJavaElementDescriptor(IJavaElement javaElement, String category,
			String description, String displayName, Object feature) {
		this(javaElement, null, category, description, displayName, feature);
	}

	/**
	 * Creates an instance with an unresolved reference to a Java type in the
	 * context of a project. All other arguments are optional.
	 * 
	 * @param context
	 *            Java project in whose context the type should be resolvable
	 * @param typeName
	 *            name of the referenced Java type
	 * @param category
	 *            category used to group descriptors
	 * @param description
	 *            description
	 * @param displayName
	 *            display name
	 * @param feature
	 *            related feature, if any
	 */
	public ItemJavaElementDescriptor(IJavaProject context, String typeName,
			String category, String description, String displayName,
			Object feature) {
		this(null, new JavaTypeReference(context, typeName), category,
				description, displayName, feature);
	}

	/**
	 * Convenience constructor for setting all encapsulated fields in one place.
	 * 
	 * @param javaElement
	 *            java element (either the element or the reference are
	 *            required)
	 * @param javaTypeReference
	 *            java type reference (either the reference or the element are
	 *            required)
	 * @param category
	 *            category
	 * @param description
	 *            description
	 * @param displayName
	 *            display name
	 * @param feature
	 *            related feature, if any
	 */
	protected ItemJavaElementDescriptor(IJavaElement javaElement,
			IJavaTypeReference javaTypeReference, String category,
			String description, String displayName, Object feature) {
		this.javaElement = javaElement;
		this.javaTypeReference = javaTypeReference;
		this.category = category;
		this.description = description;
		this.displayName = displayName;
		this.feature = feature;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getJavaElement(java.lang
	 * .Object)
	 */
	public IJavaElement getJavaElement(Object object) {
		return javaElement;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getJavaTypeReference(
	 * java.lang.Object)
	 */
	public IJavaTypeReference getJavaTypeReference(Object object) {
		return javaTypeReference;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getCategory(java.lang
	 * .Object)
	 */
	public String getCategory(Object object) {
		return category;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getDescription(java.lang
	 * .Object)
	 */
	public String getDescription(Object object) {
		return description;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getDisplayName(java.lang
	 * .Object)
	 */
	public String getDisplayName(Object object) {
		return displayName;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.emf.mint.IItemJavaElementDescriptor#getFeature(java.lang.
	 * Object)
	 */
	public Object getFeature(Object object) {
		return feature;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;

		if (obj == null)
			return false;

		if (getClass() != obj.getClass())
			return false;

		final ItemJavaElementDescriptor other = (ItemJavaElementDescriptor) obj;
		if (javaElement == null) {
			if (other.javaElement != null)
				return false;
		} else if (!javaElement.equals(other.javaElement))
			return false;

		if (javaTypeReference == null) {
			if (other.javaTypeReference != null)
				return false;
		} else if (!javaTypeReference.equals(other.javaTypeReference))
			return false;

		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((javaElement == null) ? 0 : javaElement.hashCode());
		result = prime
				* result
				+ ((javaTypeReference == null) ? 0 : javaTypeReference
						.hashCode());
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		final StringBuilder buf = new StringBuilder(getClass().getSimpleName())
				.append('[');
		buf.append("javaElement=").append(javaElement).append(';'); //$NON-NLS-1$
		buf.append("externalTypeReference=").append(javaTypeReference).append( //$NON-NLS-1$
				';');
		buf.append("category=").append(category).append(';'); //$NON-NLS-1$
		buf.append("description=").append(description).append(';'); //$NON-NLS-1$
		buf.append("displayName=").append(displayName).append(';'); //$NON-NLS-1$
		buf.append("feature=").append(feature).append(']'); //$NON-NLS-1$
		return buf.toString();
	}
}
