/*****************************************************************************
 * Copyright (c) 2017 CEA LIST and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Nicolas FAUVERGUE (CEA LIST) nicolas.fauvergue@cea.fr - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrus.interoperability.sysml14.sysml.xmi.helper;

import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.impl.EStringToStringMapEntryImpl;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.papyrus.interoperability.sysml14.sysml.utils.SysMLMigrationConstantsUtils;
import org.eclipse.uml2.uml.PackageImport;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.ProfileApplication;
import org.eclipse.uml2.uml.profile.standard.Create;

/**
 * This class defines the needed methods to preserve or create the UML XMI identifiers.
 */
public class UMLXMIIDHelper {

	/**
	 * The 'create' stereotype application identifier.
	 */
	protected static final String CREATE_STEREOTYPE = "createStereotype"; //$NON-NLS-1$

	/**
	 * The EAnnotation identifier.
	 */
	protected static final String EANNOTATION = "eAnnotation"; //$NON-NLS-1$

	/**
	 * The imported from SysML1.1 identifier.
	 */
	protected static final String IMPORTED_FROM_SYSML11 = "Imported_From_SysML_11"; //$NON-NLS-1$

	/**
	 * The PackageImport identifier.
	 */
	protected static final String PACKAGE_IMPORT = "packageImport"; //$NON-NLS-1$

	/**
	 * The ProfileApplication identifier.
	 */
	protected static final String PROFILE_APPLICATION = "profileApplication"; //$NON-NLS-1$

	/**
	 * The 'Of' separator in identifier.
	 */
	protected static final String OF = "_Of_"; //$NON-NLS-1$


	/**
	 * This allows to calculate the identifiers of created UML elements.
	 * 
	 * @param res
	 *            The UML Resource.
	 * @param current
	 *            The object to manage its identifier.
	 * @param oldIds
	 *            The existing objects and their identifiers.
	 */
	public static void calculateNeededId(final XMIResource res, final Object current, final Map<EObject, String> oldIds) {
		if (current instanceof EAnnotation && ((EAnnotation) current).getSource().equals(SysMLMigrationConstantsUtils.VERSIONING_EANNOTATION_SOURCE)) {
			calculateSysMLMigrationEAnnotation(res, (EAnnotation) current, oldIds);
		} else if (current instanceof PackageImport) {
			calculatePackageImport(res, (PackageImport) current, oldIds);
		} else if (current instanceof ProfileApplication) {
			calculateProfileApplication(res, (ProfileApplication) current, oldIds);
		}
	}

	/**
	 * This allows to calculate the identifiers of created EAnnotation.
	 * 
	 * @param res
	 *            The UML Resource.
	 * @param eAnnotation
	 *            The created EAnnotation
	 * @param oldIds
	 *            The existing objects and their identifiers.
	 */
	protected static void calculateSysMLMigrationEAnnotation(final XMIResource res, final EAnnotation eAnnotation, final Map<EObject, String> oldIds) {

		final String parentId = ((XMIResource) eAnnotation.eContainer().eResource()).getID(eAnnotation.eContainer());

		final StringBuilder stringBuilder = new StringBuilder();
		stringBuilder.append(IMPORTED_FROM_SYSML11);
		stringBuilder.append(OF);
		stringBuilder.append(parentId);
		res.setID(eAnnotation, stringBuilder.toString());

		for (final Entry<String, String> detail : eAnnotation.getDetails()) {
			if (detail instanceof EStringToStringMapEntryImpl) {
				if (detail.getKey().equals(SysMLMigrationConstantsUtils.VERSIONING_EANNOTATION_DETAIL_KEY_PAPYRUS_MIGRATION_DATE)) {
					final StringBuilder subStringBuilder = new StringBuilder();
					subStringBuilder.append(SysMLMigrationConstantsUtils.VERSIONING_EANNOTATION_DETAIL_KEY_PAPYRUS_MIGRATION_DATE);
					subStringBuilder.append(OF);
					subStringBuilder.append(parentId);
					res.setID((EStringToStringMapEntryImpl) detail, subStringBuilder.toString());
				} else if (detail.getKey().equals(SysMLMigrationConstantsUtils.VERSIONING_EANNOTATION_DETAIL_KEY_PAPYRUS_MIGRATION_BUNDLE_VERSION)) {
					final StringBuilder subStringBuilder = new StringBuilder();
					subStringBuilder.append(SysMLMigrationConstantsUtils.VERSIONING_EANNOTATION_DETAIL_KEY_PAPYRUS_MIGRATION_BUNDLE_VERSION);
					subStringBuilder.append(OF);
					subStringBuilder.append(parentId);
					res.setID((EStringToStringMapEntryImpl) detail, subStringBuilder.toString());
				} else if (detail.getKey().equals(SysMLMigrationConstantsUtils.VERSIONING_EANNOTATION_DETAIL_KEY_PAPYRUS_SOURCE_PROJECT_NAME)) {
					final StringBuilder subStringBuilder = new StringBuilder();
					subStringBuilder.append(SysMLMigrationConstantsUtils.VERSIONING_EANNOTATION_DETAIL_KEY_PAPYRUS_SOURCE_PROJECT_NAME);
					subStringBuilder.append(OF);
					subStringBuilder.append(parentId);
					res.setID((EStringToStringMapEntryImpl) detail, subStringBuilder.toString());
				}
			}
		}
	}

	/**
	 * This allows to calculate the identifiers of created PackageImport.
	 * 
	 * @param res
	 *            The UML Resource.
	 * @param packageImport
	 *            The created PackageImport
	 * @param oldIds
	 *            The existing objects and their identifiers.
	 */
	protected static void calculatePackageImport(final XMIResource res, final PackageImport packageImport, final Map<EObject, String> oldIds) {
		if (!oldIds.containsKey(packageImport)) {
			final StringBuilder stringBuilder = new StringBuilder();
			stringBuilder.append(PACKAGE_IMPORT);
			stringBuilder.append(OF);
			final org.eclipse.uml2.uml.Package importedPackage = packageImport.getImportedPackage();
			if (importedPackage.eResource() instanceof XMIResource) {
				final String importedPackageID = ((XMIResource) importedPackage.eResource()).getID(importedPackage);
				if (null != importedPackageID && !importedPackageID.isEmpty()) {
					stringBuilder.append(importedPackageID);
					res.setID(packageImport, stringBuilder.toString());
					oldIds.put(packageImport, stringBuilder.toString());
				}
			}
		}
	}

	/**
	 * This allows to calculate the identifiers of created ProfileApplication.
	 * 
	 * @param res
	 *            The UML Resource.
	 * @param profileApplication
	 *            The created ProfileApplication
	 * @param oldIds
	 *            The existing objects and their identifiers.
	 */
	protected static void calculateProfileApplication(final XMIResource res, final ProfileApplication profileApplication, final Map<EObject, String> oldIds) {
		if (!oldIds.containsKey(profileApplication)) {
			final StringBuilder stringBuilder = new StringBuilder();
			stringBuilder.append(PROFILE_APPLICATION);
			stringBuilder.append(OF);
			final Profile appliedProfile = profileApplication.getAppliedProfile();
			if (appliedProfile.eResource() instanceof XMIResource) {
				final String appliedProfileID = ((XMIResource) appliedProfile.eResource()).getID(appliedProfile);
				if (null != appliedProfileID && !appliedProfileID.isEmpty()) {
					stringBuilder.append(appliedProfileID);
					res.setID(profileApplication, stringBuilder.toString());
					oldIds.put(profileApplication, stringBuilder.toString());

					final EAnnotation eAnnotation = profileApplication.getEAnnotation("http://www.eclipse.org/uml2/2.0.0/UML"); //$NON-NLS-1$
					if (null != eAnnotation) {
						final StringBuilder eAnnotationStringBuilder = new StringBuilder();
						eAnnotationStringBuilder.append(EANNOTATION);
						eAnnotationStringBuilder.append(OF);
						eAnnotationStringBuilder.append(stringBuilder);
						res.setID(eAnnotation, eAnnotationStringBuilder.toString());
						oldIds.put(eAnnotation, eAnnotationStringBuilder.toString());
					}
				}
			}
		}
	}

	/**
	 * This allows to determinate if the created stereotype application need to be managed.
	 * 
	 * @param current
	 *            The stereotype application.
	 * @return <code>true</code> if the stereotype application identifier need to be managed, <code>false</code> otherwise.
	 */
	public static boolean isStereotypedElementCreated(final EObject current) {
		boolean result = false;
		if (current instanceof Create) {
			result = true;
		}
		return result;
	}

	/**
	 * This allows to calculate the identifiers of the stereotype applications.
	 * 
	 * @param res
	 *            The UML Resource.
	 * @param current
	 *            The stereotype application.
	 */
	public static void calculateCreateStereotypedElement(final XMIResource res, final EObject current) {
		if (current instanceof Create) {
			calculateCreate(res, (Create) current);
		}
	}

	/**
	 * This allows to calculate the identifiers of the 'create' stereotype application.
	 * 
	 * @param res
	 *            The UML Resource.
	 * @param current
	 *            The 'create' stereotype application.
	 */
	protected static void calculateCreate(final XMIResource res, final Create current) {
		final StringBuilder stringBuilder = new StringBuilder();
		stringBuilder.append(CREATE_STEREOTYPE);
		stringBuilder.append(OF);
		stringBuilder.append(((XMIResource) current.getBase_BehavioralFeature().eResource()).getID(current.getBase_BehavioralFeature()));
		res.setID(current, stringBuilder.toString());
	}

}
