/*****************************************************************************
 * Copyright (c) 2018-2020 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 *  Matteo MORELLI      matteo.morelli@cea.fr - Bug #566899
 *
 *****************************************************************************/

package org.eclipse.papyrus.robotics.ros2.codegen.launch

import java.util.ArrayList
import org.eclipse.uml2.uml.Class
import org.eclipse.uml2.uml.Port
import org.eclipse.uml2.uml.Property

import static extension org.eclipse.papyrus.robotics.core.utils.InstanceUtils.*
import static extension org.eclipse.papyrus.robotics.core.utils.InteractionUtils.*
import static extension org.eclipse.papyrus.robotics.ros2.codegen.utils.SkillUtils.*
import static extension org.eclipse.papyrus.robotics.ros2.codegen.utils.SequencerUtils.*
import static extension org.eclipse.papyrus.robotics.ros2.codegen.utils.PackageTools.pkgName
import org.eclipse.papyrus.uml.tools.utils.ConnectorUtil
import org.eclipse.papyrus.infra.tools.file.IPFileSystemAccess
import org.eclipse.papyrus.robotics.core.utils.PortUtils
import org.eclipse.uml2.uml.ConnectorEnd
import org.eclipse.uml2.uml.InstanceValue
import org.eclipse.papyrus.designer.deployment.tools.DepUtils
import java.util.List
import java.util.Collections
import static extension org.eclipse.papyrus.robotics.ros2.codegen.utils.ComponentUtils.isRegistered
import org.eclipse.papyrus.robotics.ros2.preferences.Ros2PreferenceUtils
import org.eclipse.papyrus.robotics.ros2.preferences.Ros2Distributions

class LaunchScript {
	static def createLaunchScript(Class system, List<Property> parts, boolean addSequencer, boolean activate) '''
		import launch.actions
		import launch_ros
		import lifecycle_msgs.msg

		from launch_ros.events.lifecycle import ChangeState
		from launch import LaunchDescription
		from launch_ros.actions import LifecycleNode
		from launch_ros.events.lifecycle import ChangeState
		from ament_index_python.packages import get_package_share_directory

		share_dir = get_package_share_directory('«system.nearestPackage.pkgName»')

		def generate_launch_description():

			# Launch Description
			ld = launch.LaunchDescription()

			# Add the actions to the launch description.
			# The order they are added reflects the order in which they will be executed.
			«FOR part : parts»
				«val component = part.type as Class»
				«part.name»_node = LifecycleNode(
					«IF eloquent»node_«ENDIF»name='«part.name»',
					package='«component.nearestPackage.pkgName»', «IF eloquent»node_«ENDIF»executable='«component.name»«IF component.isRegistered»_main«ENDIF»',
					remappings=[
						«FOR remap : system.remaps(part) SEPARATOR(", ")»«remap»«ENDFOR»
					],
					parameters=[share_dir+'/launch/cfg/param.yaml'],
					output='screen',
					emulate_tty=True	# assure that RCLCPP output gets flushed
				)
				ld.add_entity(«part.name»_node)
			«ENDFOR»
			«IF addSequencer»
				«system.sequencerName»_node = LifecycleNode(
					«IF eloquent»node_«ENDIF»name='«system.sequencerName»',
					package='bt_sequencer', «IF eloquent»node_«ENDIF»executable='bt_sequencer',
					parameters=[share_dir+'/launch/cfg/param.yaml'],
					output='screen',
					emulate_tty=True	# assure that RCLCPP output gets flushed
				)
				ld.add_entity(«system.sequencerName»_node)
			«ENDIF»

			«IF activate»
				# transition to configure after startup
				«FOR part : parts»
					«generateConfig(part.name)»
				«ENDFOR»

				# transition to activate, once inactive
				«FOR part : parts»
					«generateActivation(part.name)»
				«ENDFOR»

			«ENDIF»
			return ld
	'''

	/**
	 * Generate the commands for node configuration given node name
	 */
	static def generateConfig(String name) '''
		configure_«name» = launch.actions.RegisterEventHandler(
			launch.event_handlers.on_process_start.OnProcessStart(
				target_action=«name»_node,
				on_start=[
		 			launch.actions.EmitEvent(
						event=launch_ros.events.lifecycle.ChangeState(
							lifecycle_node_matcher=launch.events.matches_action(«name»_node),
							transition_id=lifecycle_msgs.msg.Transition.TRANSITION_CONFIGURE
						)
					)
				]
			)
		)
		ld.add_entity(configure_«name»)
	'''

	/**
	 * Generate the commands for node activation given node name
	 */
	static def generateActivation(String name) '''
		activate_«name» = launch.actions.RegisterEventHandler(
			launch_ros.event_handlers.OnStateTransition(
				target_lifecycle_node=«name»_node,
				start_state='configuring', goal_state='inactive',
				entities=[
					launch.actions.EmitEvent(
						event=launch_ros.events.lifecycle.ChangeState(
							lifecycle_node_matcher=launch.events.matches_action(«name»_node),
							transition_id=lifecycle_msgs.msg.Transition.TRANSITION_ACTIVATE
						)
					)
				]
			)
		)
		ld.add_entity(activate_«name»)
	'''

	/**
	 * initialize parameter values from IS and the (implicit) system sequencer model
	 * (In the current implementation, the default skill semantics realization is always selected)
	 */
	static def createParameterFile(Class system) '''
		«FOR part : system.compInstanceList»
			«IF part.defaultValue instanceof InstanceValue»
				«val is = (part.defaultValue as InstanceValue).instance»
				«IF is !== null»
					«part.name»:
					  ros__parameters:
					    «FOR slot : is.slots»
					    	«IF slot.definingFeature !== null && DepUtils.firstValue(slot) !== null»
					    		«slot.definingFeature.name» : «DepUtils.firstValue(slot).stringValue»;
					    	«ENDIF»
					     «ENDFOR»
				«ENDIF»
			«ENDIF»
		«ENDFOR»
		«IF !system.uniqueSkills.nullOrEmpty»
			«system.sequencerName»:
			  ros__parameters:
			    plugin_lib_names: [
			    «FOR skill : system.getUniqueSkills SEPARATOR ','»
			    	«"  \""+skill.realizationFileName»_bt_node"
			    «ENDFOR»
			    ]
		«ENDIF»
	'''

	/**
	 * a ROS1 launch script (XML, unused)
	 */
	static def createLaunchScriptROS1(Class system) '''
		<launch>
			«FOR part : system.compInstanceList»
				«val component = part.type as Class»
				<node pkg="«component.nearestPackage.pkgName»" type="«component.name»" name="«part.name»" cwd="node" respawn="false" output="screen">
					«FOR remap : system.remapsXML(part)»
						«remap»
					«ENDFOR»
				</node>
			«ENDFOR»
		</launch>
	'''		

	/**
	 * Remappings for XML (only ROS1, unused)
	 */
	static def remapsXML(Class system, Property part) {
		val remaps = new ArrayList<String>
		for (port : PortUtils.getAllPorts(part.type as Class)) {
			val oppositeEnd = system.getOpposite(part, port)
			if (oppositeEnd !== null) {
				remaps.add('''<remap from="«port.topic»" to="«part.getTopic(port, oppositeEnd)»"/>''')
			}
		}
		return remaps
	}	

	/**
	 * Re-mappings for command line (used for debugging a binary)
	 */
	static def remapsCmd(Class system, Property part) '''
		«FOR port : PortUtils.getAllPorts(part.type as Class) SEPARATOR " "»
			«val oppositeEnd = system.getOpposite(part, port)»
			«IF oppositeEnd !== null»
				-r «port.topic»:=«part.getTopic(port, oppositeEnd)»
			«ENDIF»
		«ENDFOR»
	'''

	static def remaps(Class system, Property part) {
		val remaps = new ArrayList<String>
		for (port : PortUtils.getAllPorts(part.type as Class)) {
			val oppositeEnd = system.getOpposite(part, port)
			if (oppositeEnd !== null && port.commObject !== null) {
				remaps.add('''('«port.topic»', '«part.getTopic(port, oppositeEnd)»')''')
			}
		}
		return remaps
	}	

	/**
	 * Return the topic of a port, currently the port name only
	 * 
	 * This information will be remapped in the launch scripts
	 */
	static def getTopic(Port port) '''«port.name»'''

	/**
	 * Return the topic of a publisher port, based on the assumption that the
	 * topic is a concatenation of communication object name and port name
	 * 
	 * This information will be replaced depending on assembly information
	 */
	static def getTopic(Property part, Port port, ConnectorEnd oppositeEnd) '''
		«IF (port.provideds.size > 0)»
			«part.name»/«port.commObject.name»/«port.name
		»«ELSE»
			«oppositeEnd.partWithPort.name»/«port.commObject.name»/«oppositeEnd.role.name»«
		ENDIF»'''

	/**
	 * Return opposite part, based on the assumption that there is a single port
	 * targeting a port
	 */
	static def getOpposite(Class system, Property part, Port port) {
		for (connector: system.ownedConnectors) {
			if (ConnectorUtil.connectsPort(connector, port)) {
				val end = ConnectorUtil.connEndNotPart(connector, part)
				if (end !== null) {
					return end
				}
			}
		}
		return null
	}

	static def generateLaunch(IPFileSystemAccess fileAccess, Class system) {
		val parts = system.compInstanceList;
		val addSequencer = !system.uniqueSkills.nullOrEmpty
		fileAccess.generateFile("launch/launch.py", system.createLaunchScript(parts, addSequencer, false).toString)
		fileAccess.generateFile("launch/activate.py", system.createLaunchScript(parts, addSequencer, true).toString)
		// create a separate launch file for each part
		for (Property part : parts) {
			fileAccess.generateFile('''launch/launch.«part.name».py''', system.createLaunchScript(Collections.singletonList(part), false, false).toString)
			fileAccess.generateFile('''launch/activate.«part.name».py''', system.createLaunchScript(Collections.singletonList(part), false, true).toString)
		}
		if (addSequencer) {
			// create a separate launch file for the sequencer
			fileAccess.generateFile('''launch/launch.«system.sequencerName».py''', system.createLaunchScript(Collections.emptyList, true, false).toString)
			// 		. do not activate the sequencer directly - the user must first specify which behavior wants to be executed  
			fileAccess.generateFile('''launch/activate.«system.sequencerName».py''', system.createLaunchScript(Collections.emptyList, true, false).toString)
		}
		// create the parameters file
		fileAccess.generateFile("launch/cfg/param.yaml", system.createParameterFile().toString)
	}

	static def isEloquent() {
		Ros2PreferenceUtils.rosDistro === Ros2Distributions.ELOQUENT;
	}
}