/*****************************************************************************
 * Copyright (c) 2025 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrus.uml.tools.commands;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.core.runtime.Assert;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.papyrus.infra.core.clipboard.PapyrusClipboard;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Extension;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * This class is used by the Copy/Paste to set the correct values for the Properties of Stereotypes typed by EObject when EReference.isContainment()==false:
 * <ul>
 * <li>stereotypeApplication</li>
 * <li>UML Element</li>
 * </ul>
 *
 * The command {@link DuplicateStereotypeCommand} already set values in EReference, but the set values are the source value. {@link DuplicateStereotypeCommand} doesn't care of the duplicated elements to set new values
 */
public class UpdateStereotypeValueCommand extends RecordingCommand {

	/**
	 * The Papyrus clibboard
	 */
	private PapyrusClipboard<Object> papyrusClipboard;

	/**
	 * This map is shared with the {@link DuplicateStereotypeCommand} executed previously. It contains the source StereotypeApplication vs the target StereotypeApplication
	 */
	private Map<EObject, EObject> stereotypeApplicationMap;

	/**
	 * Constructor.
	 *
	 * @param domain
	 * @param papyrusClipboard
	 * @param stereotypeApplicationMap
	 *            this map has been filled by the {@link DuplicateStereotypeCommand} executed previously during the copy/paste process
	 */
	public UpdateStereotypeValueCommand(TransactionalEditingDomain domain, PapyrusClipboard<Object> papyrusClipboard, Map<EObject, EObject> stereotypeApplicationMap) {
		super(domain);
		this.papyrusClipboard = papyrusClipboard;
		this.stereotypeApplicationMap = stereotypeApplicationMap;
	}

	/**
	 * @see org.eclipse.emf.transaction.RecordingCommand#doExecute()
	 *
	 */

	@Override
	protected void doExecute() {
		for (final Entry<EObject, EObject> entry : this.stereotypeApplicationMap.entrySet()) {
			mapValues(entry.getKey(), entry.getValue());
		}
	}

	/**
	 * Set the expected referenced values in the target stereotypeApplication.
	 * We use the source stereotypeApplication to deduce the values to set in the target stereotypeApplication
	 *
	 * @param sourceSteAppl
	 *            the source stereotypeApplication
	 * @param targetSteAppl
	 *            the target stereotypeApplication
	 */
	private void mapValues(final EObject sourceSteAppl, final EObject targetSteAppl) {
		// 1. the two stereotypeApplications must be instance of the same EClass (of course!)
		Assert.isTrue(sourceSteAppl.eClass() == targetSteAppl.eClass());

		// 2. we iterate on all EReferences
		for (final EReference eRef : sourceSteAppl.eClass().getEAllReferences()) {
			// we manage only EReference, which are not in containment
			if (!eRef.getName().startsWith(Extension.METACLASS_ROLE_PREFIX)
					&& eRef.isChangeable()
					&& !eRef.isContainment()) {

				// 3. multi-valued property
				if (eRef.isMany()) {
					final Collection<?> sourceValues = (Collection<?>) sourceSteAppl.eGet(eRef);
					Collection<EObject> targetValues = new ArrayList<>();
					for (final Object currentSourceValue : sourceValues) {
						if (currentSourceValue instanceof EObject currentValue) {// should always be true
							targetValues.add(getTargetValue(currentValue));
						}
					}
					targetSteAppl.eSet(eRef, targetValues);
				} else {
					// 4. mono-valued properties
					final EObject sourceValue = (EObject) sourceSteAppl.eGet(eRef);
					final EObject targetValue = getTargetValue(sourceValue);
					targetSteAppl.eSet(eRef, targetValue);
				}
			}
		}
	}

	/**
	 *
	 * @param sourceValue
	 *            the source value
	 * @return
	 *         the EObject to use as target value
	 */
	protected EObject getTargetValue(final EObject sourceValue) {
		EObject returnedValue = null;
		if (isStereotypeApplication(sourceValue)) {
			final EObject targetValue = this.stereotypeApplicationMap.get(sourceValue);

			if (targetValue == null) {
				// nothing to do
				// the target value was not included in the copied/pasted element, so we will reuse the initial value.
				// the set has already be done by the command {@link DuplicateStereotypeCommand}
				returnedValue = sourceValue;
			} else {
				returnedValue = targetValue;
			}
		} else if (sourceValue instanceof Element) {
			Map<?, ?> internalClipboardToTargetCopy = papyrusClipboard.getInternalClipboardToTargetCopy();

			EObject intermediateCopiedElement = (EObject) papyrusClipboard.getSourceToInternalClipboard().get(sourceValue);
			EObject realPastedElement = (EObject) internalClipboardToTargetCopy.get(intermediateCopiedElement);

			// if realPastedElement is null, it means that we don't copy this element (outside of the selection), so we reuse the sourceValue
			if (realPastedElement == null) {
				returnedValue = sourceValue;
			} else {
				returnedValue = realPastedElement;
			}
		}
		return returnedValue;
	}

	/**
	 *
	 * @param eobject
	 *            an eobject
	 * @return
	 *         <code>true</code> if the eobject is a StereotypeApplication
	 */
	private boolean isStereotypeApplication(final EObject eobject) {
		return UMLUtil.getBaseElement(eobject) != null;
	}
}

