/**
 * Copyright (c) 2020 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.codegen.common.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.eclipse.core.resources.IProject;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.papyrus.designer.transformation.base.utils.ModelManagement;
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException;
import org.eclipse.papyrus.designer.transformation.core.transformations.TransformationContext;
import org.eclipse.papyrus.designer.uml.tools.utils.PackageUtil;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.papyrus.robotics.core.utils.FileExtensions;
import org.eclipse.papyrus.robotics.core.utils.ScanUtils;
import org.eclipse.papyrus.robotics.profile.robotics.components.ComponentDefinitionModel;
import org.eclipse.papyrus.robotics.profile.robotics.components.SystemComponentArchitectureModel;
import org.eclipse.papyrus.robotics.profile.robotics.services.ServiceDefinitionModel;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;

@SuppressWarnings("all")
public class PackageTools {
  /**
   * Return the package name. Instead of the UML model name, the name of the project
   * hosting the model (lowerCase) is used.
   * @param model
   * @return The package name
   */
  public static String pkgName(final org.eclipse.uml2.uml.Package model) {
    try {
      if ((model == null)) {
        throw new TransformationException("pkgName is invoked with a null model (internal generator error)");
      }
      boolean _isApplied = StereotypeUtil.isApplied(model, ServiceDefinitionModel.class);
      if (_isApplied) {
        return model.getName().toLowerCase();
      } else {
        final ComponentDefinitionModel compDefModel = UMLUtil.<ComponentDefinitionModel>getStereotypeApplication(model, ComponentDefinitionModel.class);
        if (((compDefModel != null) && compDefModel.isExternal())) {
          return model.getName().toLowerCase();
        }
        return PackageTools.getProjectName(model).toLowerCase();
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Check whether a given srcCompRes has already been copied into the target
   * model. If yes, it returns the component found in there.
   * Otherwise, it returns a copy of the component found in the src-model
   * resource (the resource is not transformed)
   */
  public static org.eclipse.uml2.uml.Class findOrCopyCompDef(final Resource srcCompRes) {
    org.eclipse.uml2.uml.Class compDef = ScanUtils.getCompFromResource(srcCompRes);
    if ((compDef != null)) {
      EList<ModelManagement> _additionalRootPkgs = TransformationContext.current.copier.getAdditionalRootPkgs();
      final ArrayList<ModelManagement> mmList = new ArrayList<ModelManagement>(_additionalRootPkgs);
      mmList.add(TransformationContext.current.mm);
      for (final ModelManagement pkg : mmList) {
        boolean _isApplied = StereotypeUtil.isApplied(pkg.getModel(), ComponentDefinitionModel.class);
        if (_isApplied) {
          final org.eclipse.uml2.uml.Class compDefCandidate = ScanUtils.getCompFromResource(pkg.getModel().eResource());
          if (((compDefCandidate != null) && Objects.equals(compDefCandidate.getName(), compDef.getName()))) {
            return compDefCandidate;
          }
        }
      }
      return TransformationContext.current.copier.<org.eclipse.uml2.uml.Class>getCopy(compDef);
    }
    return null;
  }

  /**
   * Check whether a given srcSystemRes has already been copied into the target
   * model. If yes, it returns the system found in there.
   * Otherwise, it returns a copy of the system found in the src-model
   * resource (the resource is not transformed)
   */
  public static org.eclipse.uml2.uml.Class findOrCopySystem(final Resource srcSystemRes) {
    org.eclipse.uml2.uml.Class system = ScanUtils.getSystemFromResource(srcSystemRes);
    if ((system != null)) {
      EList<ModelManagement> _additionalRootPkgs = TransformationContext.current.copier.getAdditionalRootPkgs();
      final ArrayList<ModelManagement> mmList = new ArrayList<ModelManagement>(_additionalRootPkgs);
      mmList.add(TransformationContext.current.mm);
      for (final ModelManagement pkg : mmList) {
        boolean _isApplied = StereotypeUtil.isApplied(pkg.getModel(), SystemComponentArchitectureModel.class);
        if (_isApplied) {
          final org.eclipse.uml2.uml.Class systemCandidate = ScanUtils.getSystemFromResource(pkg.getModel().eResource());
          if (((systemCandidate != null) && Objects.equals(systemCandidate.getName(), system.getName()))) {
            return systemCandidate;
          }
        }
      }
      return TransformationContext.current.copier.<org.eclipse.uml2.uml.Class>getCopy(system);
    }
    return null;
  }

  /**
   * return a list of component definitions that are defined in a project.
   * If code generation is launched from a component within a package, the
   * resource set will likely not have the complete set of models defined in
   * that package as these are needed to generate the complete set.
   * Note that the different components are typically defined in multiple models, but these models
   * must be stored in a common location (Eclipse project) aligned with RobMoSys naming conventions
   * @param project an Eclipse project hosting models.
   */
  public static ArrayList<org.eclipse.uml2.uml.Class> getCompDefs(final IProject project) {
    try {
      final ArrayList<URI> compDefsURIs = new ArrayList<URI>();
      final ArrayList<org.eclipse.uml2.uml.Class> compDefs = new ArrayList<org.eclipse.uml2.uml.Class>();
      ScanUtils.processContainer(project, FileExtensions.COMPDEF_UML, compDefsURIs);
      final ResourceSet srcRS = TransformationContext.current.copier.source.eResource().getResourceSet();
      for (final URI compDefURI : compDefsURIs) {
        {
          final Resource srcCompRes = srcRS.getResource(compDefURI, true);
          final org.eclipse.uml2.uml.Class compDef = PackageTools.findOrCopyCompDef(srcCompRes);
          if ((compDef != null)) {
            final org.eclipse.uml2.uml.Package root = PackageUtil.getRootPackage(compDef);
            final ComponentDefinitionModel cdm = UMLUtil.<ComponentDefinitionModel>getStereotypeApplication(root, ComponentDefinitionModel.class);
            if (((cdm != null) && (!cdm.isExternal()))) {
              compDefs.add(compDef);
            }
          }
        }
      }
      return compDefs;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * return a list of component definitions that are defined in a project.
   * Note that the different components are typically defined in multiple models, but these models
   * must be stored in a common location (Eclipse project) aligned with RobMoSys naming conventions
   * @param project an Eclipse project hosting models.
   */
  public static org.eclipse.uml2.uml.Class getSystem(final IProject project) {
    try {
      final ArrayList<URI> systemURIs = new ArrayList<URI>();
      ScanUtils.processContainer(project, FileExtensions.SYSTEM_UML, systemURIs);
      final ResourceSet srcRS = TransformationContext.current.copier.source.eResource().getResourceSet();
      boolean _hasNext = systemURIs.iterator().hasNext();
      if (_hasNext) {
        final URI systemURI = systemURIs.iterator().next();
        final Resource srcSystemRes = srcRS.getResource(systemURI, true);
        return PackageTools.findOrCopySystem(srcSystemRes);
      }
      return null;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Get the project from an existing model
   * @param pkg an element of a model
   */
  public static String getProjectName(final org.eclipse.uml2.uml.Package pkg) {
    try {
      Resource _eResource = pkg.eResource();
      boolean _tripleEquals = (_eResource == null);
      if (_tripleEquals) {
        String _name = pkg.getName();
        String _plus = ("Package " + _name);
        String _plus_1 = (_plus + " is not in a resource - proxy?");
        throw new TransformationException(_plus_1);
      }
      final URI uri = pkg.eResource().getURI();
      if ((uri.isPlatformResource() || uri.isPlatformPlugin())) {
        int _size = ((List<String>)Conversions.doWrapArray(uri.segments())).size();
        boolean _greaterThan = (_size > 5);
        if (_greaterThan) {
          return uri.segment(3);
        }
        return URI.decode(uri.segment(1));
      }
      return URI.decode(uri.segment(0));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
