/**
 * Copyright (c) 2020, 2023 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Matteo MORELLI (CEA LIST) <matteo.morelli@cea.fr> - Bug #566899
 *  Matteo MORELLI (CEA LIST) <matteo.morelli@cea.fr> - Bug #581690
 *  Matteo MORELLI (CEA LIST) <matteo.morelli@cea.fr> - Bug #581812
 */
package org.eclipse.papyrus.robotics.ros2.codegen.cpp.skillrealization;

import com.google.common.collect.Iterables;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.languages.common.base.file.IPFileSystemAccess;
import org.eclipse.papyrus.designer.languages.common.base.file.ProtSection;
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.papyrus.robotics.profile.robotics.skills.InAttribute;
import org.eclipse.papyrus.robotics.profile.robotics.skills.OutAttribute;
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillDefinition;
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillOperationalState;
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillParameter;
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillSemantic;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.SkillUtils;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.Property;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Create the C++ code of a skill realization executed
 * on the sequencing layer (uses a config&coord interface to SW components)
 */
@SuppressWarnings("all")
public class CreateSkillRealizationCppCode {
  private static String msgAbortMultipleOperationalStatesIsUnsopported() {
    return String.format(
      "abort transformation, code-generation of skill FSMs with many operational states is not supported");
  }

  private static String msgAbortSystemHasNotCompIf(final String ifName) {
    return String.format(
      "abort transformation, there are no components in the system with the required coordination interface (%s)", ifName);
  }

  private static String msgAbortCompIfNotActionNorQuery(final String ifName) {
    return String.format(
      "abort transformation, the coordination interface has incompatible type (%s)", ifName);
  }

  /**
   * static def createCppSkillFSMManager(SkillDefinition definition, SkillSemantic semantics) '''
   * '''
   */
  public static CharSequence createCppSimplestActionSkill(final SkillDefinition skill, final SkillOperationalState ops, final String actionName) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("// Generated by Papyrus4Robotics");
    _builder.newLine();
    _builder.append("//");
    _builder.newLine();
    _builder.newLine();
    _builder.append("// ");
    String _protSection = ProtSection.protSection("BtActionNode header files");
    _builder.append(_protSection);
    _builder.newLineIfNotEmpty();
    _builder.append("// ");
    String _protSection_1 = ProtSection.protSection();
    _builder.append(_protSection_1);
    _builder.newLineIfNotEmpty();
    _builder.append("#include \"bt_utils/generic_types_conversions.hpp\"");
    _builder.newLine();
    {
      LinkedHashSet<DataType> _uniqueSkillParameterTypes = SkillUtils.getUniqueSkillParameterTypes(skill);
      for(final DataType tp : _uniqueSkillParameterTypes) {
        _builder.append("#include \"");
        String _rOS2TypeIncludePath = SkillUtils.getROS2TypeIncludePath(tp);
        _builder.append(_rOS2TypeIncludePath);
        _builder.append("\"");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("#include \"");
    String _coordinationIfIncludePath = SkillUtils.getCoordinationIfIncludePath(ops);
    _builder.append(_coordinationIfIncludePath);
    _builder.append("\"");
    _builder.newLineIfNotEmpty();
    _builder.append("#include \"nav2_behavior_tree/bt_action_node.hpp\"");
    _builder.newLine();
    _builder.newLine();
    _builder.append("class ");
    String _name = SkillUtils.getName(skill);
    _builder.append(_name);
    _builder.append("Action : public nav2_behavior_tree::BtActionNode<");
    String _coordinationIfQn = SkillUtils.getCoordinationIfQn(ops);
    _builder.append(_coordinationIfQn);
    _builder.append(">");
    _builder.newLineIfNotEmpty();
    _builder.append("{");
    _builder.newLine();
    _builder.append("public:");
    _builder.newLine();
    _builder.append("  ");
    String _name_1 = SkillUtils.getName(skill);
    _builder.append(_name_1, "  ");
    _builder.append("Action(");
    _builder.newLineIfNotEmpty();
    _builder.append("      ");
    _builder.append("const std::string& name,");
    _builder.newLine();
    _builder.append("      ");
    _builder.append("const std::string & action_name,");
    _builder.newLine();
    _builder.append("      ");
    _builder.append("const BT::NodeConfiguration& conf)");
    _builder.newLine();
    _builder.append("  ");
    _builder.append(": nav2_behavior_tree::BtActionNode<");
    String _coordinationIfQn_1 = SkillUtils.getCoordinationIfQn(ops);
    _builder.append(_coordinationIfQn_1, "  ");
    _builder.append(">(name, action_name, conf)");
    _builder.newLineIfNotEmpty();
    _builder.append("    ");
    _builder.append("{");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    {
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(skill.getIns());
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        _builder.append("  ");
        _builder.append("void on_tick() override");
        _builder.newLine();
        _builder.append("  ");
        _builder.append("{");
        _builder.newLine();
        _builder.append("  ");
        _builder.append("\t");
        _builder.append("// ");
        String _protSection_2 = ProtSection.protSection("BtActionNode on_tick()");
        _builder.append(_protSection_2, "  \t");
        _builder.newLineIfNotEmpty();
        {
          EList<InAttribute> _ins = skill.getIns();
          for(final InAttribute param : _ins) {
            _builder.append("  ");
            _builder.append("  ");
            String _rOS2TypeFromMsgName = SkillUtils.getROS2TypeFromMsgName(SkillUtils.getType(param));
            _builder.append(_rOS2TypeFromMsgName, "    ");
            _builder.append(" ");
            String _name_2 = SkillUtils.getName(param);
            _builder.append(_name_2, "    ");
            _builder.append(";");
            _builder.newLineIfNotEmpty();
            _builder.append("  ");
            _builder.append("  ");
            _builder.append("getInput(\"");
            String _name_3 = SkillUtils.getName(param);
            _builder.append(_name_3, "    ");
            _builder.append("\", ");
            String _name_4 = SkillUtils.getName(param);
            _builder.append(_name_4, "    ");
            _builder.append(");");
            _builder.newLineIfNotEmpty();
            {
              EList<Property> _ownedAttributes = SkillUtils.getType(param).getOwnedAttributes();
              for(final Property prop : _ownedAttributes) {
                _builder.append("  ");
                _builder.append("  ");
                _builder.append("goal_.");
                String _name_5 = prop.getName();
                _builder.append(_name_5, "    ");
                _builder.append(" = ");
                String _name_6 = SkillUtils.getName(param);
                _builder.append(_name_6, "    ");
                _builder.append(".");
                String _name_7 = prop.getName();
                _builder.append(_name_7, "    ");
                _builder.append(";");
                _builder.newLineIfNotEmpty();
              }
            }
          }
        }
        _builder.append("  ");
        _builder.append("\t");
        _builder.append("// ");
        String _protSection_3 = ProtSection.protSection();
        _builder.append(_protSection_3, "  \t");
        _builder.newLineIfNotEmpty();
        _builder.append("  ");
        _builder.append("}");
        _builder.newLine();
      }
    }
    _builder.newLine();
    {
      boolean _isNullOrEmpty_1 = IterableExtensions.isNullOrEmpty(skill.getOuts());
      boolean _not_1 = (!_isNullOrEmpty_1);
      if (_not_1) {
        _builder.append("  ");
        _builder.append("BT::NodeStatus on_success() override");
        _builder.newLine();
        _builder.append("  ");
        _builder.append("{");
        _builder.newLine();
        _builder.append("  ");
        _builder.append("\t");
        _builder.append("// ");
        String _protSection_4 = ProtSection.protSection("BtActionNode on_success()");
        _builder.append(_protSection_4, "  \t");
        _builder.newLineIfNotEmpty();
        {
          EList<OutAttribute> _outs = skill.getOuts();
          for(final OutAttribute param_1 : _outs) {
            _builder.append("  ");
            _builder.append("\t");
            String _rOS2TypeFromMsgName_1 = SkillUtils.getROS2TypeFromMsgName(SkillUtils.getType(param_1));
            _builder.append(_rOS2TypeFromMsgName_1, "  \t");
            _builder.append(" ");
            String _name_8 = SkillUtils.getName(param_1);
            _builder.append(_name_8, "  \t");
            _builder.append(";");
            _builder.newLineIfNotEmpty();
            {
              EList<Property> _ownedAttributes_1 = SkillUtils.getType(param_1).getOwnedAttributes();
              for(final Property prop_1 : _ownedAttributes_1) {
                _builder.append("  ");
                _builder.append("\t");
                String _name_9 = SkillUtils.getName(param_1);
                _builder.append(_name_9, "  \t");
                _builder.append(".");
                String _name_10 = prop_1.getName();
                _builder.append(_name_10, "  \t");
                _builder.append(" = result_.result->");
                String _name_11 = prop_1.getName();
                _builder.append(_name_11, "  \t");
                _builder.append(";");
                _builder.newLineIfNotEmpty();
              }
            }
            _builder.append("  ");
            _builder.append("\t");
            _builder.append("setOutput(\"");
            String _name_12 = SkillUtils.getName(param_1);
            _builder.append(_name_12, "  \t");
            _builder.append("\", ");
            String _name_13 = SkillUtils.getName(param_1);
            _builder.append(_name_13, "  \t");
            _builder.append(");");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("  ");
        _builder.append("\t");
        _builder.append("return BT::NodeStatus::SUCCESS;");
        _builder.newLine();
        _builder.append("  ");
        _builder.append("\t");
        _builder.append("// ");
        String _protSection_5 = ProtSection.protSection();
        _builder.append(_protSection_5, "  \t");
        _builder.newLineIfNotEmpty();
        _builder.append("  ");
        _builder.append("}");
        _builder.newLine();
      }
    }
    _builder.newLine();
    _builder.append("  ");
    CharSequence _createProvidedPortsMethod = CreateSkillRealizationCppCode.createProvidedPortsMethod(skill, true);
    _builder.append(_createProvidedPortsMethod, "  ");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("// ");
    String _protSection_6 = ProtSection.protSection("- BtActionNode other class methods and attributes (if any)");
    _builder.append(_protSection_6);
    _builder.newLineIfNotEmpty();
    _builder.append("// ");
    String _protSection_7 = ProtSection.protSection();
    _builder.append(_protSection_7);
    _builder.newLineIfNotEmpty();
    _builder.append("};");
    _builder.newLine();
    _builder.newLine();
    _builder.append("#include \"behaviortree_cpp_v3/bt_factory.h\"");
    _builder.newLine();
    _builder.append("BT_REGISTER_NODES(factory)");
    _builder.newLine();
    _builder.append("{");
    _builder.newLine();
    _builder.append("  ");
    _builder.append("BT::NodeBuilder builder =");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("[](const std::string & name, const BT::NodeConfiguration & config)");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("{");
    _builder.newLine();
    _builder.append("      ");
    _builder.append("return std::make_unique<");
    String _name_14 = SkillUtils.getName(skill);
    _builder.append(_name_14, "      ");
    _builder.append("Action>(name, \"");
    _builder.append(actionName, "      ");
    _builder.append("\", config);");
    _builder.newLineIfNotEmpty();
    _builder.append("    ");
    _builder.append("};");
    _builder.newLine();
    _builder.newLine();
    _builder.append("  ");
    _builder.append("factory.registerBuilder<");
    String _name_15 = SkillUtils.getName(skill);
    _builder.append(_name_15, "  ");
    _builder.append("Action>(\"");
    String _name_16 = SkillUtils.getName(skill);
    _builder.append(_name_16, "  ");
    _builder.append("\", builder);");
    _builder.newLineIfNotEmpty();
    _builder.append("}");
    _builder.newLine();
    return _builder;
  }

  public static CharSequence createCppSimplestQuerySkill(final SkillDefinition skill, final SkillOperationalState ops, final String serviceName) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("// Generated by Papyrus4Robotics");
    _builder.newLine();
    _builder.append("//");
    _builder.newLine();
    _builder.newLine();
    _builder.append("// ");
    String _protSection = ProtSection.protSection("BtConditionNode header files");
    _builder.append(_protSection);
    _builder.newLineIfNotEmpty();
    _builder.append("// ");
    String _protSection_1 = ProtSection.protSection();
    _builder.append(_protSection_1);
    _builder.newLineIfNotEmpty();
    _builder.append("#include \"bt_utils/generic_types_conversions.hpp\"");
    _builder.newLine();
    {
      LinkedHashSet<DataType> _uniqueSkillParameterTypes = SkillUtils.getUniqueSkillParameterTypes(skill);
      for(final DataType tp : _uniqueSkillParameterTypes) {
        _builder.append("#include \"");
        String _rOS2TypeIncludePath = SkillUtils.getROS2TypeIncludePath(tp);
        _builder.append(_rOS2TypeIncludePath);
        _builder.append("\"");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("#include \"");
    String _coordinationIfIncludePath = SkillUtils.getCoordinationIfIncludePath(ops);
    _builder.append(_coordinationIfIncludePath);
    _builder.append("\"");
    _builder.newLineIfNotEmpty();
    _builder.append("#include \"bt_utils/bt_condition_node.hpp\"");
    _builder.newLine();
    _builder.newLine();
    _builder.append("class ");
    String _name = SkillUtils.getName(skill);
    _builder.append(_name);
    _builder.append("Condition : public bt_utils::BtConditionNode<");
    String _coordinationIfQn = SkillUtils.getCoordinationIfQn(ops);
    _builder.append(_coordinationIfQn);
    _builder.append(">");
    _builder.newLineIfNotEmpty();
    _builder.append("{");
    _builder.newLine();
    _builder.append("public:");
    _builder.newLine();
    _builder.append("  ");
    String _name_1 = SkillUtils.getName(skill);
    _builder.append(_name_1, "  ");
    _builder.append("Condition(");
    _builder.newLineIfNotEmpty();
    _builder.append("      ");
    _builder.append("const std::string& name,");
    _builder.newLine();
    _builder.append("      ");
    _builder.append("const std::string & service_name,");
    _builder.newLine();
    _builder.append("      ");
    _builder.append("const BT::NodeConfiguration& conf)");
    _builder.newLine();
    _builder.append("  ");
    _builder.append(": bt_utils::BtConditionNode<");
    String _coordinationIfQn_1 = SkillUtils.getCoordinationIfQn(ops);
    _builder.append(_coordinationIfQn_1, "  ");
    _builder.append(">(name, service_name, conf)");
    _builder.newLineIfNotEmpty();
    _builder.append("    ");
    _builder.append("{");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    {
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(skill.getIns());
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        _builder.append("  ");
        _builder.append("void on_tick() override");
        _builder.newLine();
        _builder.append("  ");
        _builder.append("{");
        _builder.newLine();
        _builder.append("  ");
        _builder.append("\t");
        _builder.append("// ");
        String _protSection_2 = ProtSection.protSection("BtConditionNode on_tick()");
        _builder.append(_protSection_2, "  \t");
        _builder.newLineIfNotEmpty();
        {
          EList<InAttribute> _ins = skill.getIns();
          for(final InAttribute param : _ins) {
            _builder.append("  ");
            _builder.append("  ");
            String _rOS2TypeFromMsgName = SkillUtils.getROS2TypeFromMsgName(SkillUtils.getType(param));
            _builder.append(_rOS2TypeFromMsgName, "    ");
            _builder.append(" ");
            String _name_2 = SkillUtils.getName(param);
            _builder.append(_name_2, "    ");
            _builder.append(";");
            _builder.newLineIfNotEmpty();
            _builder.append("  ");
            _builder.append("  ");
            _builder.append("getInput(\"");
            String _name_3 = SkillUtils.getName(param);
            _builder.append(_name_3, "    ");
            _builder.append("\", ");
            String _name_4 = SkillUtils.getName(param);
            _builder.append(_name_4, "    ");
            _builder.append(");");
            _builder.newLineIfNotEmpty();
            {
              EList<Property> _ownedAttributes = SkillUtils.getType(param).getOwnedAttributes();
              for(final Property prop : _ownedAttributes) {
                _builder.append("  ");
                _builder.append("  ");
                _builder.append("request_->");
                String _name_5 = prop.getName();
                _builder.append(_name_5, "    ");
                _builder.append(" = ");
                String _name_6 = SkillUtils.getName(param);
                _builder.append(_name_6, "    ");
                _builder.append(".");
                String _name_7 = prop.getName();
                _builder.append(_name_7, "    ");
                _builder.append(";");
                _builder.newLineIfNotEmpty();
              }
            }
          }
        }
        _builder.append("  ");
        _builder.append("\t");
        _builder.append("// ");
        String _protSection_3 = ProtSection.protSection();
        _builder.append(_protSection_3, "  \t");
        _builder.newLineIfNotEmpty();
        _builder.append("  ");
        _builder.append("}");
        _builder.newLine();
      }
    }
    _builder.newLine();
    _builder.append("  ");
    _builder.append("BT::NodeStatus check_future(");
    _builder.newLine();
    _builder.append("      ");
    _builder.append("std::shared_future<typename ");
    String _coordinationIfQn_2 = SkillUtils.getCoordinationIfQn(ops);
    _builder.append(_coordinationIfQn_2, "      ");
    _builder.append("::Response::SharedPtr> future_result) override");
    _builder.newLineIfNotEmpty();
    _builder.append("  ");
    _builder.append("{");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("rclcpp::FutureReturnCode rc;");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("rc = rclcpp::spin_until_future_complete(");
    _builder.newLine();
    _builder.append("      ");
    _builder.append("node_,");
    _builder.newLine();
    _builder.append("      ");
    _builder.append("future_result, server_timeout_);");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("if (rc == rclcpp::FutureReturnCode::SUCCESS) {");
    _builder.newLine();
    _builder.append("      ");
    _builder.append("auto res = future_result.get()->data;");
    _builder.newLine();
    _builder.newLine();
    _builder.append("      ");
    _builder.append("return (res == true) ? BT::NodeStatus::SUCCESS : BT::NodeStatus::FAILURE;");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("} else if (rc == rclcpp::FutureReturnCode::TIMEOUT) {");
    _builder.newLine();
    _builder.append("      ");
    _builder.append("RCLCPP_WARN(");
    _builder.newLine();
    _builder.append("        ");
    _builder.append("node_->get_logger(),");
    _builder.newLine();
    _builder.append("        ");
    _builder.append("\"Node timed out while executing service call to %s.\", service_name_.c_str());");
    _builder.newLine();
    _builder.append("      ");
    _builder.append("on_wait_for_result();");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("}");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("return BT::NodeStatus::FAILURE;");
    _builder.newLine();
    _builder.append("  ");
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("  ");
    CharSequence _createProvidedPortsMethod = CreateSkillRealizationCppCode.createProvidedPortsMethod(skill, true);
    _builder.append(_createProvidedPortsMethod, "  ");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("// ");
    String _protSection_4 = ProtSection.protSection("- BtConditionNode other class methods and attributes (if any)");
    _builder.append(_protSection_4);
    _builder.newLineIfNotEmpty();
    _builder.append("// ");
    String _protSection_5 = ProtSection.protSection();
    _builder.append(_protSection_5);
    _builder.newLineIfNotEmpty();
    _builder.append("};");
    _builder.newLine();
    _builder.newLine();
    _builder.append("#include \"behaviortree_cpp_v3/bt_factory.h\"");
    _builder.newLine();
    _builder.append("BT_REGISTER_NODES(factory)");
    _builder.newLine();
    _builder.append("{");
    _builder.newLine();
    _builder.append("  ");
    _builder.append("BT::NodeBuilder builder =");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("[](const std::string & name, const BT::NodeConfiguration & config)");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("{");
    _builder.newLine();
    _builder.append("      ");
    _builder.append("return std::make_unique<");
    String _name_8 = SkillUtils.getName(skill);
    _builder.append(_name_8, "      ");
    _builder.append("Condition>(name, \"");
    _builder.append(serviceName, "      ");
    _builder.append("\", config);");
    _builder.newLineIfNotEmpty();
    _builder.append("    ");
    _builder.append("};");
    _builder.newLine();
    _builder.newLine();
    _builder.append("  ");
    _builder.append("factory.registerBuilder<");
    String _name_9 = SkillUtils.getName(skill);
    _builder.append(_name_9, "  ");
    _builder.append("Condition>(\"");
    String _name_10 = SkillUtils.getName(skill);
    _builder.append(_name_10, "  ");
    _builder.append("\", builder);");
    _builder.newLineIfNotEmpty();
    _builder.append("}");
    _builder.newLine();
    return _builder;
  }

  public static CharSequence createCppSimplestSkillNoInteraction(final SkillDefinition skill) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("// Generated by Papyrus4Robotics");
    _builder.newLine();
    _builder.append("// ...");
    _builder.newLine();
    _builder.newLine();
    _builder.append("#include <string>");
    _builder.newLine();
    _builder.append("// ");
    String _protSection = ProtSection.protSection("ActionNodeBase header files");
    _builder.append(_protSection);
    _builder.newLineIfNotEmpty();
    _builder.append("// ");
    String _protSection_1 = ProtSection.protSection();
    _builder.append(_protSection_1);
    _builder.newLineIfNotEmpty();
    _builder.append("#include \"rclcpp/rclcpp.hpp\"");
    _builder.newLine();
    _builder.append("#include \"bt_utils/generic_types_conversions.hpp\"");
    _builder.newLine();
    {
      LinkedHashSet<DataType> _uniqueSkillParameterTypes = SkillUtils.getUniqueSkillParameterTypes(skill);
      for(final DataType tp : _uniqueSkillParameterTypes) {
        _builder.append("#include \"");
        String _rOS2TypeIncludePath = SkillUtils.getROS2TypeIncludePath(tp);
        _builder.append(_rOS2TypeIncludePath);
        _builder.append("\"");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("#include \"behaviortree_cpp_v3/action_node.h\"");
    _builder.newLine();
    _builder.newLine();
    _builder.append("class ");
    String _name = SkillUtils.getName(skill);
    _builder.append(_name);
    _builder.append("Action : public BT::ActionNodeBase");
    _builder.newLineIfNotEmpty();
    _builder.append("{");
    _builder.newLine();
    _builder.newLine();
    _builder.append("public:");
    _builder.newLine();
    _builder.append("  ");
    _builder.append("// Any TreeNode with ports must have a constructor with this signature");
    _builder.newLine();
    _builder.append("  ");
    String _name_1 = SkillUtils.getName(skill);
    _builder.append(_name_1, "  ");
    _builder.append("Action(");
    _builder.newLineIfNotEmpty();
    _builder.append("    ");
    _builder.append("const std::string& name,");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("const BT::NodeConfiguration& config)");
    _builder.newLine();
    _builder.append("  ");
    _builder.append(": ActionNodeBase(name, config)");
    _builder.newLine();
    _builder.append("  ");
    _builder.append("{");
    _builder.newLine();
    _builder.append("  ");
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("  ");
    CharSequence _createProvidedPortsMethod = CreateSkillRealizationCppCode.createProvidedPortsMethod(skill, false);
    _builder.append(_createProvidedPortsMethod, "  ");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("  ");
    _builder.append("BT::NodeStatus tick() override");
    _builder.newLine();
    _builder.append("  ");
    _builder.append("{");
    _builder.newLine();
    _builder.append("  \t");
    _builder.append("// ");
    String _protSection_2 = ProtSection.protSection("ActionNodeBase tick()");
    _builder.append(_protSection_2, "  \t");
    _builder.newLineIfNotEmpty();
    {
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(skill.getIns());
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        _builder.append("\t");
        _builder.append("// Read from input ports");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("//");
        _builder.newLine();
        {
          EList<InAttribute> _ins = skill.getIns();
          for(final InAttribute param : _ins) {
            _builder.append("\t");
            String _rOS2TypeFromMsgName = SkillUtils.getROS2TypeFromMsgName(SkillUtils.getType(param));
            _builder.append(_rOS2TypeFromMsgName, "\t");
            _builder.append(" ");
            String _name_2 = SkillUtils.getName(param);
            _builder.append(_name_2, "\t");
            _builder.append(";");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("getInput(\"");
            String _name_3 = SkillUtils.getName(param);
            _builder.append(_name_3, "\t");
            _builder.append("\", ");
            String _name_4 = SkillUtils.getName(param);
            _builder.append(_name_4, "\t");
            _builder.append(");");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.newLine();
        _builder.append("\t");
        _builder.append("/* ----------------------------------------------");
        _builder.newLine();
        _builder.append("\t");
        _builder.append(" ");
        _builder.append("* USER CODE HERE USING READINGS FROM INPUT PORTS");
        _builder.newLine();
        _builder.append("\t");
        _builder.append(" ");
        _builder.append("* ----------------------------------------------");
        _builder.newLine();
        _builder.append("\t");
        _builder.append(" ");
        _builder.append("*/");
        _builder.newLine();
      }
    }
    _builder.newLine();
    {
      if ((IterableExtensions.isNullOrEmpty(skill.getIns()) && IterableExtensions.isNullOrEmpty(skill.getOuts()))) {
        _builder.append("\t");
        _builder.append("/* -------------------");
        _builder.newLine();
        _builder.append("\t");
        _builder.append(" ");
        _builder.append("* SOME USER CODE HERE");
        _builder.newLine();
        _builder.append("\t");
        _builder.append(" ");
        _builder.append("* -------------------");
        _builder.newLine();
        _builder.append("\t");
        _builder.append(" ");
        _builder.append("*/");
        _builder.newLine();
      }
    }
    _builder.newLine();
    {
      boolean _isNullOrEmpty_1 = IterableExtensions.isNullOrEmpty(skill.getOuts());
      boolean _not_1 = (!_isNullOrEmpty_1);
      if (_not_1) {
        _builder.append("\t");
        _builder.append("// Share through the blackboard");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("//");
        _builder.newLine();
        {
          EList<OutAttribute> _outs = skill.getOuts();
          for(final OutAttribute param_1 : _outs) {
            _builder.append("\t");
            String _rOS2TypeFromMsgName_1 = SkillUtils.getROS2TypeFromMsgName(SkillUtils.getType(param_1));
            _builder.append(_rOS2TypeFromMsgName_1, "\t");
            _builder.append(" ");
            String _name_5 = SkillUtils.getName(param_1);
            _builder.append(_name_5, "\t");
            _builder.append(";");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("/* -----------------------------------------");
            _builder.newLine();
            _builder.append("\t");
            _builder.append(" ");
            _builder.append("* USER CODE HERE TO INITIALIZE THE VARIABLE");
            _builder.newLine();
            _builder.append("\t");
            _builder.append(" ");
            _builder.append("* -----------------------------------------");
            _builder.newLine();
            _builder.append("\t");
            _builder.append(" ");
            _builder.append("*/");
            _builder.newLine();
            _builder.append("\t");
            _builder.append("setOutput(\"");
            String _name_6 = SkillUtils.getName(param_1);
            _builder.append(_name_6, "\t");
            _builder.append("\", ");
            String _name_7 = SkillUtils.getName(param_1);
            _builder.append(_name_7, "\t");
            _builder.append(");");
            _builder.newLineIfNotEmpty();
            _builder.newLine();
          }
        }
      }
    }
    _builder.append("  \t");
    _builder.append("return BT::NodeStatus::SUCCESS;");
    _builder.newLine();
    _builder.append("  \t");
    _builder.append("// ");
    String _protSection_3 = ProtSection.protSection();
    _builder.append(_protSection_3, "  \t");
    _builder.newLineIfNotEmpty();
    _builder.append("  ");
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("// ");
    String _protSection_4 = ProtSection.protSection("- ActionNodeBase other class methods and attributes (if any)");
    _builder.append(_protSection_4);
    _builder.newLineIfNotEmpty();
    _builder.append("// ");
    String _protSection_5 = ProtSection.protSection();
    _builder.append(_protSection_5);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("};");
    _builder.newLine();
    _builder.newLine();
    _builder.append("#include \"behaviortree_cpp_v3/bt_factory.h\"");
    _builder.newLine();
    _builder.newLine();
    _builder.append("// This function must be implemented in the .cpp file to create");
    _builder.newLine();
    _builder.append("// a plugin that can be loaded at run-time");
    _builder.newLine();
    _builder.append("BT_REGISTER_NODES(factory)");
    _builder.newLine();
    _builder.append("{");
    _builder.newLine();
    _builder.append("    ");
    _builder.append("factory.registerNodeType<");
    String _name_8 = SkillUtils.getName(skill);
    _builder.append(_name_8, "    ");
    _builder.append("Action>(\"");
    String _name_9 = SkillUtils.getName(skill);
    _builder.append(_name_9, "    ");
    _builder.append("\");");
    _builder.newLineIfNotEmpty();
    _builder.append("}");
    _builder.newLine();
    return _builder;
  }

  public static CharSequence createProvidedPortsMethod(final SkillDefinition skill, final boolean need_basic_ports) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("// ");
    String _name = SkillUtils.getName(skill);
    _builder.append(_name);
    _builder.append(" has a constructor in the form (const std::string&, const NodeConfiguration&) => must provide a providedPorts method");
    _builder.newLineIfNotEmpty();
    _builder.append("static BT::PortsList providedPorts()");
    _builder.newLine();
    _builder.append("{");
    _builder.newLine();
    _builder.append("  ");
    _builder.append("return");
    {
      if (need_basic_ports) {
        _builder.append(" providedBasicPorts(");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("  ");
    _builder.append("{");
    _builder.newLine();
    _builder.append("  \t");
    CharSequence _createPortConstructionCommands = CreateSkillRealizationCppCode.createPortConstructionCommands(skill);
    _builder.append(_createPortConstructionCommands, "  \t");
    _builder.newLineIfNotEmpty();
    _builder.append("  ");
    _builder.append("}");
    {
      if (need_basic_ports) {
        _builder.append(")");
      }
    }
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append("}");
    _builder.newLine();
    return _builder;
  }

  public static CharSequence createPortConstructionCommands(final SkillDefinition skill) {
    StringConcatenation _builder = new StringConcatenation();
    {
      EList<InAttribute> _ins = skill.getIns();
      EList<OutAttribute> _outs = skill.getOuts();
      Iterable<SkillParameter> _plus = Iterables.<SkillParameter>concat(_ins, _outs);
      boolean _hasElements = false;
      for(final SkillParameter param : _plus) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(",", "");
        }
        {
          boolean _contains = skill.getIns().contains(param);
          if (_contains) {
            _builder.append("BT::InputPort<");
          } else {
            _builder.append("BT::OutputPort<");
          }
        }
        String _rOS2TypeFromMsgName = SkillUtils.getROS2TypeFromMsgName(SkillUtils.getType(param));
        _builder.append(_rOS2TypeFromMsgName);
        _builder.append(">(\"");
        String _name = SkillUtils.getName(param);
        _builder.append(_name);
        _builder.append("\")");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  public static void genCode(final IPFileSystemAccess fileAccess, final Map<SkillDefinition, SkillSemantic> skdefToSemanticsMap, final Map<Interface, String> serviceToNameMap) {
    try {
      Set<Map.Entry<SkillDefinition, SkillSemantic>> _entrySet = skdefToSemanticsMap.entrySet();
      for (final Map.Entry<SkillDefinition, SkillSemantic> entry : _entrySet) {
        {
          final SkillDefinition definition = entry.getKey();
          final SkillSemantic semantics = entry.getValue();
          int _size = semantics.getOperational().size();
          boolean _greaterThan = (_size > 1);
          if (_greaterThan) {
            String _msgAbortMultipleOperationalStatesIsUnsopported = CreateSkillRealizationCppCode.msgAbortMultipleOperationalStatesIsUnsopported();
            throw new TransformationException(_msgAbortMultipleOperationalStatesIsUnsopported);
          } else {
            final SkillOperationalState ops = SkillUtils.getFirstOpState(semantics);
            String _realizationFileName = SkillUtils.realizationFileName(definition);
            final String skillRealizationFileName = (_realizationFileName + ".cpp");
            boolean _doesConfigAndCoordOfComponents = SkillUtils.doesConfigAndCoordOfComponents(ops);
            if (_doesConfigAndCoordOfComponents) {
              final Interface cIf = SkillUtils.getCompInterface(ops);
              boolean _containsKey = serviceToNameMap.containsKey(cIf);
              boolean _not = (!_containsKey);
              if (_not) {
                String _msgAbortSystemHasNotCompIf = CreateSkillRealizationCppCode.msgAbortSystemHasNotCompIf(
                  cIf.getName());
                throw new TransformationException(_msgAbortSystemHasNotCompIf);
              }
              if (((!InteractionUtils.isAction(InteractionUtils.getCommunicationPattern(cIf))) && (!InteractionUtils.isQuery(InteractionUtils.getCommunicationPattern(cIf))))) {
                String _msgAbortCompIfNotActionNorQuery = CreateSkillRealizationCppCode.msgAbortCompIfNotActionNorQuery(
                  cIf.getName());
                throw new TransformationException(_msgAbortCompIfNotActionNorQuery);
              }
              boolean _isAction = InteractionUtils.isAction(InteractionUtils.getCommunicationPattern(cIf));
              if (_isAction) {
                final String actionName = serviceToNameMap.get(cIf);
                fileAccess.generateFile(skillRealizationFileName, CreateSkillRealizationCppCode.createCppSimplestActionSkill(definition, ops, actionName).toString());
              } else {
                final String serviceName = serviceToNameMap.get(cIf);
                fileAccess.generateFile(skillRealizationFileName, CreateSkillRealizationCppCode.createCppSimplestQuerySkill(definition, ops, serviceName).toString());
              }
            } else {
              fileAccess.generateFile(skillRealizationFileName, CreateSkillRealizationCppCode.createCppSimplestSkillNoInteraction(definition).toString());
            }
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
