/**
 * Copyright (c) 2015 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 */
package org.eclipse.papyrus.dev.assistants.codegen.generator;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import javax.inject.Inject;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmf.codegen.gmfgen.ElementType;
import org.eclipse.gmf.codegen.gmfgen.GenChildNode;
import org.eclipse.gmf.codegen.gmfgen.GenCommonBase;
import org.eclipse.gmf.codegen.gmfgen.GenCompartment;
import org.eclipse.gmf.codegen.gmfgen.GenDiagram;
import org.eclipse.gmf.codegen.gmfgen.GenEditorGenerator;
import org.eclipse.gmf.codegen.gmfgen.GenLink;
import org.eclipse.gmf.codegen.gmfgen.GenNode;
import org.eclipse.gmf.codegen.gmfgen.GenTopLevelNode;
import org.eclipse.gmf.codegen.gmfgen.LinkModelFacet;
import org.eclipse.gmf.codegen.gmfgen.ModelFacet;
import org.eclipse.gmf.codegen.gmfgen.TypeModelFacet;
import org.eclipse.papyrus.dev.assistants.codegen.generator.ModelingAssistantProviderRule;
import org.eclipse.papyrus.infra.filters.Filter;
import org.eclipse.papyrus.infra.gmfdiag.assistant.AssistantFactory;
import org.eclipse.papyrus.infra.gmfdiag.assistant.AssistedElementTypeFilter;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ConnectionAssistant;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ElementTypeFilter;
import org.eclipse.papyrus.infra.gmfdiag.assistant.PopupAssistant;
import org.eclipse.papyrus.uml.profile.assistants.generator.FiltersUtil;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * GMFGen node/link to Assistant transformation rule.
 */
@SuppressWarnings("all")
public class GMFGenToAssistantRule {
  @Extension
  private static AssistantFactory assistantFactory = AssistantFactory.eINSTANCE;
  
  @Inject
  @Extension
  private FiltersUtil _filtersUtil;
  
  @Inject
  @Extension
  private ModelingAssistantProviderRule _modelingAssistantProviderRule;
  
  public String cleanedName(final String name) {
    String _xblockexpression = null;
    {
      int _lastIndexOf = name.lastIndexOf(".");
      int _plus = (_lastIndexOf + 1);
      String result = name.substring(_plus);
      boolean _matches = result.matches("^.*_\\d+$");
      if (_matches) {
        result = result.substring(0, result.lastIndexOf("_"));
      }
      _xblockexpression = result = result.replaceAll("([a-z])([A-Z])", "$1 $2");
    }
    return _xblockexpression;
  }
  
  protected GenEditorGenerator _rootEditor(final EObject element) {
    EObject _eContainer = element.eContainer();
    GenEditorGenerator _rootEditor = null;
    if (_eContainer!=null) {
      _rootEditor=this.rootEditor(_eContainer);
    }
    return _rootEditor;
  }
  
  protected GenEditorGenerator _rootEditor(final GenDiagram diagram) {
    return diagram.getEditorGen();
  }
  
  public Iterable<GenNode> validNodes(final GenDiagram diagram) {
    final Function1<GenNode, Boolean> _function = new Function1<GenNode, Boolean>() {
      @Override
      public Boolean apply(final GenNode it) {
        TypeModelFacet _modelFacet = it.getModelFacet();
        return Boolean.valueOf((!Objects.equal(_modelFacet, null)));
      }
    };
    return IterableExtensions.<GenNode>filter(diagram.getAllNodes(), _function);
  }
  
  public Iterable<GenLink> validLinks(final GenDiagram diagram) {
    final Function1<GenLink, Boolean> _function = new Function1<GenLink, Boolean>() {
      @Override
      public Boolean apply(final GenLink it) {
        LinkModelFacet _modelFacet = it.getModelFacet();
        return Boolean.valueOf((!Objects.equal(_modelFacet, null)));
      }
    };
    return IterableExtensions.<GenLink>filter(diagram.getLinks(), _function);
  }
  
  /**
   * Head of the polymorphic transformation.
   */
  protected ElementTypeFilter _toElementTypeFilter(final EObject element) {
    return null;
  }
  
  protected ElementTypeFilter _toElementTypeFilter(final GenDiagram diagram) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(diagram);
    final ElementTypeFilter _result;
    synchronized (_createCache_toElementTypeFilter) {
      if (_createCache_toElementTypeFilter.containsKey(_cacheKey)) {
        return _createCache_toElementTypeFilter.get(_cacheKey);
      }
      ElementTypeFilter _createElementTypeFilter = GMFGenToAssistantRule.assistantFactory.createElementTypeFilter();
      _result = _createElementTypeFilter;
      _createCache_toElementTypeFilter.put(_cacheKey, _result);
    }
    _init_toElementTypeFilter(_result, diagram);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, ElementTypeFilter> _createCache_toElementTypeFilter = CollectionLiterals.newHashMap();
  
  private void _init_toElementTypeFilter(final ElementTypeFilter it, final GenDiagram diagram) {
    it.setElementTypeID(diagram.getElementType().getUniqueIdentifier());
    it.setName(this.cleanedName(diagram.getDomainDiagramElement().getFormattedName()));
    this._modelingAssistantProviderRule.toModelingAssistantProvider(diagram.getEditorGen()).getOwnedFilters().add(it);
  }
  
  private ElementType getElementType(final ModelFacet facet) {
    EObject _eContainer = facet.eContainer();
    return ((GenCommonBase) _eContainer).getElementType();
  }
  
  protected ElementTypeFilter _toElementTypeFilter(final ModelFacet modelFacet) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(modelFacet);
    final ElementTypeFilter _result;
    synchronized (_createCache_toElementTypeFilter_1) {
      if (_createCache_toElementTypeFilter_1.containsKey(_cacheKey)) {
        return _createCache_toElementTypeFilter_1.get(_cacheKey);
      }
      ElementTypeFilter _createElementTypeFilter = GMFGenToAssistantRule.assistantFactory.createElementTypeFilter();
      _result = _createElementTypeFilter;
      _createCache_toElementTypeFilter_1.put(_cacheKey, _result);
    }
    _init_toElementTypeFilter_1(_result, modelFacet);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, ElementTypeFilter> _createCache_toElementTypeFilter_1 = CollectionLiterals.newHashMap();
  
  private void _init_toElementTypeFilter_1(final ElementTypeFilter it, final ModelFacet modelFacet) {
    it.setElementTypeID(this.getElementType(modelFacet).getUniqueIdentifier());
    it.setName(this.getElementType(modelFacet).getDisplayName());
    this._modelingAssistantProviderRule.toModelingAssistantProvider(this.rootEditor(modelFacet)).getOwnedFilters().add(it);
  }
  
  /**
   * Head of polymorphic transformation.
   */
  protected PopupAssistant _toPopupAssistant(final GenNode node) {
    return null;
  }
  
  protected PopupAssistant _toPopupAssistant(final GenTopLevelNode node) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(node);
    final PopupAssistant _result;
    synchronized (_createCache_toPopupAssistant) {
      if (_createCache_toPopupAssistant.containsKey(_cacheKey)) {
        return _createCache_toPopupAssistant.get(_cacheKey);
      }
      PopupAssistant _createPopupAssistant = GMFGenToAssistantRule.assistantFactory.createPopupAssistant();
      _result = _createPopupAssistant;
      _createCache_toPopupAssistant.put(_cacheKey, _result);
    }
    _init_toPopupAssistant(_result, node);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, PopupAssistant> _createCache_toPopupAssistant = CollectionLiterals.newHashMap();
  
  private void _init_toPopupAssistant(final PopupAssistant it, final GenTopLevelNode node) {
    it.setElementTypeID(this.getElementType(node.getModelFacet()).getUniqueIdentifier());
    it.setFilter(this.toElementTypeFilter(node.getDiagram()));
  }
  
  protected PopupAssistant _toPopupAssistant(final GenChildNode node) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(node);
    final PopupAssistant _result;
    synchronized (_createCache_toPopupAssistant_1) {
      if (_createCache_toPopupAssistant_1.containsKey(_cacheKey)) {
        return _createCache_toPopupAssistant_1.get(_cacheKey);
      }
      PopupAssistant _createPopupAssistant = GMFGenToAssistantRule.assistantFactory.createPopupAssistant();
      _result = _createPopupAssistant;
      _createCache_toPopupAssistant_1.put(_cacheKey, _result);
    }
    _init_toPopupAssistant_1(_result, node);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, PopupAssistant> _createCache_toPopupAssistant_1 = CollectionLiterals.newHashMap();
  
  private void _init_toPopupAssistant_1(final PopupAssistant it, final GenChildNode node) {
    it.setElementTypeID(this.getElementType(node.getModelFacet()).getUniqueIdentifier());
    Filter _reduce = this._filtersUtil.reduce(this.createPossibleOwnersFilter(node));
    AssistedElementTypeFilter _assistedElementTypeFilter = this.assistedElementTypeFilter(this.rootEditor(node));
    Filter _and = this._filtersUtil.operator_and(_reduce, _assistedElementTypeFilter);
    it.setFilter(_and);
    boolean _isCompound = this._filtersUtil.isCompound(it.getFilter());
    if (_isCompound) {
      it.setOwnedFilter(it.getFilter());
    }
  }
  
  private Filter createPossibleOwnersFilter(final GenChildNode node) {
    final Function1<GenNode, Boolean> _function = new Function1<GenNode, Boolean>() {
      @Override
      public Boolean apply(final GenNode it) {
        final Function1<GenCompartment, Boolean> _function = new Function1<GenCompartment, Boolean>() {
          @Override
          public Boolean apply(final GenCompartment it) {
            return Boolean.valueOf(node.getContainers().contains(it));
          }
        };
        return Boolean.valueOf(IterableExtensions.<GenCompartment>exists(it.getCompartments(), _function));
      }
    };
    final Function1<GenNode, ElementTypeFilter> _function_1 = new Function1<GenNode, ElementTypeFilter>() {
      @Override
      public ElementTypeFilter apply(final GenNode it) {
        TypeModelFacet _modelFacet = it.getModelFacet();
        ElementTypeFilter _elementTypeFilter = null;
        if (_modelFacet!=null) {
          _elementTypeFilter=GMFGenToAssistantRule.this.toElementTypeFilter(_modelFacet);
        }
        return _elementTypeFilter;
      }
    };
    final Function2<Filter, ElementTypeFilter, Filter> _function_2 = new Function2<Filter, ElementTypeFilter, Filter>() {
      @Override
      public Filter apply(final Filter filter, final ElementTypeFilter parentFilter) {
        return GMFGenToAssistantRule.this._filtersUtil.operator_or(filter, parentFilter);
      }
    };
    return IterableExtensions.<ElementTypeFilter, Filter>fold(IterableExtensions.<ElementTypeFilter>filterNull(IterableExtensions.<GenNode, ElementTypeFilter>map(IterableExtensions.<GenNode>filter(node.getDiagram().getAllNodes(), _function), _function_1)), null, _function_2);
  }
  
  public ConnectionAssistant toConnectionAssistant(final GenLink link) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(link);
    final ConnectionAssistant _result;
    synchronized (_createCache_toConnectionAssistant) {
      if (_createCache_toConnectionAssistant.containsKey(_cacheKey)) {
        return _createCache_toConnectionAssistant.get(_cacheKey);
      }
      ConnectionAssistant _createConnectionAssistant = GMFGenToAssistantRule.assistantFactory.createConnectionAssistant();
      _result = _createConnectionAssistant;
      _createCache_toConnectionAssistant.put(_cacheKey, _result);
    }
    _init_toConnectionAssistant(_result, link);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, ConnectionAssistant> _createCache_toConnectionAssistant = CollectionLiterals.newHashMap();
  
  private void _init_toConnectionAssistant(final ConnectionAssistant it, final GenLink link) {
    it.setElementTypeID(this.getElementType(link.getModelFacet()).getUniqueIdentifier());
    Filter _reduce = this._filtersUtil.reduce(this.createPossibleSourcesFilter(link));
    AssistedElementTypeFilter _assistedElementTypeFilter = this.assistedElementTypeFilter(this.rootEditor(link));
    Filter _and = this._filtersUtil.operator_and(_reduce, _assistedElementTypeFilter);
    it.setSourceFilter(_and);
    Filter _reduce_1 = this._filtersUtil.reduce(this.createPossibleTargetsFilter(link));
    AssistedElementTypeFilter _assistedElementTypeFilter_1 = this.assistedElementTypeFilter(this.rootEditor(link));
    Filter _and_1 = this._filtersUtil.operator_and(_reduce_1, _assistedElementTypeFilter_1);
    it.setTargetFilter(_and_1);
    boolean _isCompound = this._filtersUtil.isCompound(it.getSourceFilter());
    if (_isCompound) {
      it.setOwnedSourceFilter(it.getSourceFilter());
    }
    boolean _isCompound_1 = this._filtersUtil.isCompound(it.getTargetFilter());
    if (_isCompound_1) {
      it.setOwnedTargetFilter(it.getTargetFilter());
    }
  }
  
  private Filter createPossibleSourcesFilter(final GenLink link) {
    final Function1<GenNode, ElementTypeFilter> _function = new Function1<GenNode, ElementTypeFilter>() {
      @Override
      public ElementTypeFilter apply(final GenNode it) {
        TypeModelFacet _modelFacet = it.getModelFacet();
        ElementTypeFilter _elementTypeFilter = null;
        if (_modelFacet!=null) {
          _elementTypeFilter=GMFGenToAssistantRule.this.toElementTypeFilter(_modelFacet);
        }
        return _elementTypeFilter;
      }
    };
    final Function2<Filter, ElementTypeFilter, Filter> _function_1 = new Function2<Filter, ElementTypeFilter, Filter>() {
      @Override
      public Filter apply(final Filter filter, final ElementTypeFilter parentFilter) {
        return GMFGenToAssistantRule.this._filtersUtil.operator_or(filter, parentFilter);
      }
    };
    return IterableExtensions.<ElementTypeFilter, Filter>fold(IterableExtensions.<ElementTypeFilter>filterNull(IterableExtensions.<GenNode, ElementTypeFilter>map(Iterables.<GenNode>filter(link.getSources(), GenNode.class), _function)), null, _function_1);
  }
  
  private Filter createPossibleTargetsFilter(final GenLink link) {
    final Function1<GenNode, ElementTypeFilter> _function = new Function1<GenNode, ElementTypeFilter>() {
      @Override
      public ElementTypeFilter apply(final GenNode it) {
        TypeModelFacet _modelFacet = it.getModelFacet();
        ElementTypeFilter _elementTypeFilter = null;
        if (_modelFacet!=null) {
          _elementTypeFilter=GMFGenToAssistantRule.this.toElementTypeFilter(_modelFacet);
        }
        return _elementTypeFilter;
      }
    };
    final Function2<Filter, ElementTypeFilter, Filter> _function_1 = new Function2<Filter, ElementTypeFilter, Filter>() {
      @Override
      public Filter apply(final Filter filter, final ElementTypeFilter parentFilter) {
        return GMFGenToAssistantRule.this._filtersUtil.operator_or(filter, parentFilter);
      }
    };
    return IterableExtensions.<ElementTypeFilter, Filter>fold(IterableExtensions.<ElementTypeFilter>filterNull(IterableExtensions.<GenNode, ElementTypeFilter>map(Iterables.<GenNode>filter(link.getTargets(), GenNode.class), _function)), null, _function_1);
  }
  
  private AssistedElementTypeFilter assistedElementTypeFilter(final GenEditorGenerator genEditor) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(genEditor);
    final AssistedElementTypeFilter _result;
    synchronized (_createCache_assistedElementTypeFilter) {
      if (_createCache_assistedElementTypeFilter.containsKey(_cacheKey)) {
        return _createCache_assistedElementTypeFilter.get(_cacheKey);
      }
      AssistedElementTypeFilter _createAssistedElementTypeFilter = GMFGenToAssistantRule.assistantFactory.createAssistedElementTypeFilter();
      _result = _createAssistedElementTypeFilter;
      _createCache_assistedElementTypeFilter.put(_cacheKey, _result);
    }
    _init_assistedElementTypeFilter(_result, genEditor);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, AssistedElementTypeFilter> _createCache_assistedElementTypeFilter = CollectionLiterals.newHashMap();
  
  private void _init_assistedElementTypeFilter(final AssistedElementTypeFilter it, final GenEditorGenerator genEditor) {
    this._modelingAssistantProviderRule.toModelingAssistantProvider(genEditor).getOwnedFilters().add(it);
  }
  
  public GenEditorGenerator rootEditor(final EObject diagram) {
    if (diagram instanceof GenDiagram) {
      return _rootEditor((GenDiagram)diagram);
    } else if (diagram != null) {
      return _rootEditor(diagram);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(diagram).toString());
    }
  }
  
  public ElementTypeFilter toElementTypeFilter(final EObject diagram) {
    if (diagram instanceof GenDiagram) {
      return _toElementTypeFilter((GenDiagram)diagram);
    } else if (diagram instanceof ModelFacet) {
      return _toElementTypeFilter((ModelFacet)diagram);
    } else if (diagram != null) {
      return _toElementTypeFilter(diagram);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(diagram).toString());
    }
  }
  
  public PopupAssistant toPopupAssistant(final GenNode node) {
    if (node instanceof GenChildNode) {
      return _toPopupAssistant((GenChildNode)node);
    } else if (node instanceof GenTopLevelNode) {
      return _toPopupAssistant((GenTopLevelNode)node);
    } else if (node != null) {
      return _toPopupAssistant(node);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(node).toString());
    }
  }
}
