/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: RDFXMLContentHandler.java,v 1.2 2007/03/18 08:39:03 lzhang Exp $
 */

package org.eclipse.eodm.rdf.resource.parser.xml;

import java.util.HashSet;
import java.util.Stack;

import org.eclipse.eodm.vocabulary.RDF;
import org.eclipse.eodm.vocabulary.RDFS;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;

/**
 * The {@link org.xml.sax.ContentHandler}implementation.
 * <p>
 * Handle the RDF data in the form of XML syntax.
 * </p>
 */
public class RDFXMLContentHandler implements ContentHandler {
    /**
     * The RDFXMLParserImpl which deals with new element and text
     */
    private RDFXMLParserImpl rdfParser;

    /**
     * The state stack
     */
    private final Stack stateStack;

    /**
     * The text content buffer
     */
    private StringBuffer charBuffer;

    /**
     * XML literal namespace
     */
    private HashSet literalNamespaces = new HashSet();

    /**
     * All the states defined in the RDF parser. <br>
     * 0 - unkown state, a sign marks the start of a document <br>
     * 1 - RDF content start point <br>
     * 2 - an RDF Node Element <br>
     * 3 - an RDF Property Element <br>
     * 4 - an RDF text node <br>
     */
    public final static int STATE_UNKNOWN = 0;

    public final static int STATE_RDF = 1;

    public final static int STATE_NODE_ELEMENT = 2;

    public final static int STATE_PROPERTY_ELEMENT = 3;

    public final static int STATE_TEXT = 4;

    /* box all the states into Integer Object */
    private final static Integer[] INTEGER_STATES = new Integer[] {
            new Integer(STATE_UNKNOWN), new Integer(STATE_RDF),
            new Integer(STATE_NODE_ELEMENT),
            new Integer(STATE_PROPERTY_ELEMENT), new Integer(STATE_TEXT) };

    /**
     * Construct an RDFXMLContentHandler.
     * 
     * @param container
     *            the rdf syntax handler
     */
    public RDFXMLContentHandler(RDFXMLParserImpl container) {
        rdfParser = container;
        stateStack = new Stack();
        charBuffer = new StringBuffer();
    }

    public void endDocument() throws SAXException {
        // Check to see that the stacks are empty.
        if (stateStack.size() != 1) {
            rdfParser.error("stateStack not empty");
        }
        if (rdfParser.popElement() != null) {
            rdfParser.error("_Element stack not empty");
        }
    }

    public void startDocument() throws SAXException {
    }

    public void startPrefixMapping(String prefix, String uri)
            throws SAXException {
        rdfParser.addNamespace(prefix, uri);
    }

    public void endPrefixMapping(String prefix) throws SAXException {
        rdfParser.removeNamespace(prefix);
    }

    /**
     * Deal with a new element in RDF document
     */
    public void startElement(String namespaceURI, String localName,
            String qName, Attributes atts) throws SAXException {

        int currentState = getTopState();
        if (currentState == STATE_UNKNOWN) {
            if (localName.equals(RDF.S_RDF)
                && namespaceURI.equals(RDF.NAMESPACE)) {
                pushState(STATE_RDF);
                currentState = STATE_RDF;
            } else {
                // The <RDF> tag is not mandatory, if missing, then just move to
                // the STATE_NODE.
                pushState(STATE_NODE_ELEMENT);
                currentState = STATE_NODE_ELEMENT;
            }
        }

        String _xmlBase = atts.getValue("xml:base");
        // Register XML base
        if (_xmlBase != null) {
            rdfParser.pushXmlBase(_xmlBase);
        } else {
            rdfParser.xmlBaseURI.push(rdfParser.xmlBaseURI.peek());
        }

        // Determine the type of the element
        switch (currentState) {
        case STATE_RDF:
            pushState(STATE_NODE_ELEMENT);
            break;
        case STATE_NODE_ELEMENT:
            rdfParser.addNodeElement(namespaceURI, localName, atts);
            pushState(STATE_PROPERTY_ELEMENT);
            break;
        case STATE_PROPERTY_ELEMENT:
            rdfParser.addPropertyElement(namespaceURI, localName, atts);
            pushState(STATE_NODE_ELEMENT);
            String parseType = atts.getValue(RDF.NAMESPACE, RDF.S_PARSETYPE);
            if (parseType != null) {
                if (RDFS.C_LITERAL.equals(parseType))
                    pushState(STATE_TEXT);
                else if (RDFS.C_RESOURCE.equals(parseType))
                    pushState(STATE_PROPERTY_ELEMENT);
            }
            break;
        case STATE_TEXT:
            // For XMLLiteral
            if (qName.startsWith(namespaceURI)) {
                charBuffer.append('<').append(qName).append('>');
            } else {
                // Need to output the name spaces in the XML tags.
                String prefix = null;
                int prefixLenght = qName.indexOf(':');
                if (prefixLenght > 0) {
                    prefix = qName.substring(0, prefixLenght);
                } else {
                    prefix = "";
                }
                if (literalNamespaces.contains(prefix)) {
                    charBuffer.append('<').append(qName).append('>');
                } else {
                    if (prefix.length() > 0) {
                        charBuffer.append('<').append(qName).append(" xmlns:")
                                .append(prefix).append("=\\\"").append(
                                        namespaceURI).append("\\\">");
                    } else {
                        charBuffer.append('<').append(qName).append(" xmlns")
                                .append(prefix).append("=\\\"").append(
                                        namespaceURI).append("\\\">");
                    }
                    literalNamespaces.add(prefix);
                }
            }
            pushState(STATE_TEXT);
            rdfParser.pushElement((Element) null);
            break;
        default:
            break;
        }
    }

    /**
     * Deal with the last element
     */
    public void endElement(String namespaceURI, String localName, String qName)
            throws SAXException {

        // pop up implicit blank nodes
        Element e = rdfParser.peekElement();
        if ((e != null)
            && (e instanceof NodeElement) && !((NodeElement) e).isExplicit()) {
            popState();
            rdfParser.popElement();
        }

        if (getTopState() == STATE_TEXT) {
            // Need to add the closing tag, only if this is still within the
            // literal.
            if (((Integer) (stateStack.get(stateStack.size() - 2))).intValue() == STATE_TEXT) {
                charBuffer.append("</").append(qName).append('>');
            } else {
                // Need to pop off the dummy state.
                rdfParser.removeElement();
                popState();
            }
        }
        String text = null;
        boolean gotNonEmptyString = charBuffer.length() > 0;
        if (gotNonEmptyString) {
            text = charBuffer.toString();
        }

        /* Maintain the state machine */
        popState();
        int currentState = getTopState();
        if (currentState > STATE_RDF) {
            /*
             * If the rdf parsing state is in Literal context, all the chars
             * following should be treated as RDF Literal
             */
            if ((currentState != STATE_TEXT) && gotNonEmptyString) {
                rdfParser.addText(text);
                charBuffer = new StringBuffer();
                literalNamespaces.clear();
            }
            rdfParser.removeElement();
        }
        rdfParser.popXmlBase();
    }

    public void setDocumentLocator(Locator locator) {
        rdfParser.setLocator(locator);
    }

    /**
     * Deal with characters from SAX parser
     */
    public void characters(char ch[], int start, int length)
            throws SAXException {
        int i = 0;
        if (getTopState() == STATE_TEXT) {
            charBuffer.append(ch, start, length);
        } else if ((getTopState() == STATE_NODE_ELEMENT)) {
            while (i < length) {
                if (ch[start + i] == '\n') {
                    i++;
                } else if (ch[start + i] == '\t') {
                    i++;
                } else if (ch[start + i] == ' ') {
                    i++;
                } else {
                    break;
                }
            }
            if (i < length) {
                charBuffer.append(ch, start, length);
            }
        }
    }

    public void ignorableWhitespace(char ch[], int start, int length)
            throws SAXException {
    }

    public void processingInstruction(String target, String data)
            throws SAXException {
    }

    public void skippedEntity(String name) throws SAXException {
    }

    /**
     * Push the state stack
     * 
     * @param state
     *            the current state
     */
    void pushState(int state) {
        stateStack.push(getIntegerState(state));
    }

    /**
     * Popup an object from the state stack
     * 
     * @return a state object
     */
    private int popState() {
        if (!stateStack.isEmpty())
            return ((Integer) stateStack.pop()).intValue();
        else
            return STATE_UNKNOWN;
    }

    /**
     * Get the current state
     * 
     * @return the current state
     */
    private int getTopState() {
        if (!stateStack.isEmpty())
            return ((Integer) stateStack.peek()).intValue();
        else
            return STATE_UNKNOWN;
    }

    /**
     * Get a state object by a given int
     * 
     * @param state
     *            the input int
     * @return the state object
     */
    private Integer getIntegerState(int state) {
        if (state > INTEGER_STATES.length - 1) {
            return INTEGER_STATES[STATE_UNKNOWN];
        } else {
            return INTEGER_STATES[state];
        }
    }

}