/*******************************************************************************
 * Copyright (c) 2018, 2024 ArSysOp
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * https://www.eclipse.org/legal/epl-2.0/.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     ArSysOp - initial API and implementation
 *******************************************************************************/
package org.eclipse.passage.loc.dashboard.ui.panel;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;

import org.eclipse.e4.core.contexts.EclipseContextFactory;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.e4.ui.di.Focus;
import org.eclipse.e4.ui.di.Persist;
import org.eclipse.e4.ui.di.UIEventTopic;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.passage.lic.agreements.model.api.Agreement;
import org.eclipse.passage.lic.agreements.model.api.AgreementGroup;
import org.eclipse.passage.lic.emf.meta.ComposableClassMetadata;
import org.eclipse.passage.lic.features.model.api.Feature;
import org.eclipse.passage.lic.features.model.api.FeatureSet;
import org.eclipse.passage.lic.features.model.api.FeatureVersion;
import org.eclipse.passage.lic.internal.agreements.model.AgreementsClassMetadata;
import org.eclipse.passage.lic.internal.features.model.FeaturesClassMetadata;
import org.eclipse.passage.lic.internal.licenses.model.LicensesClassMetadata;
import org.eclipse.passage.lic.internal.products.model.ProductsClassMetadata;
import org.eclipse.passage.lic.internal.users.model.UsersClassMetadata;
import org.eclipse.passage.lic.licenses.model.api.LicensePlan;
import org.eclipse.passage.lic.products.model.api.Product;
import org.eclipse.passage.lic.products.model.api.ProductLine;
import org.eclipse.passage.lic.products.model.api.ProductVersion;
import org.eclipse.passage.lic.products.model.api.ProductVersionFeature;
import org.eclipse.passage.lic.users.model.api.User;
import org.eclipse.passage.lic.users.model.api.UserOrigin;
import org.eclipse.passage.loc.internal.agreements.AgreementRegistry;
import org.eclipse.passage.loc.internal.agreements.AgreementRegistryEvents;
import org.eclipse.passage.loc.internal.features.FeatureRegistry;
import org.eclipse.passage.loc.internal.features.FeatureRegistryEvents;
import org.eclipse.passage.loc.internal.licenses.LicenseRegistry;
import org.eclipse.passage.loc.internal.licenses.LicenseRegistryEvents;
import org.eclipse.passage.loc.internal.products.ProductRegistry;
import org.eclipse.passage.loc.internal.products.ProductRegistryEvents;
import org.eclipse.passage.loc.internal.users.UserRegistry;
import org.eclipse.passage.loc.internal.users.UserRegistryEvents;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.osgi.framework.FrameworkUtil;

@SuppressWarnings("unused")
public class DashboardPanelPart {

	private final FeatureRegistry features;
	private final ProductRegistry products;
	private final UserRegistry users;
	private final AgreementRegistry agreements;
	private final LicenseRegistry licenses;
	private final DashboardPanelAdvisor dashboard;

	@Inject
	public DashboardPanelPart(IEclipseContext context) {
		this.features = context.get(FeatureRegistry.class);
		this.products = context.get(ProductRegistry.class);
		this.users = context.get(UserRegistry.class);
		this.agreements = context.get(AgreementRegistry.class);
		this.licenses = context.get(LicenseRegistry.class);
		this.dashboard = dashboard(context);
	}

	private DashboardPanelAdvisor dashboard(IEclipseContext context) {
		DashboardPanelAdvisor advisor = context.get(DashboardPanelAdvisor.class);
		if (advisor == null) {
			advisor = new DefaultDashboardPanelAdvisor();
		}
		return advisor;
	}

	@PostConstruct
	public void postConstruct(Composite parent, IEclipseContext context) {
		dashboard.init(context);
		Composite area = new Composite(parent, SWT.NONE);
		area.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).create());
		area.setLayout(GridLayoutFactory.swtDefaults().create());
		createHeaderInfo(area);
		createFeatureInfo(area);
		createProductInfo(area);
		createUserInfo(area);
		createAgreementInfo(area);
		createLicenseInfo(area);
		createFooterInfo(area);
		// FIXME: replace this with OSGi component registration
		ComposableClassMetadata metadata = EclipseContextFactory
				.getServiceContext(FrameworkUtil.getBundle(getClass()).getBundleContext())
				.get(ComposableClassMetadata.class);
		metadata.consider(new FeaturesClassMetadata());
		metadata.consider(new ProductsClassMetadata());
		metadata.consider(new UsersClassMetadata());
		metadata.consider(new AgreementsClassMetadata());
		metadata.consider(new LicensesClassMetadata());
	}

	protected void createHeaderInfo(Composite parent) {
		dashboard.createHeaderInfo(parent);
	}

	protected void createFeatureInfo(Composite parent) {
		dashboard.createFeatureInfo(parent, features);
	}

	protected void createProductInfo(Composite parent) {
		dashboard.createProductInfo(parent, products);
	}

	protected void createUserInfo(Composite parent) {
		dashboard.createUserInfo(parent, users);
	}

	protected void createAgreementInfo(Composite parent) {
		dashboard.createAgreementInfo(parent, agreements);
	}

	protected void createLicenseInfo(Composite parent) {
		dashboard.createLicenseInfo(parent, licenses);
	}

	protected void createFooterInfo(Composite parent) {
		dashboard.createFooterInfo(parent);
	}

	@Inject
	@Optional
	public void createdFeatureSet(@UIEventTopic(FeatureRegistryEvents.FEATURE_SET_CREATE) FeatureSet input) {
		dashboard.updateFeatureInfo(features);
	}

	@Inject
	@Optional
	public void deletedFeatureSet(@UIEventTopic(FeatureRegistryEvents.FEATURE_SET_DELETE) FeatureSet input) {
		dashboard.updateFeatureInfo(features);
	}

	@Inject
	@Optional
	public void createdFeature(@UIEventTopic(FeatureRegistryEvents.FEATURE_CREATE) Feature input) {
		dashboard.updateFeatureInfo(features);
	}

	@Inject
	@Optional
	public void deletedFeature(@UIEventTopic(FeatureRegistryEvents.FEATURE_DELETE) Feature input) {
		dashboard.updateFeatureInfo(features);
	}

	@Inject
	@Optional
	public void createdFeatureVersion(
			@UIEventTopic(FeatureRegistryEvents.FEATURE_VERSION_CREATE) FeatureVersion input) {
		dashboard.updateFeatureInfo(features);
	}

	@Inject
	@Optional
	public void deletedFeatureVersion(
			@UIEventTopic(FeatureRegistryEvents.FEATURE_VERSION_DELETE) FeatureVersion input) {
		dashboard.updateFeatureInfo(features);
	}

	@Inject
	@Optional
	public void createdProductLine(@UIEventTopic(ProductRegistryEvents.PRODUCT_LINE_CREATE) ProductLine input) {
		dashboard.updateProductInfo(products);
	}

	@Inject
	@Optional
	public void deletedProductLine(@UIEventTopic(ProductRegistryEvents.PRODUCT_LINE_DELETE) ProductLine input) {
		dashboard.updateProductInfo(products);
	}

	@Inject
	@Optional
	public void createdProduct(@UIEventTopic(ProductRegistryEvents.PRODUCT_CREATE) Product input) {
		dashboard.updateProductInfo(products);
	}

	@Inject
	@Optional
	public void deletedProduct(@UIEventTopic(ProductRegistryEvents.PRODUCT_DELETE) Product input) {
		dashboard.updateProductInfo(products);
	}

	@Inject
	@Optional
	public void createdProductVersion(
			@UIEventTopic(ProductRegistryEvents.PRODUCT_VERSION_CREATE) ProductVersion input) {
		dashboard.updateProductInfo(products);
	}

	@Inject
	@Optional
	public void deletedProductVersion(
			@UIEventTopic(ProductRegistryEvents.PRODUCT_VERSION_DELETE) ProductVersion input) {
		dashboard.updateProductInfo(products);
	}

	@Inject
	@Optional
	public void createdProductVersionFeature(
			@UIEventTopic(ProductRegistryEvents.PRODUCT_VERSION_FEATURE_CREATE) ProductVersionFeature input) {
		dashboard.updateProductInfo(products);
	}

	@Inject
	@Optional
	public void deletedProductVersionFeature(
			@UIEventTopic(ProductRegistryEvents.PRODUCT_VERSION_FEATURE_DELETE) ProductVersionFeature input) {
		dashboard.updateProductInfo(products);
	}

	@Inject
	@Optional
	public void createdUserOrigin(@UIEventTopic(UserRegistryEvents.USER_ORIGIN_CREATE) UserOrigin input) {
		dashboard.updateUserInfo(users);
	}

	@Inject
	@Optional
	public void deletedUserOrigin(@UIEventTopic(UserRegistryEvents.USER_ORIGIN_DELETE) UserOrigin input) {
		dashboard.updateUserInfo(users);
	}

	@Inject
	@Optional
	public void createdUser(@UIEventTopic(UserRegistryEvents.USER_CREATE) User input) {
		dashboard.updateUserInfo(users);
	}

	@Inject
	@Optional
	public void deletedUser(@UIEventTopic(UserRegistryEvents.USER_DELETE) User input) {
		dashboard.updateUserInfo(users);
	}

	@Inject
	@Optional
	public void createdAgreementsGroup(
			@UIEventTopic(AgreementRegistryEvents.AGREEMENT_GROUP_CREATE) AgreementGroup input) {
		dashboard.updateAgreementInfo(agreements);
	}

	@Inject
	@Optional
	public void deletedAgreementsGroup(
			@UIEventTopic(AgreementRegistryEvents.AGREEMENT_GROUP_DELETE) AgreementGroup input) {
		dashboard.updateAgreementInfo(agreements);
	}

	@Inject
	@Optional
	public void createdAgreement(@UIEventTopic(AgreementRegistryEvents.AGREEMENT_CREATE) Agreement input) {
		dashboard.updateAgreementInfo(agreements);
	}

	@Inject
	@Optional
	public void deletedAgreement(@UIEventTopic(AgreementRegistryEvents.AGREEMENT_DELETE) Agreement input) {
		dashboard.updateAgreementInfo(agreements);
	}

	@Inject
	@Optional
	public void createdLicensePlan(@UIEventTopic(LicenseRegistryEvents.LICENSE_PLAN_CREATE) LicensePlan input) {
		dashboard.updateLicenseInfo(licenses);
	}

	@Inject
	@Optional
	public void deletedLicensePlan(@UIEventTopic(LicenseRegistryEvents.LICENSE_PLAN_DELETE) LicensePlan input) {
		dashboard.updateLicenseInfo(licenses);
	}

	@PreDestroy
	public void preDestroy(IEclipseContext context) {
		dashboard.dispose(context);
	}

	@Focus
	public void onFocus() {

	}

	@Persist
	public void save() {

	}

}