/**
 * Copyright (c) 2016 RCP Vision (http://www.rcp-vision.com) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Lorenzo Bettini - initial API and implementation
 */
package org.eclipse.emf.parsley.dsl.pluginxml;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.List;
import java.util.Map;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.parsley.dsl.pluginxml.PluginXmlUtils;
import org.eclipse.jface.text.Document;
import org.eclipse.pde.core.plugin.IPluginAttribute;
import org.eclipse.pde.core.plugin.IPluginElement;
import org.eclipse.pde.core.plugin.IPluginExtension;
import org.eclipse.pde.core.plugin.IPluginObject;
import org.eclipse.pde.internal.core.text.DocumentElementNode;
import org.eclipse.pde.internal.core.text.IDocumentAttributeNode;
import org.eclipse.pde.internal.core.text.IDocumentElementNode;
import org.eclipse.pde.internal.core.text.plugin.PluginAttribute;
import org.eclipse.pde.internal.core.text.plugin.PluginElementNode;
import org.eclipse.pde.internal.core.text.plugin.PluginExtensionNode;
import org.eclipse.pde.internal.core.text.plugin.PluginModel;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class PluginXmlLoader extends PluginModel {
  private List<PluginExtensionNode> pluginExtensionNodes;
  
  private List<PluginElementNode> pluginExtensionElementNodes;
  
  public PluginXmlLoader(final String source) {
    super(new Document(source), true);
  }
  
  /**
   * The nodes corresponding to &lt;extension&gt; elements in the plugin.xml file
   */
  public List<PluginExtensionNode> getExtensionNodes() {
    if ((this.pluginExtensionNodes == null)) {
      this.initializeExtensionNodes();
    }
    return this.pluginExtensionNodes;
  }
  
  protected List<PluginExtensionNode> initializeExtensionNodes() {
    return this.pluginExtensionNodes = IterableExtensions.<PluginExtensionNode>toList(Iterables.<PluginExtensionNode>filter(((Iterable<?>)Conversions.doWrapArray(this.getPlugin().getExtensions())), PluginExtensionNode.class));
  }
  
  /**
   * The nodes corresponding to elements inside &lt;extension&gt;, for example,
   * &lt;view&gt;, &lt;editor&gt;
   */
  public List<PluginElementNode> getExtensionElements() {
    if ((this.pluginExtensionElementNodes == null)) {
      this.initializeExtensionElements();
    }
    return this.pluginExtensionElementNodes;
  }
  
  protected List<PluginElementNode> initializeExtensionElements() {
    final Function1<PluginExtensionNode, Iterable<PluginElementNode>> _function = new Function1<PluginExtensionNode, Iterable<PluginElementNode>>() {
      @Override
      public Iterable<PluginElementNode> apply(final PluginExtensionNode it) {
        return PluginXmlLoader.this.mapToNodes(it);
      }
    };
    return this.pluginExtensionElementNodes = IterableExtensions.<PluginElementNode>toList(Iterables.<PluginElementNode>concat(ListExtensions.<PluginExtensionNode, Iterable<PluginElementNode>>map(this.getExtensionNodes(), _function)));
  }
  
  private Iterable<PluginElementNode> mapToNodes(final DocumentElementNode it) {
    return Iterables.<PluginElementNode>filter(((Iterable<?>)Conversions.doWrapArray(it.getChildNodes())), PluginElementNode.class);
  }
  
  public List<PluginElementNode> getExtensionChildren(final PluginElementNode node) {
    return IterableExtensions.<PluginElementNode>toList(this.mapToNodes(node));
  }
  
  public Iterable<Map.Entry<String, IDocumentAttributeNode>> getPluginAttributesEntrySet(final PluginElementNode node) {
    return PluginXmlUtils.getPluginAttributesEntrySet(node);
  }
  
  public PluginExtensionNode getExtensionByPoint(final String p) {
    final Function1<PluginExtensionNode, Boolean> _function = new Function1<PluginExtensionNode, Boolean>() {
      @Override
      public Boolean apply(final PluginExtensionNode it) {
        String _point = it.getPoint();
        return Boolean.valueOf(Objects.equal(_point, p));
      }
    };
    return IterableExtensions.<PluginExtensionNode>findFirst(this.getExtensionNodes(), _function);
  }
  
  public String getElementExtension(final PluginElementNode node) {
    IPluginObject _parent = node.getParent();
    return ((PluginExtensionNode) _parent).getPoint();
  }
  
  public PluginElementNode getElementByTagAndId(final String xmlTag, final String id) {
    final Function1<PluginElementNode, Boolean> _function = new Function1<PluginElementNode, Boolean>() {
      @Override
      public Boolean apply(final PluginElementNode it) {
        String _xMLTagName = it.getXMLTagName();
        return Boolean.valueOf(Objects.equal(_xMLTagName, xmlTag));
      }
    };
    final Function1<PluginElementNode, Boolean> _function_1 = new Function1<PluginElementNode, Boolean>() {
      @Override
      public Boolean apply(final PluginElementNode it) {
        boolean _xblockexpression = false;
        {
          final IDocumentAttributeNode v = PluginXmlLoader.this.getId(it);
          boolean _xifexpression = false;
          if ((v != null)) {
            String _attributeAsString = PluginXmlLoader.this.getAttributeAsString(v);
            _xifexpression = Objects.equal(_attributeAsString, id);
          } else {
            _xifexpression = false;
          }
          _xblockexpression = _xifexpression;
        }
        return Boolean.valueOf(_xblockexpression);
      }
    };
    return IterableExtensions.<PluginElementNode>findFirst(IterableExtensions.<PluginElementNode>filter(this.getExtensionElements(), _function), _function_1);
  }
  
  public IDocumentAttributeNode getId(final DocumentElementNode node) {
    return PluginXmlUtils.getId(node);
  }
  
  public String getAttributeAsString(final IDocumentAttributeNode a) {
    return ((PluginAttribute) a).getValue();
  }
  
  /**
   * Copies all the extension and extension elements from the source
   * plugin xml into this plugin xml.  Attributes with the same id
   * will be overwritten in this plugin xml.
   */
  public void copyFromPluginXml(final String source) throws CoreException {
    List<PluginElementNode> _extensionElements = new PluginXmlLoader(source).getExtensionElements();
    for (final PluginElementNode e : _extensionElements) {
      this.copy(e);
    }
  }
  
  /**
   * Assumes that the source has an id.  If a corresponding element
   * in this plugin xml file is not found it will be inserted first.
   */
  public void copy(final PluginElementNode source) throws CoreException {
    final String xmlTagName = source.getXMLTagName();
    PluginElementNode target = this.getElementByTagAndId(xmlTagName, this.getAttributeAsString(this.getId(source)));
    if ((target == null)) {
      target = this.insertExtensionElement(this.getElementExtension(source), xmlTagName);
    }
    this.copy(source, target);
  }
  
  /**
   * This assumes that both the source and the target are not null
   */
  public void copy(final PluginElementNode source, final PluginElementNode target) throws CoreException {
    final IPluginAttribute[] atts = source.getAttributes();
    for (final IPluginAttribute a : atts) {
      {
        final PluginAttribute att = ((PluginAttribute) a);
        final PluginAttribute copy = new PluginAttribute();
        copy.setName(att.getName());
        copy.setValue(att.getValue());
        PluginXmlUtils.getNodeAttributesMap(target).put(att.getName(), copy);
      }
    }
    final IDocumentElementNode[] children = target.getChildNodes();
    IDocumentElementNode[] _childNodes = source.getChildNodes();
    for (final IDocumentElementNode c : _childNodes) {
      {
        final Function1<IDocumentElementNode, Boolean> _function = new Function1<IDocumentElementNode, Boolean>() {
          @Override
          public Boolean apply(final IDocumentElementNode it) {
            String _xMLTagName = it.getXMLTagName();
            String _xMLTagName_1 = c.getXMLTagName();
            return Boolean.valueOf(Objects.equal(_xMLTagName, _xMLTagName_1));
          }
        };
        IDocumentElementNode myChild = IterableExtensions.<IDocumentElementNode>findFirst(((Iterable<IDocumentElementNode>)Conversions.doWrapArray(children)), _function);
        if ((myChild == null)) {
          IPluginElement _createElement = this.getPluginFactory().createElement(target);
          final PluginElementNode newChild = ((PluginElementNode) _createElement);
          newChild.setXMLTagName(c.getXMLTagName());
          target.addChildNode(newChild);
          this.copy(((PluginElementNode) c), newChild);
        } else {
          this.copy(((PluginElementNode) c), ((PluginElementNode) myChild));
        }
      }
    }
  }
  
  public PluginExtensionNode insertExtension(final String point) throws CoreException {
    final IPluginExtension e = this.getPluginFactory().createExtension();
    e.setPoint(point);
    this.getPlugin().add(e);
    this.initializeExtensionNodes();
    return ((PluginExtensionNode) e);
  }
  
  /**
   * If there is no extension element with the specified point, it
   * will be automatically inserted.
   */
  public PluginElementNode insertExtensionElement(final String point, final String xmlTag) throws CoreException {
    PluginExtensionNode ext = this.getExtensionByPoint(point);
    if ((ext == null)) {
      ext = this.insertExtension(point);
    }
    IPluginElement _createElement = this.getPluginFactory().createElement(ext);
    final PluginElementNode element = ((PluginElementNode) _createElement);
    ext.addChildNode(element);
    element.setXMLTagName(xmlTag);
    this.initializeExtensionElements();
    return element;
  }
  
  public String getContentsAsString() {
    StringConcatenation _builder = new StringConcatenation();
    String _replaceFirst = this.getPlugin().toString().replaceFirst("eclipse version=\"3.0", "eclipse version=\"3.4").replaceFirst("(<plugin)\\r?\\n(>)", "<plugin>");
    _builder.append(_replaceFirst);
    _builder.newLineIfNotEmpty();
    return _builder.toString();
  }
}
