/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.dialogs;

import static org.eclipse.emf.databinding.EMFProperties.value;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.createCheckButton;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.createLabelWithText;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.gd;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.gdGrabH;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.gdGrabV;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.gl;
import static org.eclipse.epp.logging.aeri.core.IModelPackage.Literals.SYSTEM_SETTINGS__SEND_MODE;
import static org.eclipse.epp.logging.aeri.core.IModelPackage.Literals.USER_SETTINGS__ANONYMIZE_MESSAGES;
import static org.eclipse.epp.logging.aeri.core.IModelPackage.Literals.USER_SETTINGS__ANONYMIZE_STACK_TRACES;
import static org.eclipse.epp.logging.aeri.core.IModelPackage.Literals.USER_SETTINGS__REPORTER_EMAIL;
import static org.eclipse.epp.logging.aeri.core.IModelPackage.Literals.USER_SETTINGS__REPORTER_NAME;
import static org.eclipse.jface.databinding.swt.WidgetProperties.selection;
import static org.eclipse.jface.databinding.swt.WidgetProperties.text;
import static org.eclipse.jface.dialogs.Dialog.applyDialogFont;

import javax.inject.Inject;

import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.emf.databinding.EMFProperties;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.Messages;
import org.eclipse.epp.internal.logging.aeri.ide.utils.Browsers;
import org.eclipse.epp.internal.logging.aeri.ide.utils.IDEConstants;
import org.eclipse.epp.logging.aeri.core.IModelPackage;
import org.eclipse.epp.logging.aeri.core.ISystemSettings;
import org.eclipse.epp.logging.aeri.core.SendMode;
import org.eclipse.jface.databinding.swt.ISWTObservableValue;
import org.eclipse.jface.databinding.swt.WidgetProperties;
import org.eclipse.jface.databinding.viewers.IViewerObservableValue;
import org.eclipse.jface.databinding.viewers.ViewersObservables;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Text;

public class SystemSettingsPage extends WizardPage {

    private ISystemSettings systemSettings;
    private DataBindingContext context;

    @Inject
    public SystemSettingsPage(ISystemSettings systemSettings) {
        super(SystemSettingsPage.class.getName());
        this.systemSettings = systemSettings;
        context = new DataBindingContext();
        setTitle(Messages.WIZPAGE_TITLE_SYSTEM_SETTINGS);
        setDescription(Messages.WIZPAGE_DESCRIPTION_SYSTEM_SETTINGS);
    }

    @Override
    public void createControl(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);
        Composite personalGroup = createPersonalGroup(container);
        Composite anonymizeDefaults = createAnonymizeGroup(container);
        Composite sendModeDefaults = createSendModeControl(container);
        Composite links = createLinksControl(container);

        gl().margins(5, 5).applyTo(container);
        gd().applyTo(container);
        gdGrabH().indent(0, 10).applyTo(personalGroup);
        gd().indent(0, 10).applyTo(anonymizeDefaults);
        gd().align(SWT.RIGHT, SWT.CENTER).applyTo(sendModeDefaults);
        gdGrabV().align(SWT.BEGINNING, SWT.END).applyTo(links);
        applyDialogFont(container);
        setControl(container);
    }

    private Composite createLinksControl(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);
        gl().numColumns(2).applyTo(container);
        Link learnMore = new Link(container, SWT.NONE);
        learnMore.setText(Messages.LINK_TEXT_LEARN_MORE);
        learnMore.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                Browsers.openInExternalBrowser(IDEConstants.HELP_URL);
            }
        });

        return container;
    }

    // TODO Move to UI class. Similar methods in PreferencePage and ConfigureServerDialog
    private Composite createPersonalGroup(Composite parent) {
        Group personalGroup = new Group(parent, SWT.NONE);
        personalGroup.setText(Messages.GROUP_TEXT_CONTACT_INFORMATION);
        gl().margins(5, 5).numColumns(2).applyTo(personalGroup);
        {
            Text name = createLabelWithText(personalGroup, Messages.FIELD_LABEL_NAME, Messages.FIELD_MESSAGE_NAME,
                    Messages.FIELD_MESSAGE_NAME);
            ISWTObservableValue swt = text(SWT.FocusOut).observe(name);
            IObservableValue emf = value(USER_SETTINGS__REPORTER_NAME).observe(systemSettings);
            context.bindValue(swt, emf);
        }
        {
            String tooltip = Messages.FIELD_MESSAGE_EMAIL + '\n' + Messages.TOOLTIP_SETTINGS_EMAIL;
            Text email = createLabelWithText(personalGroup, Messages.FIELD_LABEL_EMAIL, Messages.FIELD_MESSAGE_EMAIL, tooltip);
            ISWTObservableValue swt = WidgetProperties.text(SWT.Modify).observe(email);
            IObservableValue emf = EMFProperties.value(USER_SETTINGS__REPORTER_EMAIL).observe(systemSettings);
            context.bindValue(swt, emf);
        }
        return personalGroup;
    }

    private Group createAnonymizeGroup(Composite parent) {
        Group container = new Group(parent, SWT.SHADOW_ETCHED_IN | SWT.SHADOW_ETCHED_OUT | SWT.SHADOW_IN | SWT.SHADOW_OUT);
        container.setLayout(new RowLayout(SWT.VERTICAL));
        container.setText(Messages.GROUP_TEXT_ANONYMIZATION);
        {
            Button stackTraces = createCheckButton(container, Messages.FIELD_LABEL_ANONYMIZE_STACKTRACES,
                    Messages.TOOLTIP_SETTINGS_MAKE_STACKTRACE_ANONYMOUS);
            stackTraces.setFocus();
            IObservableValue swt = selection().observe(stackTraces);
            IObservableValue emf = value(USER_SETTINGS__ANONYMIZE_STACK_TRACES).observe(systemSettings);
            context.bindValue(swt, emf);
        }
        {
            Button messages = createCheckButton(container, Messages.FIELD_LABEL_ANONYMIZE_MESSAGES,
                    Messages.TOOLTIP_SETTINGS_MAKE_MESSAGES_ANONYMOUS);
            IObservableValue swt = selection().observe(messages);
            IObservableValue emf = value(USER_SETTINGS__ANONYMIZE_MESSAGES).observe(systemSettings);
            context.bindValue(swt, emf);
        }
        {
            Button disableWiringAnalysis = createCheckButton(container, Messages.FIELD_LABEL_DISABLE_AUTOMATIC_WIRING_ANALYSIS,
                    Messages.TOOLTIP_DISABLE_AUTOMATIC_WIRING_ANALYSIS);
            IObservableValue swt = selection().observe(disableWiringAnalysis);
            IObservableValue emf = value(IModelPackage.Literals.USER_SETTINGS__DISABLE_AUTOMATIC_WIRING_ANALYSIS).observe(systemSettings);
            context.bindValue(swt, emf);
        }
        return container;
    }

    private Composite createSendModeControl(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);

        ComboViewer sendMode = new ComboViewer(container);
        sendMode.getControl().setToolTipText(Messages.TOOLTIP_SEND_MODE);
        sendMode.setContentProvider(ArrayContentProvider.getInstance());
        sendMode.setInput(SendMode.VALUES);
        sendMode.setLabelProvider(new SendModeLabelProvider());

        IViewerObservableValue jface = ViewersObservables.observeSinglePostSelection(sendMode);
        IObservableValue emf = EMFProperties.value(SYSTEM_SETTINGS__SEND_MODE).observe(systemSettings);
        context.bindValue(jface, emf);

        gl().numColumns(2).applyTo(container);
        gdGrabH().align(SWT.END, SWT.CENTER).span(2, 1).applyTo(container);
        return container;
    }

    @Override
    public void dispose() {
        context.dispose();
        super.dispose();
    }
}
