/**
 * Copyright (c) 2016 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.processors;

import java.util.List;

import org.eclipse.epp.internal.logging.aeri.ide.ILogEvent;
import org.eclipse.epp.internal.logging.aeri.ide.IProcessorDescriptor;
import org.eclipse.epp.logging.aeri.core.IProblemState;

public class Processors {

    public static boolean shouldProcess(List<IProcessorDescriptor> descriptors, ILogEvent event) {
        return descriptors.stream().filter(processor -> shouldProcess(processor, event)).findFirst().isPresent();
    }

    /**
     * A processor should process the report of an event if the processor can contribute to the report and the server of the event is
     * interested in the outcome of the processor or the processor itself wants to contribute to the report.
     */
    public static boolean shouldProcess(IProcessorDescriptor descriptor, ILogEvent event) {
        IProblemState interest = event.getInterest();
        if (interest == null || interest.getNeedinfo() == null) {
            return false;
        }
        return canContribute(descriptor, event) && (needinfo(descriptor, interest) || wantsToContribue(descriptor, event));

    }

    private static boolean canContribute(IProcessorDescriptor descriptor, ILogEvent event) {
        return descriptor.getProcessor().canContribute(event.getStatus(), event.getContext());
    }

    private static boolean needinfo(IProcessorDescriptor descriptor, IProblemState interest) {
        return interest.getNeedinfo().contains(descriptor.getDirective());
    }

    private static boolean wantsToContribue(IProcessorDescriptor descriptor, ILogEvent event) {
        return descriptor.getProcessor().wantsToContribute(event.getStatus(), event.getContext());
    }
}
