/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.di;

import static com.google.common.base.Preconditions.checkNotNull;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.e4.core.contexts.ContextFunction;
import org.eclipse.e4.core.contexts.ContextInjectionFactory;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.e4.core.services.events.IEventBroker;
import org.eclipse.epp.internal.logging.aeri.ide.ILogEventsQueue;
import org.eclipse.epp.internal.logging.aeri.ide.filters.ActiveShellStatusFilter;
import org.eclipse.epp.internal.logging.aeri.ide.filters.ActiveWorkbenchStatusFilter;
import org.eclipse.epp.internal.logging.aeri.ide.filters.QueueSizeLimitFilter;
import org.eclipse.epp.logging.aeri.core.SystemControl;
import org.eclipse.epp.logging.aeri.core.filters.EclipseBuildIdFilter;
import org.eclipse.epp.logging.aeri.core.filters.DecoratingDebugFilter;
import org.eclipse.epp.logging.aeri.core.filters.NoErrorStatusFilter;
import org.eclipse.epp.logging.aeri.core.filters.RecentlySeenFilter;
import org.eclipse.epp.logging.aeri.core.filters.SystemEnabledFilter;
import org.eclipse.epp.logging.aeri.core.util.LogListener;
import org.eclipse.ui.PlatformUI;

import com.google.common.base.Predicate;
import com.google.common.base.Predicates;

public class LogListenerCreationFunction extends ContextFunction {

    @SuppressWarnings("null")
    @Override
    public Object compute(IEclipseContext context, String contextKey) {
        contextKey = checkNotNull(contextKey);
        IEventBroker broker = context.get(IEventBroker.class);
        ILogEventsQueue queue = context.get(ILogEventsQueue.class);
        QueueSizeLimitFilter queueSizeLimitFilter = new QueueSizeLimitFilter(queue, 20);
        IEclipseContext systemContext = SystemControl.getSystemContext();
        ContextInjectionFactory.inject(queueSizeLimitFilter, systemContext);
        Predicate<? super IStatus>[] statusPredicates = DecoratingDebugFilter.decorate(
                // @formatter:off
                new EclipseBuildIdFilter(),
                new NoErrorStatusFilter(),
                new ActiveShellStatusFilter(),
                new SystemEnabledFilter(),
                new RecentlySeenFilter(),
                new ActiveWorkbenchStatusFilter(PlatformUI.getWorkbench()),
                queueSizeLimitFilter
                // @formatter:on
        );
        Predicate<IStatus> statusFilters = Predicates.and(statusPredicates);
        LogListener listener = new LogListener(statusFilters, broker);
        systemContext.set(contextKey, listener);

        // register the queue size filter with the system context to get notified about review dialog changes
        systemContext.set(QueueSizeLimitFilter.class, queueSizeLimitFilter);
        return listener;
    }
}
