/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.logging.aeri.core.util;

import javax.inject.Inject;
import javax.inject.Named;

import org.eclipse.core.runtime.ILogListener;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.e4.core.services.events.IEventBroker;
import org.eclipse.epp.logging.aeri.core.Constants;
import org.eclipse.epp.logging.aeri.core.SystemControl;
import org.eclipse.jdt.annotation.Nullable;

import com.google.common.base.Predicate;

public class LogListener implements ILogListener {

    public static final String TOPIC_NEW_STATUS_LOGGED = "org/eclipse/epp/internal/logging/aeri/ui/status/new";
    public static final String CTX_STATUS_FILTER = Constants.BUNDLE_ID + ".statusFilters";

    private Predicate<IStatus> statusFilters;
    private IEventBroker broker;

    @Inject
    public LogListener(@Named(CTX_STATUS_FILTER) Predicate<IStatus> statusFilters, IEventBroker broker) {
        this.statusFilters = statusFilters;
        this.broker = broker;
    }

    @Override
    public void logging(@Nullable IStatus status, @Nullable String plugin) {
        // Javadoc does not say anything about null
        if (status == null) {
            return;
        }
        // stop early if not enabled:
        if (!SystemControl.isActive()) {
            return;
        }

        // fast status filtering
        if (!statusFilters.apply(status)) {
            return;
        }
        // No exception? Then this error is almost useless. Create a copy and fill in the current thread's stack.
        if (status.getException() == null) {
            Throwable missing = new NoStackTrace().fillInStackTrace();
            status = copy(status, missing);
        }
        // but then, go quickly into the background
        notifyNewStatus(status);
    }

    private IStatus copy(IStatus status, Throwable missing) {
        if (status.isMultiStatus()) {
            status = new MultiStatus(status.getPlugin(), status.getCode(), status.getChildren(), status.getMessage(), missing);
        } else {
            status = new Status(status.getSeverity(), status.getPlugin(), status.getMessage(), missing);
        }
        return status;
    }

    protected void notifyNewStatus(IStatus status) {
        broker.post(TOPIC_NEW_STATUS_LOGGED, status);
    }
}
