import { Schema, Type } from 'js-yaml';
import { filterInvalidCustomTags } from './arrUtils';
import { ErrorCode } from 'vscode-json-languageservice/lib/umd/jsonLanguageTypes';
export const DUPLICATE_KEY_REASON = 'duplicate key';
/**
 * Convert a YAML node exception to a
 * special diagnostic type (NOT YET THE
 * LANGUAGE SERVER DIAGNOSTIC).
 */
function exceptionToDiagnostic(e) {
    return {
        message: `${e.reason}`,
        location: {
            start: e.mark.position,
            end: e.mark.position + 1,
            toLineEnd: e.mark.toLineEnd,
        },
        severity: 2,
        code: ErrorCode.Undefined,
    };
}
/**
 * We have to convert the exceptions returned by the AST parser
 * into diagnostics for consumption by the server client.
 */
export function formatErrors(exceptions) {
    return exceptions.filter((e) => e.reason !== DUPLICATE_KEY_REASON && !e.isWarning).map((e) => exceptionToDiagnostic(e));
}
//Patch ontop of yaml-ast-parser to disable duplicate key message on merge key
export function isDuplicateAndNotMergeKey(error, yamlText) {
    const errorStart = error.mark.position;
    const errorEnd = error.mark.position + error.mark.column;
    if (error.reason === DUPLICATE_KEY_REASON && yamlText.substring(errorStart, errorEnd).startsWith('<<')) {
        return false;
    }
    return true;
}
export function formatWarnings(exceptions, text) {
    return exceptions
        .filter((e) => (e.reason === DUPLICATE_KEY_REASON && isDuplicateAndNotMergeKey(e, text)) || e.isWarning)
        .map((e) => exceptionToDiagnostic(e));
}
export function customTagsToAdditionalOptions(customTags) {
    const filteredTags = filterInvalidCustomTags(customTags);
    const schemaWithAdditionalTags = Schema.create(filteredTags.map((tag) => {
        const typeInfo = tag.split(' ');
        return new Type(typeInfo[0], {
            kind: (typeInfo[1] && typeInfo[1].toLowerCase()) || 'scalar',
        });
    }));
    /**
     * Collect the additional tags into a map of string to possible tag types
     */
    const tagWithAdditionalItems = new Map();
    filteredTags.forEach((tag) => {
        const typeInfo = tag.split(' ');
        const tagName = typeInfo[0];
        const tagType = (typeInfo[1] && typeInfo[1].toLowerCase()) || 'scalar';
        if (tagWithAdditionalItems.has(tagName)) {
            tagWithAdditionalItems.set(tagName, tagWithAdditionalItems.get(tagName).concat([tagType]));
        }
        else {
            tagWithAdditionalItems.set(tagName, [tagType]);
        }
    });
    tagWithAdditionalItems.forEach((additionalTagKinds, key) => {
        const newTagType = new Type(key, { kind: additionalTagKinds[0] || 'scalar' });
        newTagType.additionalKinds = additionalTagKinds;
        schemaWithAdditionalTags.compiledTypeMap[key] = newTagType;
    });
    const additionalOptions = {
        schema: schemaWithAdditionalTags,
    };
    return additionalOptions;
}
//# sourceMappingURL=parseUtils.js.map