package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link ProgressMessage}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableProgressMessage.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableProgressMessage implements ProgressMessage {
  private final @Nullable String id;
  private final @Nullable String status;
  private final @Nullable String stream;
  private final @Nullable String error;
  private final @Nullable String progress;
  private final @Nullable ProgressDetail progressDetail;
  private transient final @Nullable String buildImageId;
  private transient final @Nullable String digest;

  private ImmutableProgressMessage(
      @Nullable String id,
      @Nullable String status,
      @Nullable String stream,
      @Nullable String error,
      @Nullable String progress,
      @Nullable ProgressDetail progressDetail) {
    this.id = id;
    this.status = status;
    this.stream = stream;
    this.error = error;
    this.progress = progress;
    this.progressDetail = progressDetail;
    this.buildImageId = initShim.buildImageId();
    this.digest = initShim.digest();
    this.initShim = null;
  }

  private static final byte STAGE_INITIALIZING = -1;
  private static final byte STAGE_UNINITIALIZED = 0;
  private static final byte STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private byte buildImageIdBuildStage = STAGE_UNINITIALIZED;
    private String buildImageId;

    String buildImageId() {
      if (buildImageIdBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (buildImageIdBuildStage == STAGE_UNINITIALIZED) {
        buildImageIdBuildStage = STAGE_INITIALIZING;
        this.buildImageId = buildImageIdInitialize();
        buildImageIdBuildStage = STAGE_INITIALIZED;
      }
      return this.buildImageId;
    }

    private byte digestBuildStage = STAGE_UNINITIALIZED;
    private String digest;

    String digest() {
      if (digestBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (digestBuildStage == STAGE_UNINITIALIZED) {
        digestBuildStage = STAGE_INITIALIZING;
        this.digest = digestInitialize();
        digestBuildStage = STAGE_INITIALIZED;
      }
      return this.digest;
    }

    private String formatInitCycleMessage() {
      List<String> attributes = new ArrayList<>();
      if (buildImageIdBuildStage == STAGE_INITIALIZING) attributes.add("buildImageId");
      if (digestBuildStage == STAGE_INITIALIZING) attributes.add("digest");
      return "Cannot build ProgressMessage, attribute initializers form cycle " + attributes;
    }
  }

  private @Nullable String buildImageIdInitialize() {
    return ProgressMessage.super.buildImageId();
  }

  private @Nullable String digestInitialize() {
    return ProgressMessage.super.digest();
  }

  /**
   * @return The value of the {@code id} attribute
   */
  @JsonProperty("id")
  @Override
  public @Nullable String id() {
    return id;
  }

  /**
   * @return The value of the {@code status} attribute
   */
  @JsonProperty("status")
  @Override
  public @Nullable String status() {
    return status;
  }

  /**
   * @return The value of the {@code stream} attribute
   */
  @JsonProperty("stream")
  @Override
  public @Nullable String stream() {
    return stream;
  }

  /**
   * @return The value of the {@code error} attribute
   */
  @JsonProperty("error")
  @Override
  public @Nullable String error() {
    return error;
  }

  /**
   * @return The value of the {@code progress} attribute
   */
  @JsonProperty("progress")
  @Override
  public @Nullable String progress() {
    return progress;
  }

  /**
   * @return The value of the {@code progressDetail} attribute
   */
  @JsonProperty("progressDetail")
  @Override
  public @Nullable ProgressDetail progressDetail() {
    return progressDetail;
  }

  /**
   * @return The computed-at-construction value of the {@code buildImageId} attribute
   */
  @JsonProperty("buildImageId")
  @JsonIgnore
  @Override
  public @Nullable String buildImageId() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.buildImageId()
        : this.buildImageId;
  }

  /**
   * @return The computed-at-construction value of the {@code digest} attribute
   */
  @JsonProperty("digest")
  @JsonIgnore
  @Override
  public @Nullable String digest() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.digest()
        : this.digest;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProgressMessage#id() id} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for id (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProgressMessage withId(@Nullable String value) {
    if (Objects.equals(this.id, value)) return this;
    return new ImmutableProgressMessage(value, this.status, this.stream, this.error, this.progress, this.progressDetail);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProgressMessage#status() status} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for status (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProgressMessage withStatus(@Nullable String value) {
    if (Objects.equals(this.status, value)) return this;
    return new ImmutableProgressMessage(this.id, value, this.stream, this.error, this.progress, this.progressDetail);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProgressMessage#stream() stream} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for stream (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProgressMessage withStream(@Nullable String value) {
    if (Objects.equals(this.stream, value)) return this;
    return new ImmutableProgressMessage(this.id, this.status, value, this.error, this.progress, this.progressDetail);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProgressMessage#error() error} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for error (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProgressMessage withError(@Nullable String value) {
    if (Objects.equals(this.error, value)) return this;
    return new ImmutableProgressMessage(this.id, this.status, this.stream, value, this.progress, this.progressDetail);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProgressMessage#progress() progress} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for progress (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProgressMessage withProgress(@Nullable String value) {
    if (Objects.equals(this.progress, value)) return this;
    return new ImmutableProgressMessage(this.id, this.status, this.stream, this.error, value, this.progressDetail);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProgressMessage#progressDetail() progressDetail} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for progressDetail (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProgressMessage withProgressDetail(@Nullable ProgressDetail value) {
    if (this.progressDetail == value) return this;
    return new ImmutableProgressMessage(this.id, this.status, this.stream, this.error, this.progress, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableProgressMessage} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableProgressMessage
        && equalTo((ImmutableProgressMessage) another);
  }

  private boolean equalTo(ImmutableProgressMessage another) {
    return Objects.equals(id, another.id)
        && Objects.equals(status, another.status)
        && Objects.equals(stream, another.stream)
        && Objects.equals(error, another.error)
        && Objects.equals(progress, another.progress)
        && Objects.equals(progressDetail, another.progressDetail)
        && Objects.equals(buildImageId, another.buildImageId)
        && Objects.equals(digest, another.digest);
  }

  /**
   * Computes a hash code from attributes: {@code id}, {@code status}, {@code stream}, {@code error}, {@code progress}, {@code progressDetail}, {@code buildImageId}, {@code digest}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(id);
    h += (h << 5) + Objects.hashCode(status);
    h += (h << 5) + Objects.hashCode(stream);
    h += (h << 5) + Objects.hashCode(error);
    h += (h << 5) + Objects.hashCode(progress);
    h += (h << 5) + Objects.hashCode(progressDetail);
    h += (h << 5) + Objects.hashCode(buildImageId);
    h += (h << 5) + Objects.hashCode(digest);
    return h;
  }

  /**
   * Prints the immutable value {@code ProgressMessage} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ProgressMessage{"
        + "id=" + id
        + ", status=" + status
        + ", stream=" + stream
        + ", error=" + error
        + ", progress=" + progress
        + ", progressDetail=" + progressDetail
        + ", buildImageId=" + buildImageId
        + ", digest=" + digest
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ProgressMessage} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ProgressMessage instance
   */
  public static ImmutableProgressMessage copyOf(ProgressMessage instance) {
    if (instance instanceof ImmutableProgressMessage) {
      return (ImmutableProgressMessage) instance;
    }
    return ImmutableProgressMessage.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableProgressMessage ImmutableProgressMessage}.
   * <pre>
   * ImmutableProgressMessage.builder()
   *    .id(String | null) // nullable {@link ProgressMessage#id() id}
   *    .status(String | null) // nullable {@link ProgressMessage#status() status}
   *    .stream(String | null) // nullable {@link ProgressMessage#stream() stream}
   *    .error(String | null) // nullable {@link ProgressMessage#error() error}
   *    .progress(String | null) // nullable {@link ProgressMessage#progress() progress}
   *    .progressDetail(org.mandas.docker.client.messages.ProgressDetail | null) // nullable {@link ProgressMessage#progressDetail() progressDetail}
   *    .build();
   * </pre>
   * @return A new ImmutableProgressMessage builder
   */
  public static ImmutableProgressMessage.Builder builder() {
    return new ImmutableProgressMessage.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableProgressMessage ImmutableProgressMessage}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements ProgressMessage.Builder {
    private String id;
    private String status;
    private String stream;
    private String error;
    private String progress;
    private ProgressDetail progressDetail;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ProgressMessage} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ProgressMessage instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable String idValue = instance.id();
      if (idValue != null) {
        id(idValue);
      }
      @Nullable String statusValue = instance.status();
      if (statusValue != null) {
        status(statusValue);
      }
      @Nullable String streamValue = instance.stream();
      if (streamValue != null) {
        stream(streamValue);
      }
      @Nullable String errorValue = instance.error();
      if (errorValue != null) {
        error(errorValue);
      }
      @Nullable String progressValue = instance.progress();
      if (progressValue != null) {
        progress(progressValue);
      }
      @Nullable ProgressDetail progressDetailValue = instance.progressDetail();
      if (progressDetailValue != null) {
        progressDetail(progressDetailValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ProgressMessage#id() id} attribute.
     * @param id The value for id (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("id")
    public final Builder id(@Nullable String id) {
      this.id = id;
      return this;
    }

    /**
     * Initializes the value for the {@link ProgressMessage#status() status} attribute.
     * @param status The value for status (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("status")
    public final Builder status(@Nullable String status) {
      this.status = status;
      return this;
    }

    /**
     * Initializes the value for the {@link ProgressMessage#stream() stream} attribute.
     * @param stream The value for stream (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("stream")
    public final Builder stream(@Nullable String stream) {
      this.stream = stream;
      return this;
    }

    /**
     * Initializes the value for the {@link ProgressMessage#error() error} attribute.
     * @param error The value for error (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("error")
    public final Builder error(@Nullable String error) {
      this.error = error;
      return this;
    }

    /**
     * Initializes the value for the {@link ProgressMessage#progress() progress} attribute.
     * @param progress The value for progress (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("progress")
    public final Builder progress(@Nullable String progress) {
      this.progress = progress;
      return this;
    }

    /**
     * Initializes the value for the {@link ProgressMessage#progressDetail() progressDetail} attribute.
     * @param progressDetail The value for progressDetail (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("progressDetail")
    public final Builder progressDetail(@Nullable ProgressDetail progressDetail) {
      this.progressDetail = progressDetail;
      return this;
    }

    /**
     * Builds a new {@link ImmutableProgressMessage ImmutableProgressMessage}.
     * @return An immutable instance of ProgressMessage
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableProgressMessage build() {
      return new ImmutableProgressMessage(id, status, stream, error, progress, progressDetail);
    }
  }
}
