/**
 * Copyright (c) 2016, 2017 itemis AG and others.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Alexander Nyßen (itemis AG) - initial API and implementation
 *     Tamas Miklossy  (itemis AG) - minor refactorings
 */
package org.eclipse.gef.dot.internal;

import java.io.File;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import javafx.collections.ObservableList;
import org.eclipse.gef.common.attributes.IAttributeStore;
import org.eclipse.gef.dot.internal.language.dot.GraphType;
import org.eclipse.gef.dot.internal.language.terminals.ID;
import org.eclipse.gef.graph.Edge;
import org.eclipse.gef.graph.Graph;
import org.eclipse.gef.graph.Node;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * A serializer that creates a Graphviz DOT string or file from a {@link Graph} with {@link DotAttributes}.
 * 
 * @author anyssen
 */
@SuppressWarnings("all")
public class DotExport {
  public String exportDot(final Graph graph) {
    return this.exportDot(CollectionLiterals.<Graph>newArrayList(graph));
  }

  public String exportDot(final List<Graph> graphs) {
    String _xblockexpression = null;
    {
      final StringBuilder builder = new StringBuilder();
      for (final Graph graph : graphs) {
        {
          GraphType _type = this.type(graph);
          boolean _tripleEquals = (_type == null);
          if (_tripleEquals) {
            throw new IllegalArgumentException(
              (((("The " + DotAttributes._TYPE__G) + " attribute has to be set on the input graph ") + graph) + "."));
          }
          final Function1<Node, Boolean> _function = (Node it) -> {
            Graph _nestedGraph = it.getNestedGraph();
            return Boolean.valueOf((_nestedGraph == null));
          };
          final Function1<Node, Boolean> _function_1 = (Node it) -> {
            boolean _hasName = this.hasName(it);
            return Boolean.valueOf((!_hasName));
          };
          boolean _exists = IterableExtensions.<Node>exists(IterableExtensions.<Node>filter(graph.getNodes(), _function), _function_1);
          if (_exists) {
            throw new IllegalArgumentException(
              (((("The " + DotAttributes._NAME__GNE) + " attribute has to be set for all nodes of the input graph ") + graph) + "."));
          }
          builder.append(this.print(graph));
        }
      }
      _xblockexpression = builder.toString();
    }
    return _xblockexpression;
  }

  public File exportDot(final Graph graph, final String pathname) {
    String _exportDot = this.exportDot(graph);
    File _file = new File(pathname);
    return DotFileUtils.write(_exportDot, _file);
  }

  public File exportDot(final List<Graph> graphs, final String pathname) {
    String _exportDot = this.exportDot(graphs);
    File _file = new File(pathname);
    return DotFileUtils.write(_exportDot, _file);
  }

  private String print(final Graph it) {
    StringConcatenation _builder = new StringConcatenation();
    GraphType _type = this.type(it);
    _builder.append(_type);
    _builder.append(" ");
    {
      boolean _hasName = this.hasName(it);
      if (_hasName) {
        String _name = this.name(it);
        _builder.append(_name);
        _builder.append(" ");
      }
    }
    _builder.append("{");
    _builder.newLineIfNotEmpty();
    {
      boolean _hasNonMetaAttributes = this.hasNonMetaAttributes(it);
      if (_hasNonMetaAttributes) {
        _builder.append("\t");
        String _printNonMetaAttributes = this.printNonMetaAttributes(it, ";");
        _builder.append(_printNonMetaAttributes, "\t");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("\t");
    final Function1<Node, String> _function = (Node it_1) -> {
      return this.print(it_1);
    };
    String _join = IterableExtensions.join(ListExtensions.<Node, String>map(it.getNodes(), _function), "; ");
    _builder.append(_join, "\t");
    _builder.newLineIfNotEmpty();
    {
      ObservableList<Edge> _edges = it.getEdges();
      for(final Edge edge : _edges) {
        _builder.append("\t");
        String _name_1 = this.name(edge);
        _builder.append(_name_1, "\t");
        {
          boolean _hasNonMetaAttributes_1 = this.hasNonMetaAttributes(edge);
          if (_hasNonMetaAttributes_1) {
            _builder.append(" [");
            String _printNonMetaAttributes_1 = this.printNonMetaAttributes(edge, ",");
            _builder.append(_printNonMetaAttributes_1, "\t");
            _builder.append("]");
          }
        }
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("}");
    _builder.newLine();
    return _builder.toString();
  }

  private boolean isMetaAttribute(final String it) {
    return it.startsWith("_");
  }

  private String print(final Node it) {
    String _xifexpression = null;
    Graph _nestedGraph = it.getNestedGraph();
    boolean _tripleNotEquals = (_nestedGraph != null);
    if (_tripleNotEquals) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("subgraph ");
      {
        boolean _hasName = this.hasName(it.getNestedGraph());
        if (_hasName) {
          String _name = this.name(it.getNestedGraph());
          _builder.append(_name);
          _builder.append(" ");
        }
      }
      _builder.append("{");
      _builder.newLineIfNotEmpty();
      {
        boolean _hasNonMetaAttributes = this.hasNonMetaAttributes(it.getNestedGraph());
        if (_hasNonMetaAttributes) {
          _builder.append("\t");
          String _printNonMetaAttributes = this.printNonMetaAttributes(it.getNestedGraph(), ";");
          _builder.append(_printNonMetaAttributes, "\t");
          _builder.newLineIfNotEmpty();
        }
      }
      _builder.append("\t");
      final Function1<Node, String> _function = (Node it_1) -> {
        return this.print(it_1);
      };
      String _join = IterableExtensions.join(ListExtensions.<Node, String>map(it.getNestedGraph().getNodes(), _function), "; ");
      _builder.append(_join, "\t");
      _builder.newLineIfNotEmpty();
      {
        ObservableList<Edge> _edges = it.getNestedGraph().getEdges();
        for(final Edge edge : _edges) {
          _builder.append("\t");
          String _name_1 = this.name(edge);
          _builder.append(_name_1, "\t");
          {
            boolean _hasNonMetaAttributes_1 = this.hasNonMetaAttributes(edge);
            if (_hasNonMetaAttributes_1) {
              _builder.append(" [");
              String _printNonMetaAttributes_1 = this.printNonMetaAttributes(edge, ",");
              _builder.append(_printNonMetaAttributes_1, "\t");
              _builder.append("]");
            }
          }
          _builder.newLineIfNotEmpty();
        }
      }
      _builder.append("}");
      _xifexpression = _builder.toString();
    } else {
      String _name_2 = this.name(it);
      String _xifexpression_1 = null;
      boolean _hasNonMetaAttributes_2 = this.hasNonMetaAttributes(it);
      if (_hasNonMetaAttributes_2) {
        String _printNonMetaAttributes_2 = this.printNonMetaAttributes(it, ",");
        String _plus = (" [" + _printNonMetaAttributes_2);
        _xifexpression_1 = (_plus + "]");
      } else {
        _xifexpression_1 = "";
      }
      _xifexpression = (_name_2 + _xifexpression_1);
    }
    return _xifexpression;
  }

  private boolean hasName(final IAttributeStore it) {
    Object _get = it.getAttributes().get(DotAttributes._NAME__GNE);
    return (_get != null);
  }

  private GraphType type(final Graph it) {
    return DotAttributes._getType(it);
  }

  private String _name(final IAttributeStore it) {
    Object _get = it.getAttributes().get(DotAttributes._NAME__GNE);
    return ((ID) _get).toValue();
  }

  private String _name(final Edge it) {
    return DotAttributes._getName(it);
  }

  private boolean hasNonMetaAttributes(final IAttributeStore it) {
    final Function1<String, Boolean> _function = (String it_1) -> {
      boolean _isMetaAttribute = this.isMetaAttribute(it_1);
      return Boolean.valueOf((!_isMetaAttribute));
    };
    return IterableExtensions.<String>exists(it.getAttributes().keySet(), _function);
  }

  private String printNonMetaAttributes(final IAttributeStore it, final String separator) {
    final Function1<Map.Entry<String, Object>, Boolean> _function = (Map.Entry<String, Object> it_1) -> {
      boolean _isMetaAttribute = this.isMetaAttribute(it_1.getKey());
      return Boolean.valueOf((!_isMetaAttribute));
    };
    final Function1<Map.Entry<String, Object>, String> _function_1 = (Map.Entry<String, Object> it_1) -> {
      String _key = it_1.getKey();
      String _plus = (_key + "=");
      String _string = it_1.getValue().toString();
      return (_plus + _string);
    };
    return IterableExtensions.join(IterableExtensions.<String>sort(IterableExtensions.<Map.Entry<String, Object>, String>map(IterableExtensions.<Map.Entry<String, Object>>filter(it.getAttributes().entrySet(), _function), _function_1)), (separator + " "));
  }

  private String name(final IAttributeStore it) {
    if (it instanceof Edge) {
      return _name((Edge)it);
    } else if (it != null) {
      return _name(it);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(it).toString());
    }
  }
}
