/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.tests.model.listener;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import junit.framework.TestCase;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.collection.CollectionTools;
import org.eclipse.persistence.tools.utility.iterator.CloneIterator;
import org.eclipse.persistence.tools.utility.model.AbstractModel;
import org.eclipse.persistence.tools.utility.model.event.CollectionAddEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionClearEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionRemoveEvent;
import org.eclipse.persistence.tools.utility.model.listener.CollectionChangeListener;
import org.eclipse.persistence.tools.utility.model.listener.ListChangeListener;
import org.eclipse.persistence.tools.utility.model.listener.ReflectiveChangeListener;

@SuppressWarnings("nls")
public class ReflectiveCollectionChangeListenerTests extends TestCase {

	public ReflectiveCollectionChangeListenerTests(String name) {
		super(name);
	}

	private CollectionChangeListener buildZeroArgumentListener(Object target) {
		return ReflectiveChangeListener.buildCollectionChangeListener(target, "itemAddedZeroArgument", "itemRemovedZeroArgument", "collectionClearedZeroArgument", "collectionChangedZeroArgument");
	}

	private CollectionChangeListener buildSingleArgumentListener(Object target) {
		return ReflectiveChangeListener.buildCollectionChangeListener(target, "itemAddedSingleArgument", "itemRemovedSingleArgument", "collectionClearedSingleArgument", "collectionChangedSingleArgument");
	}

	public void testItemAddedZeroArgumentNamedCollection() {
		TestModel testModel = new TestModel();
		String string = "foo";
		Target target = new Target(testModel, TestModel.STRINGS_COLLECTION, string);
		testModel.addCollectionChangeListener(TestModel.STRINGS_COLLECTION, this.buildZeroArgumentListener(target));
		testModel.addString(string);
		assertTrue(target.itemAddedZeroArgumentFlag);
		assertFalse(target.itemAddedSingleArgumentFlag);
		assertFalse(target.itemRemovedZeroArgumentFlag);
		assertFalse(target.itemRemovedSingleArgumentFlag);
		assertFalse(target.collectionClearedZeroArgumentFlag);
		assertFalse(target.collectionClearedSingleArgumentFlag);
		assertFalse(target.collectionChangedZeroArgumentFlag);
		assertFalse(target.collectionChangedSingleArgumentFlag);
	}

	public void testItemAddedSingleArgumentNamedCollection() {
		TestModel testModel = new TestModel();
		String string = "foo";
		Target target = new Target(testModel, TestModel.STRINGS_COLLECTION, string);
		testModel.addCollectionChangeListener(TestModel.STRINGS_COLLECTION, this.buildSingleArgumentListener(target));
		testModel.addString(string);
		assertFalse(target.itemAddedZeroArgumentFlag);
		assertTrue(target.itemAddedSingleArgumentFlag);
		assertFalse(target.itemRemovedZeroArgumentFlag);
		assertFalse(target.itemRemovedSingleArgumentFlag);
		assertFalse(target.collectionClearedZeroArgumentFlag);
		assertFalse(target.collectionClearedSingleArgumentFlag);
		assertFalse(target.collectionChangedZeroArgumentFlag);
		assertFalse(target.collectionChangedSingleArgumentFlag);
	}

	public void testItemRemovedZeroArgumentNamedCollection() {
		TestModel testModel = new TestModel();
		String string = "foo";
		testModel.addString(string);
		Target target = new Target(testModel, TestModel.STRINGS_COLLECTION, string);
		testModel.addCollectionChangeListener(TestModel.STRINGS_COLLECTION, this.buildZeroArgumentListener(target));
		testModel.removeString(string);
		assertFalse(target.itemAddedZeroArgumentFlag);
		assertFalse(target.itemAddedSingleArgumentFlag);
		assertTrue(target.itemRemovedZeroArgumentFlag);
		assertFalse(target.itemRemovedSingleArgumentFlag);
		assertFalse(target.collectionClearedZeroArgumentFlag);
		assertFalse(target.collectionClearedSingleArgumentFlag);
		assertFalse(target.collectionChangedZeroArgumentFlag);
		assertFalse(target.collectionChangedSingleArgumentFlag);
	}

	public void testItemRemovedSingleArgumentNamedCollection() {
		TestModel testModel = new TestModel();
		String string = "foo";
		testModel.addString(string);
		Target target = new Target(testModel, TestModel.STRINGS_COLLECTION, string);
		testModel.addCollectionChangeListener(TestModel.STRINGS_COLLECTION, this.buildSingleArgumentListener(target));
		testModel.removeString(string);
		assertFalse(target.itemAddedZeroArgumentFlag);
		assertFalse(target.itemAddedSingleArgumentFlag);
		assertFalse(target.itemRemovedZeroArgumentFlag);
		assertTrue(target.itemRemovedSingleArgumentFlag);
		assertFalse(target.collectionClearedZeroArgumentFlag);
		assertFalse(target.collectionClearedSingleArgumentFlag);
		assertFalse(target.collectionChangedZeroArgumentFlag);
		assertFalse(target.collectionChangedSingleArgumentFlag);
	}

	public void testCollectionClearedZeroArgumentNamedCollection() {
		TestModel testModel = new TestModel();
		String string = "foo";
		testModel.addString(string);
		Target target = new Target(testModel, TestModel.STRINGS_COLLECTION, string);
		testModel.addCollectionChangeListener(TestModel.STRINGS_COLLECTION, this.buildZeroArgumentListener(target));
		testModel.clearStrings();
		assertFalse(target.itemAddedZeroArgumentFlag);
		assertFalse(target.itemAddedSingleArgumentFlag);
		assertFalse(target.itemRemovedZeroArgumentFlag);
		assertFalse(target.itemRemovedSingleArgumentFlag);
		assertTrue(target.collectionClearedZeroArgumentFlag);
		assertFalse(target.collectionClearedSingleArgumentFlag);
		assertFalse(target.collectionChangedZeroArgumentFlag);
		assertFalse(target.collectionChangedSingleArgumentFlag);
	}

	public void testCollectionClearedSingleArgumentNamedCollection() {
		TestModel testModel = new TestModel();
		String string = "foo";
		testModel.addString(string);
		Target target = new Target(testModel, TestModel.STRINGS_COLLECTION, string);
		testModel.addCollectionChangeListener(TestModel.STRINGS_COLLECTION, this.buildSingleArgumentListener(target));
		testModel.clearStrings();
		assertFalse(target.itemAddedZeroArgumentFlag);
		assertFalse(target.itemAddedSingleArgumentFlag);
		assertFalse(target.itemRemovedZeroArgumentFlag);
		assertFalse(target.itemRemovedSingleArgumentFlag);
		assertFalse(target.collectionClearedZeroArgumentFlag);
		assertTrue(target.collectionClearedSingleArgumentFlag);
		assertFalse(target.collectionChangedZeroArgumentFlag);
		assertFalse(target.collectionChangedSingleArgumentFlag);
	}

	public void testCollectionChangedZeroArgumentNamedCollection() {
		TestModel testModel = new TestModel();
		String string = "foo";
		testModel.addString(string);
		Target target = new Target(testModel, TestModel.STRINGS_COLLECTION, string);
		testModel.addCollectionChangeListener(TestModel.STRINGS_COLLECTION, this.buildZeroArgumentListener(target));
		testModel.replaceStrings(new String[] {"bar", "baz"});
		assertFalse(target.itemAddedZeroArgumentFlag);
		assertFalse(target.itemAddedSingleArgumentFlag);
		assertFalse(target.itemRemovedZeroArgumentFlag);
		assertFalse(target.itemRemovedSingleArgumentFlag);
		assertFalse(target.collectionClearedZeroArgumentFlag);
		assertFalse(target.collectionClearedSingleArgumentFlag);
		assertTrue(target.collectionChangedZeroArgumentFlag);
		assertFalse(target.collectionChangedSingleArgumentFlag);
	}

	public void testCollectionChangedSingleArgumentNamedCollection() {
		TestModel testModel = new TestModel();
		String string = "foo";
		testModel.addString(string);
		Target target = new Target(testModel, TestModel.STRINGS_COLLECTION, string);
		testModel.addCollectionChangeListener(TestModel.STRINGS_COLLECTION, this.buildSingleArgumentListener(target));
		testModel.replaceStrings(new String[] {"bar", "baz"});
		assertFalse(target.itemAddedZeroArgumentFlag);
		assertFalse(target.itemAddedSingleArgumentFlag);
		assertFalse(target.itemRemovedZeroArgumentFlag);
		assertFalse(target.itemRemovedSingleArgumentFlag);
		assertFalse(target.collectionClearedZeroArgumentFlag);
		assertFalse(target.collectionClearedSingleArgumentFlag);
		assertFalse(target.collectionChangedZeroArgumentFlag);
		assertTrue(target.collectionChangedSingleArgumentFlag);
	}

	public void testBogusDoubleArgument1() {
		TestModel testModel = new TestModel();
		String string = "foo";
		Target target = new Target(testModel, TestModel.STRINGS_COLLECTION, string);
		boolean exCaught = false;
		try {
			CollectionChangeListener listener = ReflectiveChangeListener.buildCollectionChangeListener(target, "collectionChangedDoubleArgument");
			fail("bogus listener: " + listener);
		} catch (RuntimeException ex) {
			if (ex.getCause().getClass() == NoSuchMethodException.class) {
				exCaught = true;
			}
		}
		assertTrue(exCaught);
	}

	public void testBogusDoubleArgument2() throws Exception {
		TestModel testModel = new TestModel();
		String string = "foo";
		Target target = new Target(testModel, TestModel.STRINGS_COLLECTION, string);
		Method method = ObjectTools.method(target, "collectionChangedDoubleArgument", new Class[] {CollectionChangeEvent.class, Object.class});
		boolean exCaught = false;
		try {
			CollectionChangeListener listener = ReflectiveChangeListener.buildCollectionChangeListener(target, method);
			fail("bogus listener: " + listener);
		} catch (RuntimeException ex) {
			if (ex.getMessage().equals(method.toString())) {
				exCaught = true;
			}
		}
		assertTrue(exCaught);
	}

	public void testListenerMismatch() {
		TestModel testModel = new TestModel();
		String string = "foo";
		Target target = new Target(testModel, TestModel.STRINGS_COLLECTION, string);
		// build a COLLECTION change listener and hack it so we
		// can add it as a LIST change listener
		Object listener = ReflectiveChangeListener.buildCollectionChangeListener(target, "collectionEventSingleArgument");
		testModel.addListChangeListener("bogus list", (ListChangeListener) listener);

		boolean exCaught = false;
		try {
			testModel.changeList();
			fail("listener mismatch: " + listener);
		} catch (IllegalArgumentException ex) {
			exCaught = true;
		}
		assertTrue(exCaught);
	}


	class TestModel extends AbstractModel {
		private Collection<String> strings = new ArrayList<String>();
			public static final String STRINGS_COLLECTION = "strings";
		TestModel() {
			super();
		}
		Iterator<String> strings() {
			return new CloneIterator<String>(this.strings, new CloneIterator.Remover<String>() {
				@Override
				public void remove(String s) {
					TestModel.this.removeString(s);
				}
			});
		}
		void addString(String string) {
			this.addItemToCollection(string, this.strings, STRINGS_COLLECTION);
		}
		void removeString(String string) {
			this.removeItemFromCollection(string, this.strings, STRINGS_COLLECTION);
		}
		void clearStrings() {
			this.clearCollection(this.strings, STRINGS_COLLECTION);
		}
		void replaceStrings(String[] newStrings) {
			this.strings.clear();
			CollectionTools.addAll(this.strings, newStrings);
			this.fireCollectionChanged(STRINGS_COLLECTION, this.strings);
		}
		void changeList() {
			this.fireListChanged("bogus list", Collections.emptyList());
		}
	}

	class Target {
		TestModel testModel;
		String collectionName;
		String string;
		boolean itemAddedZeroArgumentFlag = false;
		boolean itemAddedSingleArgumentFlag = false;
		boolean itemRemovedZeroArgumentFlag = false;
		boolean itemRemovedSingleArgumentFlag = false;
		boolean collectionClearedZeroArgumentFlag = false;
		boolean collectionClearedSingleArgumentFlag = false;
		boolean collectionChangedZeroArgumentFlag = false;
		boolean collectionChangedSingleArgumentFlag = false;
		boolean collectionEventSingleArgumentFlag = false;
		Target(TestModel testModel, String collectionName, String string) {
			super();
			this.testModel = testModel;
			this.collectionName = collectionName;
			this.string = string;
		}
		void itemAddedZeroArgument() {
			this.itemAddedZeroArgumentFlag = true;
		}
		void itemAddedSingleArgument(CollectionAddEvent e) {
			this.itemAddedSingleArgumentFlag = true;
			assertSame(this.testModel, e.getSource());
			assertEquals(this.collectionName, e.getCollectionName());
			assertEquals(this.string, e.getItems().iterator().next());
		}
		void itemRemovedZeroArgument() {
			this.itemRemovedZeroArgumentFlag = true;
		}
		void itemRemovedSingleArgument(CollectionRemoveEvent e) {
			this.itemRemovedSingleArgumentFlag = true;
			assertSame(this.testModel, e.getSource());
			assertEquals(this.collectionName, e.getCollectionName());
			assertEquals(this.string, e.getItems().iterator().next());
		}
		void collectionClearedZeroArgument() {
			this.collectionClearedZeroArgumentFlag = true;
		}
		void collectionClearedSingleArgument(CollectionClearEvent e) {
			this.collectionClearedSingleArgumentFlag = true;
			assertSame(this.testModel, e.getSource());
			assertEquals(this.collectionName, e.getCollectionName());
		}
		void collectionChangedZeroArgument() {
			this.collectionChangedZeroArgumentFlag = true;
		}
		void collectionChangedSingleArgument(CollectionChangeEvent e) {
			this.collectionChangedSingleArgumentFlag = true;
			assertSame(this.testModel, e.getSource());
			assertEquals(this.collectionName, e.getCollectionName());
		}
		void collectionEventSingleArgument(CollectionEvent e) {
			this.collectionEventSingleArgumentFlag = true;
			assertSame(this.testModel, e.getSource());
			assertEquals(this.collectionName, e.getCollectionName());
		}
		void collectionChangedDoubleArgument(CollectionChangeEvent e, Object o) {
			fail("bogus event: " + e + " object: " + o);
		}
	}
}