/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.InheritanceType;
import org.eclipse.persistence.tools.mapping.orm.ExternalAssociationOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalAttributeOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicNamedQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalDiscriminatorColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntity;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntitySecondaryTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalNamedQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalNativeQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalPrimaryKeyJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalSQLResultSetMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalStoredProcedureQuery;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of an entity, which is a child of an ORM configuration.
 *
 * @see ORMConfiguration
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class Entity extends MappedSuperClassEntity
                             implements ExternalEntity {

	/**
	 * The element name used to store and retrieve the class-extractor child node.
	 */
	static final String CLASS_EXTRACTOR = "class-extractor";

	/**
	 * The element name used to store and retrieve the column child text node.
	 */
	static final String DISCRIMINATOR_VALUE = "discriminator-value";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String ENTITY = "entity";

	/**
	 * The element name used to store and retrieve the inheritance child node.
	 */
	static final String INHERITANCE = "inheritance";

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	static final String NAME = "name";

	/**
	 * The attribute name used to store and retrieve the strategy property of the inheritance child node.
	 */
	static final String STRATEGY = "strategy";

	/**
	 * Creates a new <code>Entity</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	Entity(ORMConfiguration parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addAssociationOverride(int index, String name) {
		AssociationOverride associationOverride = buildAssociationOverride(index);
		associationOverride.addSelf();
		associationOverride.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addAttributeOverride(int index, String name) {
		AttributeOverride attributeOverride = buildAttributeOverride(index);
		attributeOverride.addSelf();
		attributeOverride.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalDiscriminatorColumn addDiscriminatorColumn(String columnName) {

		DiscriminatorColumn column = getDiscriminatorColumn();

		if (column == null) {
			column = buildDiscriminatorColumn();
			column.addSelf();
		}

		column.setName(columnName);
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addNamedQuery(int index, String name) {
		NamedQuery namedQuery = buildNamedQuery(index);
		namedQuery.addSelf();
		namedQuery.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addNativeQuery(int index, String name) {
		NamedNativeQuery namedNativeQuery = buildNamedNativeQuery(index);
		namedNativeQuery.addSelf();
		namedNativeQuery.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addPrimaryKeyJoinColumn(int index, String name) {
		PrimaryKeyJoinColumn pkJoinColumn = buildPrimaryKeyJoinColumn(index);
		pkJoinColumn.addSelf();
		pkJoinColumn.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityTable addPrimaryTable(String tableName) {
		PrimaryTable primaryTable = buildPrimaryTable();
		primaryTable.addSelf();
		primaryTable.setName(tableName);
		return primaryTable;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addSecondaryTable(int index, String name, String catalog, String schema) {
		SecondaryTable secondaryTable = buildSecondaryTable(index);
		secondaryTable.addSelf();
		secondaryTable.setName(name);
		secondaryTable.setCatalogName(catalog);
		secondaryTable.setSchemaName(schema);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addSqlResultSetMapping(int index, String name) {
		SQLResultSetMapping sqlResultSetMapping = buildSqlResultSetMapping(index);
		sqlResultSetMapping.addSelf();
		sqlResultSetMapping.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addStoredProcedureQuery(int index, String name, String procedureName) {
		NamedStoredProcedureQuery storedProcedureQuery = buildStoredProcedureQuery(index);
		storedProcedureQuery.addSelf();
		storedProcedureQuery.setName(name);
		storedProcedureQuery.setProcedureName(procedureName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalAssociationOverride> associationOverrides() {

		int count = associationOverridesSize();
		List<ExternalAssociationOverride> associationOverrides = new ArrayList<ExternalAssociationOverride>(count);

		for (int index = count; --index >= 0;) {
			ExternalAssociationOverride associationOverride = buildAssociationOverride(index);
			associationOverrides.add(0, associationOverride);
		}

		return new ListListIterable<ExternalAssociationOverride>(associationOverrides);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int associationOverridesSize() {
		return getChildrenSize(AttributeOverride.ATTRIBUTE_OVERRIDE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalAttributeOverride> attributeOverrides() {

		int count = attributeOverridesSize();
		List<ExternalAttributeOverride> attributeOverrides = new ArrayList<ExternalAttributeOverride>(count);

		for (int index = count; --index >= 0;) {
			ExternalAttributeOverride attributeOverride = buildAttributeOverride(index);
			attributeOverrides.add(0, attributeOverride);
		}

		return new ListListIterable<ExternalAttributeOverride>(attributeOverrides);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int attributeOverridesSize() {
		return getChildrenSize(AssociationOverride.ASSOCIATION_OVERRIDE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(CLASS);
		names.add(PARENT_CLASS);
		names.add(ACCESS);
		names.add(CACHEABLE);
		names.add(METADATA_COMPLETE);
		names.add(READ_ONLY);
		names.add(EXISTENCE_CHECKING);
		names.add(EXCLUDE_DEFAULT_MAPPINGS);
		return names;
	}

	private AssociationOverride buildAssociationOverride(int index) {
		return new AssociationOverride(this, index);
	}

	private AttributeOverride buildAttributeOverride(int index) {
		return new AttributeOverride(this, index);
	}

	private DiscriminatorColumn buildDiscriminatorColumn() {
		return new DiscriminatorColumn(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		names.add(CUSTOMIZER);
		names.add(CHANGE_TRACKING);
		names.add(PrimaryTable.TABLE);
		names.add(SecondaryTable.SECONDARY_TABLE);
		names.add(PrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN);
		names.add(NoSql.NO_SQL);
		names.add(ID_CLASS);
		names.add(PrimaryKey.PRIMARY_KEY);
		names.add(INHERITANCE);
		names.add(DISCRIMINATOR_VALUE);
		names.add(DiscriminatorColumn.DISCRIMINATOR_COLUMN);
		names.add(CLASS_EXTRACTOR);
		names.add(OptimisticLocking.OPTIMISTIC_LOCKING);
		names.add(Cache.CACHE);
		names.add(Converter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(CopyPolicy.COPY_POLICY);
		names.add(InstantiationCopyPolicy.INSTANTIATION_COPY_POLICY);
		names.add(CloneCopyPolicy.CLONE_COPY_POLICY);
		names.add(SequenceGenerator.SEQUENCE_GENERATOR);
		names.add(TableGenerator.TABLE_GENERATOR);
		names.add(NamedQuery.NAMED_QUERY);
		names.add(NamedNativeQuery.NAMED_NATIVE_QUERY);
		names.add(NamedStoredProcedureQuery.NAMED_STORED_PROCEDURE_QUERY);
		names.add(SQLResultSetMapping.SQL_RESULT_SET_MAPPING);
		names.add(EXCLUDE_DEFAULT_LISTENERS);
		names.add(EXCLUDE_SUPERCLASS_LISTENERS);
		names.add(EntityListener.ENTITY_LISTENERS);
		names.add(PRE_PERSIST);
		names.add(POST_PERSIST);
		names.add(PRE_REMOVE);
		names.add(POST_REMOVE);
		names.add(PRE_UPDATE);
		names.add(POST_UPDATE);
		names.add(POST_LOAD);
		names.add(Property.PROPERTY);
		names.add(AttributeOverride.ATTRIBUTE_OVERRIDE);
		names.add(AssociationOverride.ASSOCIATION_OVERRIDE);
		names.add(Mapping.ATTRIBUTES);
		return names;
	}

	private NamedNativeQuery buildNamedNativeQuery(int index) {
		return new NamedNativeQuery(this, index);
	}

	private NamedQuery buildNamedQuery(int index) {
		return new NamedQuery(this, index);
	}

	private PrimaryKeyJoinColumn buildPrimaryKeyJoinColumn(int index) {
		return new PrimaryKeyJoinColumn(this, index);
	}

	private PrimaryTable buildPrimaryTable() {
		return new PrimaryTable(this);
	}

	private SecondaryTable buildSecondaryTable(int index) {
		return new SecondaryTable(this, index);
	}

	private SQLResultSetMapping buildSqlResultSetMapping(int index) {
		return new SQLResultSetMapping(this, index);
	}

	private NamedStoredProcedureQuery buildStoredProcedureQuery(int index) {
		return new NamedStoredProcedureQuery(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride getAssociationOverride(int index) {

		Element element = getChild(AssociationOverride.ASSOCIATION_OVERRIDE, index);

		if (element == null) {
			return null;
		}

		return buildAssociationOverride(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride getAttributeOverride(int index) {

		Element element = getChild(AttributeOverride.ATTRIBUTE_OVERRIDE, index);

		if (element == null) {
			return null;
		}

		return buildAttributeOverride(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getClassExtractorClassName() {
		return getChildAttribute(CLASS_EXTRACTOR, CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public DiscriminatorColumn getDiscriminatorColumn() {

		Element element = getChild(DiscriminatorColumn.DISCRIMINATOR_COLUMN);

		if (element == null) {
			return null;
		}

		return buildDiscriminatorColumn();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDiscriminatorValue() {
		return getChildTextNode(DISCRIMINATOR_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ENTITY;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public InheritanceType getInheritenceStrategy() {

		Element element = getChild(INHERITANCE);

		if (element == null) {
			return null;
		}

		return getEnumAttribute(element, STRATEGY, InheritanceType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalBasicNamedQuery getNamedQuery(int index) {

		Element element = getChild(NamedQuery.NAMED_QUERY, index);

		if (element == null) {
			return null;
		}

		return buildNamedQuery(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalBasicNamedQuery getNamedQuery(int index, String name) {

		ExternalBasicNamedQuery namedQuery = getNamedQuery(index);

		if (namedQuery == null) {
			return null;
		}

		if (ObjectTools.notEquals(name, namedQuery.getName())) {
			namedQuery = null;
		}

		return namedQuery;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalNativeQuery getNativeQuery(int index) {

		Element element = getChild(NamedNativeQuery.NAMED_NATIVE_QUERY, index);

		if (element == null) {
			return null;
		}

		return buildNamedNativeQuery(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalNativeQuery getNativeQuery(int index, String name) {

		ExternalNativeQuery nativeQuery = getNativeQuery(index);

		if (nativeQuery == null) {
			return null;
		}

		if (ObjectTools.notEquals(name, nativeQuery.getName())) {
			nativeQuery = null;
		}

		return nativeQuery;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalPrimaryKeyJoinColumn getPrimaryKeyJoinColumn(int index) {

		Element element = getChild(PrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildPrimaryKeyJoinColumn(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityTable getPrimaryTable() {

		Element element = getChild(PrimaryTable.TABLE);

		if (element == null) {
			return null;
		}

		return buildPrimaryTable();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntitySecondaryTable getSecondaryTable(int index) {

		Element element = getChild(SecondaryTable.SECONDARY_TABLE);

		if (element == null) {
			return null;
		}

		return buildSecondaryTable(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSQLResultSetMapping getSqlResultSetMapping(int index) {

		Element element = getChild(SQLResultSetMapping.SQL_RESULT_SET_MAPPING, index);

		if (element == null) {
			return null;
		}

		return buildSqlResultSetMapping(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStoredProcedureQuery getStoredProcedureQuery(int index) {

		Element element = getChild(NamedStoredProcedureQuery.NAMED_STORED_PROCEDURE_QUERY, index);

		if (element == null) {
			return null;
		}

		return buildStoredProcedureQuery(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStoredProcedureQuery getStoredProcedureQuery(int index, String name) {

		ExternalStoredProcedureQuery storedProcedureQuery = getStoredProcedureQuery(index);

		if (storedProcedureQuery == null) {
			return null;
		}

		if (ObjectTools.notEquals(name, storedProcedureQuery.getName())) {
			storedProcedureQuery = null;
		}

		return storedProcedureQuery;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalNamedQuery> namedQueries() {

		int count = namedQueriesSize();
		List<ExternalNamedQuery> namedQueries = new ArrayList<ExternalNamedQuery>(count);

		for (int index = count; --index >= 0;) {
			ExternalNamedQuery namedQuery = buildNamedQuery(index);
			namedQueries.add(0, namedQuery);
		}

		return new ListListIterable<ExternalNamedQuery>(namedQueries);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int namedQueriesSize() {
		return getChildrenSize(NamedQuery.NAMED_QUERY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalNativeQuery> nativeQueries() {

		int count = nativeQueriesSize();
		List<ExternalNativeQuery> nativeQueries = new ArrayList<ExternalNativeQuery>(count);

		for (int index = count; --index >= 0;) {
			ExternalNativeQuery nativeQuery = buildNamedNativeQuery(index);
			nativeQueries.add(0, nativeQuery);
		}

		return new ListListIterable<ExternalNativeQuery>(nativeQueries);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int nativeQueriesSize() {
		return getChildrenSize(NamedNativeQuery.NAMED_NATIVE_QUERY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalPrimaryKeyJoinColumn> primaryKeyJoinColumns() {

		int count = primaryKeyJoinColumnsSize();
		List<ExternalPrimaryKeyJoinColumn> pkJoinColumns = new ArrayList<ExternalPrimaryKeyJoinColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalPrimaryKeyJoinColumn pkJoinColumn = buildPrimaryKeyJoinColumn(index);
			pkJoinColumns.add(0, pkJoinColumn);
		}

		return new ListListIterable<ExternalPrimaryKeyJoinColumn>(pkJoinColumns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int primaryKeyJoinColumnsSize() {
		return getChildrenSize(PrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAssociationOverride(int index) {
		AssociationOverride associationOverride = buildAssociationOverride(index);
		associationOverride.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAttributeOverride(int index) {
		AttributeOverride attributeOverride = buildAttributeOverride(index);
		attributeOverride.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeNamedQuery(int index) {
		NamedQuery namedQuery = buildNamedQuery(index);
		namedQuery.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeNativeQuery(int index) {
		NamedNativeQuery namedNativeQuery = buildNamedNativeQuery(index);
		namedNativeQuery.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removePrimaryKeyJoinColumn(int index) {
		PrimaryKeyJoinColumn pkJoinColumn = buildPrimaryKeyJoinColumn(index);
		pkJoinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removePrimaryTable() {
		PrimaryTable primaryTable = buildPrimaryTable();
		primaryTable.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSecondaryTable(int index) {
		SecondaryTable secondaryTable = buildSecondaryTable(index);
		secondaryTable.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSqlResultSetMapping(int index) {
		SQLResultSetMapping sqlResultSetMapping = buildSqlResultSetMapping(index);
		sqlResultSetMapping.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeStoredProcedureQuery(int index) {
		NamedStoredProcedureQuery storedProcedureQuery = buildStoredProcedureQuery(index);
		storedProcedureQuery.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalEntitySecondaryTable> secondaryTables() {

		int count = secondaryTablesSize();
		List<ExternalEntitySecondaryTable> secondaryTables = new ArrayList<ExternalEntitySecondaryTable>(count);

		for (int index = count; --index >= 0;) {
			ExternalEntitySecondaryTable secondaryTable = buildSecondaryTable(index);
			secondaryTables.add(0, secondaryTable);
		}

		return new ListListIterable<ExternalEntitySecondaryTable>(secondaryTables);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int secondaryTablesSize() {
		return getChildrenSize(SecondaryTable.SECONDARY_TABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setClassExtractorClassName(String className) {

		Element element = getChild(CLASS_EXTRACTOR);

		if ((element == null) && (className != null)) {
			element = addChild(CLASS_EXTRACTOR);
		}

		if (className != null) {
			setAttribute(element, CLASS, className);
		}
		else if (element != null) {
			removeChild(CLASS_EXTRACTOR);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDiscriminatorValue(String discriminatorValue) {
		updateChildTextNode(DISCRIMINATOR_VALUE, discriminatorValue);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setInheritanceStrategy(InheritanceType value) {

		Element element = getChild(INHERITANCE);

		if ((element == null) && (value == null)) {
			return;
		}

		if ((element == null) && (value != null)) {
			element = addChild(INHERITANCE);
		}

		if (value != null) {
			setAttribute(element, STRATEGY, value);
		}
		else if (element != null) {
			removeChild(INHERITANCE);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalSQLResultSetMapping> sqlResultSetMappings() {

		int count = sqlResultSetMappingsSize();
		List<ExternalSQLResultSetMapping> sqlResultSetMappings = new ArrayList<ExternalSQLResultSetMapping>(count);

		for (int index = count; --index >= 0;) {
			ExternalSQLResultSetMapping sqlResultSetMapping = buildSqlResultSetMapping(index);
			sqlResultSetMappings.add(0, sqlResultSetMapping);
		}

		return new ListListIterable<ExternalSQLResultSetMapping>(sqlResultSetMappings);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int sqlResultSetMappingsSize() {
		return getChildrenSize(SQLResultSetMapping.SQL_RESULT_SET_MAPPING);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalStoredProcedureQuery> storedProcedureQueries() {

		int count = storedProcedureQueriesSize();
		List<ExternalStoredProcedureQuery> storedProcedureQueries = new ArrayList<ExternalStoredProcedureQuery>(count);

		for (int index = count; --index >= 0;) {
			ExternalStoredProcedureQuery nativeQuery = buildStoredProcedureQuery(index);
			storedProcedureQueries.add(0, nativeQuery);
		}

		return new ListListIterable<ExternalStoredProcedureQuery>(storedProcedureQueries);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int storedProcedureQueriesSize() {
		return getChildrenSize(NamedStoredProcedureQuery.NAMED_STORED_PROCEDURE_QUERY);
	}
}