/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.w3c.dom.Element;

/**
 * The external form of a Join column.
 *
 * @see Mapping
 * @see JoinTable
 * @see AssociationOverride
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
class JoinColumn extends AbstractColumn
                 implements ExternalJoinColumn {

	/**
	 * The name of the element represented by this external form.
	 */
	private final String elementName;

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private final int index;

	/**
	 * The attribute name used to store and retrieve the base property.
	 */
	static final String BASE = "base";

	/**
	 * The attribute name used to store and retrieve the contiguous property.
	 */
	static final String CONTIGUOUS = "contiguous";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String JOIN_COLUMN = "join-column";

	/**
	 * The attribute name used to store and retrieve the mapped-by property.
	 */
	static final String REFERENCED_COLUMN_NAME = "referenced-column-name";

	/**
	 * Creates a new <code>JoinColumn</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	JoinColumn(AbstractExternalForm parent, int index) {
		this(parent, JOIN_COLUMN, index);
	}

	/**
	 * Creates a new <code>JoinColumn</code>.
	 *
	 * @param parent The parent of this external form
	 * @param elementName The name of the element represented by this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	JoinColumn(AbstractExternalForm parent, String elementName, int index) {
		super(parent);
		this.index = index;
		this.elementName = elementName;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected final Element addSelf(String elementName, List<String> elementNamesOrder) {
		return addChild(getParent(), elementName, index, elementNamesOrder);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(REFERENCED_COLUMN_NAME);
		names.add(UNIQUE);
		names.add(NULLABLE);
		names.add(INSERTABLE);
		names.add(UPDATABLE);
		names.add(COLUMN_DEFINITION);
		names.add(CONTIGUOUS);
		names.add(BASE);
		names.add(TABLE);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Integer getBase() {
		return getIntegerAttribute(BASE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Element getElement() {
		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return elementName;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getReferenceColumnName() {
		return getAttribute(REFERENCED_COLUMN_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Boolean isContiguous() {
		return getBooleanAttribute(CONTIGUOUS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setBase(Integer base) {
		setAttribute(BASE, base);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setContiguous(Boolean contiguous) {
		setAttribute(CONTIGUOUS, contiguous);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setReferenceColumnName(String name) {
		setAttribute(REFERENCED_COLUMN_NAME, name);
	}
}