/*******************************************************************************
 * Copyright (c) 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.gen;

import java.util.List;
import org.eclipse.persistence.tools.gen.db.Column;
import org.eclipse.persistence.tools.gen.db.Database;
import org.eclipse.persistence.tools.gen.db.ForeignKey;
import org.eclipse.persistence.tools.gen.db.Schema;
import org.eclipse.persistence.tools.gen.db.Table;
import org.eclipse.persistence.tools.gen.internal.BaseEntityGenCustomizer;
import org.eclipse.persistence.tools.gen.internal.DatabaseAnnotationNameBuilder;
import org.eclipse.persistence.tools.gen.internal.EntityGeneratorDatabaseAnnotationNameBuilder;
import org.eclipse.persistence.tools.gen.internal.GenericEntityGeneratorDatabaseAnnotationNameBuilder;
import org.eclipse.persistence.tools.gen.internal.MappingFileGenerator;
import org.eclipse.persistence.tools.gen.internal.ORMGenCustomizer;
import org.eclipse.persistence.tools.utility.collection.ListTools;
import org.eclipse.persistence.tools.utility.iterable.TransformationIterable;
import org.eclipse.persistence.tools.utility.iterator.TransformationIterator;

/**
 * This class is an entry point for dynamic entity generation. It provides API for discovering
 * database metadata (schemata, table names, etc) and generating EclipseLink JPA, mapping files.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.5
 * @author John Bracken
 */
public class RelationalEntityGenerator {

	/**
	 * JPT database reference.
	 */
	private Database database;

	/**
	 * Constructs a new {@link RelationalEntityGenerator} based on the provided db connection
	 *
	 * @param database the provided database to use in this dynamic entity generator.
	 */
	public RelationalEntityGenerator(Database database) {
		super();
		this.database = database;
	}

	/**
	 * Constructs a {@link DatabaseAnnotationNameBuilder}.
	 */
	private DatabaseAnnotationNameBuilder buildDatabaseAnnotationNameBuilder() {
		return new LocalDatabaseAnnotationNameBuilder(GenericEntityGeneratorDatabaseAnnotationNameBuilder.instance());
	}

	/**
	 * Creates the {@link ORMGenCustomizer} to be used generating dynamic entities.
	 */
	private ORMGenCustomizer createORMGenCustomizer(Schema schema, List<String> tableNames){

		ORMGenCustomizer customizer = new BaseEntityGenCustomizer();
		customizer.init(null, schema);
		customizer.setTableNames(tableNames);

		return customizer;
	}

	/**
	 * Generates a {@link String} representation of the EclipseLink orm.xml file based on all of the
	 * tables in the provided schema.
	 *
	 * @param schemaName the schema name.
	 * @return a mapping xml file that describes the dynamic mappings.
	 */
	public String generateMappings(String schemaName) {

		Schema schema = this.database.getSchemaNamed(schemaName);
		ORMGenCustomizer ormGenCustomizer = createORMGenCustomizer(schema, transformToTableNames(schema.getTables()));
		ormGenCustomizer.setDatabaseAnnotationNameBuilder(buildDatabaseAnnotationNameBuilder());

		return MappingFileGenerator.generate(ormGenCustomizer);
	}

	/**
	 * Generates a {@link String} representation of the EclipseLink orm.xml file based on the provided
	 * schema name and {@link List} of table names.
	 *
	 * @param schemaName the schema name.
	 * @param tableNames the names of the tables to generate mappings for.
	 * @return a mapping xml file that describes the dynamic mappings
	 */
	public String generateMappings(String schemaName, List<String> tableNames) {

		ORMGenCustomizer ormGenCustomizer = createORMGenCustomizer(this.database.getSchemaNamed(schemaName), tableNames);
		ormGenCustomizer.setDatabaseAnnotationNameBuilder(buildDatabaseAnnotationNameBuilder());

		return  MappingFileGenerator.generate(ormGenCustomizer);
	}

	/**
	 * Lists the schema names associated with this generator's database.
	 *
	 * @return {@link List} of the schema names associated with the database.
	 */
	public List<String> getSchemaNames(){
		return ListTools.list(new TransformationIterator<Schema, String>(database.getSchemata()) {
			@Override
			protected String transform(Schema next) {
				return next.getName();
			}
		});
	}

	/**
	 * Lists the table names associated with this generator's database based on the provided schema name.
	 *
	 * @param schemaName the schema name.
	 * @return {@link List} of the schema names associated with the database.
	 */
	public List<String> getTableNames(String schemaName) {
		return transformToTableNames(this.database.getSchemaNamed(schemaName).getTables());
	}

	/**
	 * Creates a {@link List} of table names based on the provided {@link Table}s.
	 */
	private List<String> transformToTableNames(Iterable<Table> tables) {
		return ListTools.list(new TransformationIterable<Table, String>(tables) {
			@Override
			protected String transform(Table table) {
				return table.getName();
			}
		});
	}

	/**
	 * Adapt the JPA platform-supplied builder to the builder interface expected by the entity generator.
	 */
	private class LocalDatabaseAnnotationNameBuilder implements DatabaseAnnotationNameBuilder {
		private EntityGeneratorDatabaseAnnotationNameBuilder builder;
		LocalDatabaseAnnotationNameBuilder(EntityGeneratorDatabaseAnnotationNameBuilder builder) {
			super();
			this.builder = builder;
		}
		@Override
		public String buildColumnAnnotationName(String attributeName, Column column) {
			return this.builder.buildColumnAnnotationName(attributeName, column);
		}
		@Override
		public String buildJoinColumnAnnotationName(Column column) {
			return this.builder.buildJoinColumnAnnotationName(column);
		}
		@Override
		public String buildJoinColumnAnnotationName(String attributeName, ForeignKey foreignKey) {
			return this.builder.buildJoinColumnAnnotationName(attributeName, foreignKey);
		}
		@Override
		public String buildJoinTableAnnotationName(Table table) {
			return this.builder.buildJoinTableAnnotationName(table);
		}
		@Override
		public String buildTableAnnotationName(String entityName, Table table) {
			return this.builder.buildTableAnnotationName(entityName, table);
		}
	}
}