/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinTable;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of a Join table, which is a child of a relationship mapping.
 *
 * @see RelationshipMapping
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class JoinTable extends Table
                      implements ExternalJoinTable {

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String JOIN_TABLE = "join-table";

	/**
	 * Creates a new <code>JoinTable</code>.
	 * 
	 * @param parent The parent of this external form
	 */
	JoinTable(AbstractExternalForm parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addInverseJoinColumn(int index, String name) {
		InverseJoinColumn joinColumn = buildInverseJoinColumn(index);
		joinColumn.addSelf();
		joinColumn.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addJoinColumn(int index, String name) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.addSelf();
		joinColumn.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(JoinColumn.JOIN_COLUMN);
		names.add(InverseJoinColumn.INVERSE_JOIN_COLUMN);
		names.add(UniqueConstraint.UNIQUE_CONSTRAINT);
		return names;
	}

	private InverseJoinColumn buildInverseJoinColumn(int index) {
		return new InverseJoinColumn(this, index);
	}

	private JoinColumn buildJoinColumn(int index) {
		return new JoinColumn(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return JOIN_TABLE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getInverseJoinColumn(int index) {

		Element element = getChild(InverseJoinColumn.INVERSE_JOIN_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildInverseJoinColumn(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getJoinColumn(int index) {

		Element element = getChild(JoinColumn.JOIN_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildJoinColumn(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalJoinColumn> inverseJoinColumns() {

		int count = inverseJoinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalJoinColumn joinColumn = buildInverseJoinColumn(index);
			joinColumns.add(0, joinColumn);
		}

		return new ListListIterable<ExternalJoinColumn>(joinColumns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int inverseJoinColumnsSize() {
		return getChildrenSize(InverseJoinColumn.INVERSE_JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isDefault() {
		return !(hasChildren() || hasAttributes());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalJoinColumn> joinColumns() {

		int count = joinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalJoinColumn joinColumn = buildJoinColumn(index);
			joinColumns.add(0, joinColumn);
		}

		return new ListListIterable<ExternalJoinColumn>(joinColumns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinColumnsSize() {
		return getChildrenSize(JoinColumn.JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeInverseJoinColumn(int index) {
		InverseJoinColumn joinColumn = buildInverseJoinColumn(index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.removeSelf();
	}
}