/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.annotations.Direction;
import org.eclipse.persistence.tools.mapping.orm.ExternalStoredProcedureParameter;
import org.w3c.dom.Element;

/**
 * The external form of a stored procedure parameter, which is a child of an
 * named stored procedure query.
 *
 * @see NamedStoredProcedureQuery
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class StoredProcedureParameter extends AbstractExternalForm
                                     implements ExternalStoredProcedureParameter {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * The attribute name used to store and retrieve the direction property.
	 */
	static final String DIRECTION = "direction";

	/**
	 * The attribute name used to store and retrieve the jdbc-type property.
	 */
	static final String JDBC_TYPE = "jdbc-type";

	/**
	 * The attribute name used to store and retrieve the jdbc-type-name property.
	 */
	static final String JDBC_TYPE_NAME = "jdbc-type-name";

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	static final String NAME = "name";

	/**
	 * The attribute name used to store and retrieve the query-parameter property.
	 */
	static final String QUERY_PARAMETER = "query-parameter";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String STORED_PROCEDURE_PARAMETER = "parameter";

	/**
	 * The attribute name used to store and retrieve the type property.
	 */
	static final String TYPE = "type";

	/**
	 * Creates a new <code>StoredProcedureParameter</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with
	 * the same type owned by the parent
	 */
	StoredProcedureParameter(NamedStoredProcedureQuery parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {
		return addChild(getParent(), elementName, index, elementNamesOrder);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DIRECTION);
		names.add(NAME);
		names.add(QUERY_PARAMETER);
		names.add(TYPE);
		names.add(JDBC_TYPE);
		names.add(JDBC_TYPE_NAME);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Direction getDirection() {
		return getEnumAttribute(DIRECTION, Direction.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getChild(getParent(), STORED_PROCEDURE_PARAMETER, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return STORED_PROCEDURE_PARAMETER;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getJdbcType() {
		return getIntegerAttribute(JDBC_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getJdbcTypeName() {
		return getAttribute(JDBC_TYPE_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getQueryParameter() {
		return getAttribute(QUERY_PARAMETER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getType() {
		return getAttribute(TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDirection(Direction directionType) {
		setAttribute(DIRECTION, directionType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setJdbcType(Integer jdbcType) {
		setAttribute(JDBC_TYPE, jdbcType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setJdbcTypeName(String jdbcTypeName) {
		setAttribute(JDBC_TYPE_NAME, jdbcTypeName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setQueryParameter(String queryParameterName) {
		setAttribute(QUERY_PARAMETER, queryParameterName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setType(String typeName) {
		setAttribute(TYPE, typeName);
	}
}