/*******************************************************************************
 * Copyright (c) 2009, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.collection;

import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;

/**
 * Thread-safe implementation of the {@link Bag} interface.
 */
public class SynchronizedBag<E>
	implements Bag<E>, Serializable
{
	/** Backing bag. */
	private final Bag<E> bag;

	/** Object to synchronize on. */
	private final Object mutex;

	private static final long serialVersionUID = 1L;


	// ********** constructors **********

	/**
	 * Construct a synchronized bag that wraps the
	 * specified bag and locks on the specified mutex.
	 */
	public SynchronizedBag(Bag<E> bag, Object mutex) {
		super();
		if ((bag == null) || (mutex == null)) {
			throw new NullPointerException();
		}
		this.bag = bag;
		this.mutex = mutex;
	}

	/**
	 * Construct a synchronized bag that wraps the
	 * specified bag and locks on itself.
	 */
	public SynchronizedBag(Bag<E> bag) {
		super();
		if (bag == null) {
			throw new NullPointerException();
		}
		this.bag = bag;
		this.mutex = this;
	}

	/**
	 * Construct a synchronized bag that locks on the specified mutex.
	 */
	public SynchronizedBag(Object mutex) {
		this(new HashBag<E>(), mutex);
	}

	/**
	 * Construct a synchronized bag that locks on itself.
	 */
	public SynchronizedBag() {
		this(new HashBag<E>());
	}


	// ********** Bag implementation **********

	@Override
	public boolean add(E o, int count) {
		synchronized (this.mutex) {
			return this.bag.add(o, count);
		}
	}

	@Override
	public int count(Object o) {
		synchronized (this.mutex) {
			return this.bag.count(o);
		}
	}

	@Override
	public Iterator<Bag.Entry<E>> entries() {
		synchronized (this.mutex) {
			return this.bag.entries();
		}
	}

	@Override
	public boolean remove(Object o, int count) {
		synchronized (this.mutex) {
			return this.bag.remove(o, count);
		}
	}

	@Override
	public int uniqueCount() {
		synchronized (this.mutex) {
			return this.bag.uniqueCount();
		}
	}

	@Override
	public Iterator<E> uniqueIterator() {
		synchronized (this.mutex) {
			return this.bag.uniqueIterator();
		}
	}


	// ********** Collection implementation **********

	@Override
	public boolean add(E e) {
		synchronized (this.mutex) {
			return this.bag.add(e);
		}
	}

	@Override
	public boolean addAll(Collection<? extends E> c) {
		synchronized (this.mutex) {
			return this.bag.addAll(c);
		}
	}

	@Override
	public void clear() {
		synchronized (this.mutex) {
			this.bag.clear();
		}
	}

	@Override
	public boolean contains(Object o) {
		synchronized (this.mutex) {
			return this.bag.contains(o);
		}
	}

	@Override
	public boolean containsAll(Collection<?> c) {
		synchronized (this.mutex) {
			return this.bag.containsAll(c);
		}
	}

	@Override
	public boolean isEmpty() {
		synchronized (this.mutex) {
			return this.bag.isEmpty();
		}
	}

	@Override
	public Iterator<E> iterator() {
		synchronized (this.mutex) {
			return this.bag.iterator();
		}
	}

	@Override
	public boolean remove(Object o) {
		synchronized (this.mutex) {
			return this.bag.remove(o);
		}
	}

	@Override
	public boolean removeAll(Collection<?> c) {
		synchronized (this.mutex) {
			return this.bag.removeAll(c);
		}
	}

	@Override
	public boolean retainAll(Collection<?> c) {
		synchronized (this.mutex) {
			return this.bag.retainAll(c);
		}
	}

	@Override
	public int size() {
		synchronized (this.mutex) {
			return this.bag.size();
		}
	}

	@Override
	public Object[] toArray() {
		synchronized (this.mutex) {
			return this.bag.toArray();
		}
	}

	@Override
	public <T> T[] toArray(T[] a) {
		synchronized (this.mutex) {
			return this.bag.toArray(a);
		}
	}


	// ********** additional public protocol **********

	/**
	 * Return the object the stack locks on while performing
	 * its operations.
	 */
	public Object getMutex() {
		return this.mutex;
	}


	// ********** Object overrides **********

	@Override
	public String toString() {
		synchronized (this.mutex) {
			return this.bag.toString();
		}
	}

	private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException {
		synchronized (this.mutex) {
			s.defaultWriteObject();
		}
	}
}