/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.model.value;

import java.util.Arrays;
import org.eclipse.persistence.tools.utility.model.Model;
import org.eclipse.persistence.tools.utility.model.event.PropertyChangeEvent;
import org.eclipse.persistence.tools.utility.model.listener.PropertyChangeListener;

/**
 * Extend {@link ValueAspectAdapter} to listen to one or more
 * properties of the value in the wrapped value model.
 */
@SuppressWarnings("nls")
public class ValuePropertyAdapter<V extends Model>
	extends ValueAspectAdapter<V>
{
	/** The names of the value's properties we listen to. */
	protected final String[] propertyNames;

	/** Listener that listens to the value. */
	protected final PropertyChangeListener valuePropertyListener;


	// ********** constructors **********

	/**
	 * Construct an adapter for the specified value properties.
	 */
	public ValuePropertyAdapter(ModifiablePropertyValueModel<V> valueHolder, String... propertyNames) {
		super(valueHolder);
		if (propertyNames == null) {
			throw new NullPointerException();
		}
		this.propertyNames = propertyNames;
		this.valuePropertyListener = this.buildValuePropertyListener();
	}


	// ********** initialization **********

	protected PropertyChangeListener buildValuePropertyListener() {
		return new PropertyChangeListener() {
			@Override
			public void propertyChanged(PropertyChangeEvent event) {
				ValuePropertyAdapter.this.propertyChanged(event);
			}
			@Override
			public String toString() {
				return "value property listener: " + Arrays.asList(ValuePropertyAdapter.this.propertyNames);
			}
		};
	}


	// ********** ValueAspectAdapter implementation **********

	@Override
	protected void engageValue_() {
		for (String propertyName : this.propertyNames) {
			this.value.addPropertyChangeListener(propertyName, this.valuePropertyListener);
		}
	}

	@Override
	protected void disengageValue_() {
		for (String propertyName : this.propertyNames) {
			this.value.removePropertyChangeListener(propertyName, this.valuePropertyListener);
		}
	}


	// ********** change events **********

	protected void propertyChanged(@SuppressWarnings("unused") PropertyChangeEvent event) {
		this.valueAspectChanged();
	}
}