/*
 * Copyright (c) 2006, 2012, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is the proprietary information of Oracle Corporation.
 * Use is subject to license terms.
 */
package org.eclipse.persistence.tools.mapping.orm;

import javax.persistence.InheritanceType;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;

/**
 * This interface and inherited behavior describes the configurable properties for an entity. One or
 * more entities can be declared as part of an ORM configuration.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @see javax.persistence.Entity
 * @see http://jcp.org/aboutJava/communityprocess/final/jsr220/index.html, persistence specificaton
 *
 * @version 2.5
 * @author John Bracken
 */
public interface ExternalEntity extends ExternalMappedSuperClassEntity {

	/**
	 * Adds a override with the given name.
	 */
	void addAssociationOverride(int index, String name);

	/**
	 * Adds a override with the given name.
	 */
	void addAttributeOverride(int index, String name);

	/**
	 * Creates a new discriminator column definition replacing the old one.
	 */
	ExternalDiscriminatorColumn addDiscriminatorColumn(String columnName);

	/**
	 * Adds a named query with the given name.
	 */
	void addNamedQuery(int index, String name);

	/**
	 * Adds a native query with the given name.
	 */
	void addNativeQuery(int index, String name);

	/**
	 * Adds a primary key join column with the given name.
	 */
	void addPrimaryKeyJoinColumn(int index, String name);

	/**
	 * Creates new primary table for this entity replacing the old one.
	 */
	ExternalEntityTable addPrimaryTable(String tableName);

	/**
	 * Adds a secondary table with the given name.
	 */
	void addSecondaryTable(int index, String name, String catalog, String schema);

	/**
	 * Adds a sql result set mapping with the given name.
	 */
	void addSqlResultSetMapping(int index, String name);

	/**
	 * Adds a stored procedure query with the given name.
	 */
	void addStoredProcedureQuery(int index, String name, String procedureName);

	/**
	 * Returns a list of association overrides defined on this entity.
	 */
	ListIterable<ExternalAssociationOverride> associationOverrides();

	/**
	 * Returns the count of association overrides defined on this entity.
	 */
	int associationOverridesSize();

	/**
	 * Returns a list of attribute overrides defined for this entity.
	 */
	ListIterable<ExternalAttributeOverride> attributeOverrides();

	/**
	 * Returns the count attribute overrides defined for this entity.
	 */
	int attributeOverridesSize();

	/**
	 * Returns the association override at the specified index.
	 */
	ExternalAssociationOverride getAssociationOverride(int index);

	/**
	 * Returns the association override at the specified index.
	 */
	ExternalAttributeOverride getAttributeOverride(int index);

	/**
	 * Returns the class extractor class name defined for this entity.
	 */
	String getClassExtractorClassName();

	/**
	 * Returns the discriminator column definition.
	 */
	ExternalDiscriminatorColumn getDiscriminatorColumn();

	/**
	 * Returns the discriminator value.
	 */
	String getDiscriminatorValue();

	/**
	 * Returns the inheritance strategy type if one has been declared.
	 */
	InheritanceType getInheritenceStrategy();

	/**
	 * Returns the name of the entity.
	 */
	String getName();

	/**
	 * Returns a named query with the given name.
	 */
	ExternalBasicNamedQuery getNamedQuery(int index);

	/**
	 * Returns a named query with the given name.
	 */
	ExternalBasicNamedQuery getNamedQuery(int index, String name);

	/**
	 * Returns a native query with the given name.
	 */
	ExternalNativeQuery getNativeQuery(int index);

	/**
	 * Returns a native query with the given name.
	 */
	ExternalNativeQuery getNativeQuery(int index, String name);

	/**
	 * Returns a primary key join column with the given name.
	 */
	ExternalPrimaryKeyJoinColumn getPrimaryKeyJoinColumn(int index);

	/**
	 * Returns the primary table definition for this entity.
	 */
	ExternalEntityTable getPrimaryTable();

	/**
	 * Returns a the secondary table definition for the given catalog, schema and table name.
	 */
	ExternalEntitySecondaryTable getSecondaryTable(int index);

	/**
	 * Returns the SQL Result Set Mappings at the given position.
	 */
	ExternalSQLResultSetMapping getSqlResultSetMapping(int index);

	/**
	 * Returns the stored procedure query at the index.
	 */
	ExternalStoredProcedureQuery getStoredProcedureQuery(int index);

	/**
	 * Returns the stored procedure query with the given name.
	 */
	ExternalStoredProcedureQuery getStoredProcedureQuery(int index, String name);

	/**
	 * Returns a list of named queries defined for this entity.
	 */
	ListIterable<ExternalNamedQuery> namedQueries();

	/**
	 * Returns the count of named queried defined for this entity.
	 */
	int namedQueriesSize();

	/**
	 * Returns a list of native queries defined on this entity.
	 */
	ListIterable<ExternalNativeQuery> nativeQueries();

	/**
	 * Returns the count of native queries defined on this entity.
	 */
	int nativeQueriesSize();

	/**
	 * Returns a list of the primary key join columns.
	 */
	ListIterable<ExternalPrimaryKeyJoinColumn> primaryKeyJoinColumns();

	/**
	 * Returns the count of primary key join columns.
	 */
	int primaryKeyJoinColumnsSize();

	/**
	 * Removes the override named.
	 */
	void removeAssociationOverride(int index);

	/**
	 * Removes the override named.
	 */
	void removeAttributeOverride(int index);

	/**
	 * Removes the named query named.
	 */
	void removeNamedQuery(int index);

	/**
	 * Removes the native query named.
	 */
	void removeNativeQuery(int index);

	/**
	 * Removes the primary key join column named.
	 */
	void removePrimaryKeyJoinColumn(int index);

	/**
	 * Removes the primary table definition from this entity if it exists.
	 */
	void removePrimaryTable();

	/**
	 * Removes the secondary table named.
	 */
	void removeSecondaryTable(int index);

	/**
	 * Removes the sql result set mapping named.
	 */
	void removeSqlResultSetMapping(int index);

	/**
	 * Removes the stored procedure query named.
	 */
	void removeStoredProcedureQuery(int index);

	/**
	 * Returns a list of the secondary table definitions for this entity.
	 */
	ListIterable<ExternalEntitySecondaryTable> secondaryTables();

	/**
	 * Returns the count of secondary table definitions for this entity.
	 */
	int secondaryTablesSize();

	/**
	 * Sets the name of the class extractor class declared for this entity.
	 */
	void setClassExtractorClassName(String className);

	/**
	 * set the discriminator value.
	 */
	void setDiscriminatorValue(String discriminatorValue);

	/**
	 * Set the inheritance strategy.
	 */
	void setInheritanceStrategy(InheritanceType type);

	/**
	 * Set the entity name.
	 */
	void setName(String name);

	/**
	 * Returns a list of sql result set mappings defined on this entity.
	 */
	ListIterable<ExternalSQLResultSetMapping> sqlResultSetMappings();

	/**
	 * Returns the count of SQL result set mappings defined on this entity.
	 */
	int sqlResultSetMappingsSize();

	/**
	 * Returns a list of named stored procedure queries defined for this entity.
	 */
	ListIterable<ExternalStoredProcedureQuery> storedProcedureQueries();

	/**
	 * Returns the count of named stored procedure queried defined for this entity.
	 */
	int storedProcedureQueriesSize();
}