/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.FetchType;
import org.eclipse.persistence.annotations.JoinFetchType;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicCollectionMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalBatchFetch;
import org.eclipse.persistence.tools.mapping.orm.ExternalCollectionTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;

/**
 * The external form for a basic collection mapping, which is a child of an entity.
 *
 * @see Entity
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class BasicCollectionMapping extends ConvertibleMapping
                                   implements ExternalBasicCollectionMapping {

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String BASIC_COLLECTION = "basic-collection";

	/**
	 * The attribute name used to store and retrieve the fetch property.
	 */
	static final String FETCH = "fetch";

	/**
	 * The element name used to store and retrieve the join-fetch child text node.
	 */
	static final String JOIN_FETCH = "join-fetch";

	/**
	 * The element name used to store and retrieve the value-column child node.
	 */
	static final String VALUE_COLUMN = "value-column";

	/**
	 * Creates a new <code>BasicCollectionMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	BasicCollectionMapping(EmbeddableEntity parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void addBatchFetch() {
		BatchFetch batchFetch = buildBatchFetch();
		batchFetch.addSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addCollectionTable(String name) {
		CollectionTable collectionTable = buildCollectionTable();
		collectionTable.addSelf();
		collectionTable.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addValueColumn(String columnName) {
		EntityColumn valueColumn = buildValueColumn();
		valueColumn.addSelf();
		valueColumn.setName(columnName);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(FETCH);
		names.add(ACCESS);
		return names;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(VALUE_COLUMN);
		names.add(CONVERT);
		names.add(Converter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(CollectionTable.COLLECTION_TABLE);
		names.add(JOIN_FETCH);
		names.add(BatchFetch.BATCH_FETCH);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private BatchFetch buildBatchFetch() {
		return new BatchFetch(this);
	}

	private CollectionTable buildCollectionTable() {
		return new CollectionTable(this);
	}


	private EntityColumn buildValueColumn() {
		return new EntityColumn(this, VALUE_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalBatchFetch getBatchFetch() {

		if (hasChild(BatchFetch.BATCH_FETCH)) {
			return buildBatchFetch();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalCollectionTable getCollectionTable() {

		if (hasChild(CollectionTable.COLLECTION_TABLE)) {
			return buildCollectionTable();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return BASIC_COLLECTION;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FetchType getFetchType() {
		return getEnumAttribute(FETCH, FetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public JoinFetchType getJoinFetchType() {
		return getChildEnumNode(JOIN_FETCH, JoinFetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getValueColumn() {

		if (hasChild(VALUE_COLUMN)) {
			return buildValueColumn();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeBatchFetch() {
		removeChild(BatchFetch.BATCH_FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeCollectionTable() {
		CollectionTable collectionTable = buildCollectionTable();
		collectionTable.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeValueColumn() {
		EntityColumn valueColumn = buildValueColumn();
		valueColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setFetchType(FetchType type) {
		setAttribute(FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setJoinFetchType(JoinFetchType type) {
		updateChildTextNode(JOIN_FETCH, type);
	}
}