/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.persistence.dom;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import javax.persistence.ValidationMode;
import org.eclipse.persistence.config.PersistenceUnitProperties;
import org.eclipse.persistence.tools.mapping.orm.dom.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.persistence.ExternalPersistenceUnit;
import org.eclipse.persistence.tools.mapping.persistence.ExternalProperty;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of a persistence unit.
 *
 * @see PersistenceConfiguration
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class PersistenceUnit extends AbstractExternalForm
                            implements ExternalPersistenceUnit {

	/**
	 * The position of this external form within the list of persistence units contained in the parent.
	 */
	private int index;

	/**
	 * The element name of the property for the default cache type.
	 */
	static final String CACHE_TYPE = "cache-type";

	/**
	 * The element name of the child text nodes for the listed entity class names.
	 */
	static final String CLASS = "class";

	/**
	 * The element name of the child text node for the description.
	 */
	static final String DESCRIPTION = "description";

	/**
	 * The element name of the child boolean node for the exclude unlisted classes flag.
	 */
	static final String EXCLUDE_UNLISTED_CLASSES = "exclude-unlisted-classes";

	/**
	 * The element name of the child text nodes for the Java archive files.
	 */
	static final String JAR_FILE = "jar-file";

	/**
	 * The element name of the child text node for the JTA data source.
	 */
	static final String JTA_DATA_SOURCE = "jta-data-source";

	/**
	 * The element name of the child text nodes for the JPA mapping files.
	 */
	static final String MAPPING_FILE = "mapping-file";

	/**
	 * The element name of the property for the persistence unit name.
	 */
	static final String NAME = "name";

	/**
	 * The element name of the child text node for the non-JTA data source.
	 */
	static final String NON_JTA_DATA_SOURCE = "non-jta-data-source";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String PERSISTENCE_UNIT = "persistence-unit";

	/**
	 * The element name of the child text node for the provider.
	 */
	static final String PROVIDER = "provider";

	/**
	 * The element name of the property for the default transaction type.
	 */
	static final String TRANSACTION_TYPE = "transaction-type";

	/**
	 * The element name of the property for the default validation mode type.
	 */
	static final String VALIDATION_MODE = "validation-mode";

	/**
	 * Creates a new <code>PersistenceUnit</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of this external form within the parent's children
	 */
	PersistenceUnit(PersistenceConfiguration parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalProperty addProperty(int index, String name, String value) {
		Property property = buildProperty(index);
		property.addSelf();
		property.setName(name);
		property.setValue(value);
		return property;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalProperty addProperty(String name, String value) {
		Property property = buildProperty(propertiesSize());
		property.addSelf();
		property.setName(name);
		property.setValue(value);
		return property;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {
		return addChild(getParent(), elementName, index, elementNamesOrder);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		names.add(PROVIDER);
		names.add(JTA_DATA_SOURCE);
		names.add(NON_JTA_DATA_SOURCE);
		names.add(MAPPING_FILE);
		names.add(JAR_FILE);
		names.add(CLASS);
		names.add(EXCLUDE_UNLISTED_CLASSES);
		names.add(CACHE_TYPE);
		names.add(VALIDATION_MODE);
		names.add(Property.PROPERTIES);
		return names;
	}

	private Property buildProperty(int index) {
		return new Property(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CacheType getCachType() {
		return getChildEnumNode(CACHE_TYPE, CacheType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return PERSISTENCE_UNIT;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getJdbcDriver() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.JDBC_DRIVER);
		return (property != null) ? property.getValue() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getJdbcPassword() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.JDBC_PASSWORD);
		return (property != null) ? property.getValue() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getJdbcUrl() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.JDBC_URL);
		return (property != null) ? property.getValue() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getJdbcUser() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.JDBC_USER);
		return (property != null) ? property.getValue() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getJtaDatasource() {
		return getChildTextNode(JTA_DATA_SOURCE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getNonJtaDatasource() {
		return getChildTextNode(NON_JTA_DATA_SOURCE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getPessimisticLockTimeout() {

		ExternalProperty property = getProperty(PersistenceUnitProperties.PESSIMISTIC_LOCK_TIMEOUT);

		if (property != null) {
			try {
				return Integer.valueOf(property.getValue());
			}
			catch (NumberFormatException ex) {
				return null;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalProperty> getProperties(String name) {

		List<ExternalProperty> properties = new ArrayList<ExternalProperty>();

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				properties.add(property);
			}
		}

		return properties;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalProperty getProperty(int index) {

		Element element = getChild(Property.PROPERTIES);

		if (element == null) {
			return null;
		}

		element = getChild(element, Property.PROPERTY, index);

		if (element == null) {
			return null;
		}

		return buildProperty(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalProperty getProperty(String name) {

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				return property;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalProperty getProperty(String name, int index) {

		ExternalProperty property = getProperty(index);

		if ((property != null) && ObjectTools.equals(name, property.getName())) {
			return property;
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalProperty getProperty(String name, String value) {

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(),  name) &&
			    ObjectTools.equals(property.getValue(), value)) {

				return property;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getProvider() {
		return getChildTextNode(PROVIDER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getQueryTimeout() {

		ExternalProperty property = getProperty(PersistenceUnitProperties.QUERY_TIMEOUT);

		if (property != null) {
			try {
				return Integer.valueOf(property.getValue());
			}
			catch (NumberFormatException ex) {
				return null;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TransactionType getTransactionType() {
		return getEnumAttribute(TRANSACTION_TYPE, TransactionType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ValidationMode getValidationMode() {
		return getChildEnumNode(VALIDATION_MODE, ValidationMode.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<String> includedClasses() {
		return new ListListIterable<String>(getChildrenTextNode(CLASS));
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int includedClassesSize() {
		return getChildrenSize(CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<String> jarFiles() {
		return new ListListIterable<String>(getChildrenTextNode(JAR_FILE));
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int jarFilesSize() {
		return getChildrenSize(JAR_FILE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<String> mappingFiles() {
		return new ListListIterable<String>(getChildrenTextNode(MAPPING_FILE));
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mappingFilesSize() {
		return getChildrenSize(MAPPING_FILE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalProperty> properties() {

		int count = propertiesSize();
		List<ExternalProperty> properties = new ArrayList<ExternalProperty>(count);

		for (int index = count; --index >= 0;) {
			ExternalProperty externalProperty = buildProperty(index);
			properties.add(0, externalProperty);
		}

		return new ListListIterable<ExternalProperty>(properties);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int propertiesSize() {

		Element element = getChild(Property.PROPERTIES);

		if (element != null) {
			return getChildrenSize(element, Property.PROPERTY);
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int propertiesSize(String name) {

		int count = 0;

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				count++;
			}
		}

		return count;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeProperty(int index) {
		Property property = buildProperty(index);
		property.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeProperty(String name, String value) {
		Property property = (Property) getProperty(name, value);
		property.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setCacheType(CacheType type) {
		updateChildTextNode(CACHE_TYPE, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setExcludeUnlistedClasses(boolean excludeUnlistedClasses) {
		updateChildTextNode(EXCLUDE_UNLISTED_CLASSES, excludeUnlistedClasses);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setIncludedClasses(ListIterator<String> classes) {

		removeChildren(CLASS);

		while (classes.hasNext()) {
			addChildTextNode(CLASS, classes.next());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setJarFiles(ListIterator<String> jarFiles) {

		removeChildren(JAR_FILE);

		while (jarFiles.hasNext()) {
			addChildTextNode(JAR_FILE, jarFiles.next());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setJtaDatasource(String jtaDatasource) {
		updateChildTextNode(JTA_DATA_SOURCE, jtaDatasource);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMappingFiles(ListIterator<String> mappingFiles) {

		removeChildren(MAPPING_FILE);

		while (mappingFiles.hasNext()) {
			addChildTextNode(MAPPING_FILE, mappingFiles.next());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setNonJtaDatasource(String nonJtaDatasource) {
		updateChildTextNode(NON_JTA_DATA_SOURCE, nonJtaDatasource);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPessimisticLockTimeout(Integer length) {

		ExternalProperty property = getProperty(PersistenceUnitProperties.PESSIMISTIC_LOCK_TIMEOUT);

		if ((length == null) && (property != null)) {
			removeProperty(PersistenceUnitProperties.PESSIMISTIC_LOCK_TIMEOUT, property.getValue());
		}
		else if ((length != null) && (property == null)) {
			addProperty(PersistenceUnitProperties.PESSIMISTIC_LOCK_TIMEOUT, length.toString());
		}
		else if ((length != null) && (property != null)) {
			property.setValue(length.toString());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setProvider(String provider) {
		updateChildTextNode(PROVIDER, provider);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setQueryTimeout(Integer length) {

		ExternalProperty property = getProperty(PersistenceUnitProperties.QUERY_TIMEOUT);

		if ((length == null) && (property != null)) {
			removeProperty(PersistenceUnitProperties.QUERY_TIMEOUT, property.getValue());
		}
		else if ((length != null) && (property == null)) {
			addProperty(PersistenceUnitProperties.QUERY_TIMEOUT, length.toString());
		}
		else if ((length != null) && (property != null)) {
			property.setValue(length.toString());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTransactionType(TransactionType transactionType) {
		setAttribute(TRANSACTION_TYPE, transactionType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setValidationMode(ValidationMode type) {
		updateChildTextNode(VALIDATION_MODE, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean shouldExcludeUnlistedClasses() {
		Boolean value = getChildBooleanNode(EXCLUDE_UNLISTED_CLASSES);
		return (value == null) ? false : value;
	}
}