/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm;

import java.net.URL;
import java.util.List;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * This interface defines the root-level contract for providing ORM.xml metadata. The properties
 * defined by this interface define all root and subsequently nested information available in an
 * ORM.xml.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @see oracle.toplink.workbench.mappingsmodel.spi.jpa.orm.ExternalORMRepositoryFactory
 * @see oracle.toplink.workbench.mappingsmodel.spi.SPIManager
 * @see http://jcp.org/aboutJava/communityprocess/final/jsr220/index.html, persistence specificaton
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public interface ExternalORMConfiguration extends ExternalTenantDiscriminatorColumnProvider,
                                                  ExternalAccessType,
                                                  ExternalConverterProvider {

	/**
	 * The element name of the child text node for the default catalog name.
	 */
	String CATALOG = "catalog";

	/**
	 * The element name of the child text node for the description.
	 */
	String DESCRIPTION = "description";

	/**
	 * The node name used to store and retrieve the element encapsulated by this external form.
	 */
	String ENTITY_MAPPINGS = "entity-mappings";

	/**
	 * The element name of the child text node for the default package name.
	 */
	String PACKAGE = "package";

	/**
	 * The element name of the child text node for the default schema name.
	 */
	String SCHEMA = "schema";

	/**
	 * Adds a embedded entity with the given name.
	 */
	ExternalEmbeddable addEmbeddable(String embeddableClassName);

	/**
	 * Adds a entity for the given class name.
	 */
	ExternalEntity addEntity(String entityClassName);

	/**
	 * Adds a mapped super class entity with the given name
	 */
	ExternalMappedSuperClass addMappedSuperClass(String mappedClassName);

	/**
	 * Adds a native query with the given name.
	 */
	ExternalNativeQuery addNamedNativeQuery(String name);

	/**
	 * Adds a named query with the given name.
	 */
	ExternalBasicNamedQuery addNamedQuery(String name);

	/**
	 * Adds a stored procedure query with the given name.
	 */
	ExternalNamedStoredProcedureQuery addNamedStoredProcedureQuery(String name);

	/**
	 * Creates new persistence unit representing this ORM's persistence unit metadata.
	 */
	ExternalPersistenceUnit addPersistenceUnitMetaData();

	/**
	 * Adds a sequence generator with the given name.
	 */
	ExternalSequenceGenerator addSequenceGenerator(String name);

	/**
	 * Adds a SQL result set mapping with the given name.
	 */
	ExternalSQLResultSetMapping addSqlResultSetMapping(String name);

	/**
	 * Adds a table generator with the given name
	 */
	ExternalTableGenerator addTableGenerator(String name);

	/**
	 * Returns a list of all embedded entities associated with this ORM configuration.
	 */
	List<ExternalEmbeddable> embeddables();

	/**
	 * Returns the count of embedded entities associated with this ORM configuration.
	 */
	int embeddablesSize();

	/**
	 * Returns a list of all entities associated with this ORM configuration.
	 */
	List<ExternalEntity> entities();

	/**
	 * Returns the count of entities associated with this ORM configuration.
	 */
	int entitiesSize();

	/**
	 * Returns the current build version of the product.
	 *
	 * @return The product's current version
	 */
	String getBuildVersion();

	/**
	 * Returns the catalog name to be applied to all entities in this ORM xml if applicable.
	 */
	String getCatalogName();

	/**
	 * Returns the {@link TextRange} of the default catalog name value.
	 */
	TextRange getCatalogNameTextRange();

	/**
	 * Returns the description noted in the ORM.xml.
	 */
	String getDescription();

	/**
	 * Returns the document type of this ORM configuration.
	 */
	ORMDocumentType getDocumentType();

	/**
	 * Returns the embeddable at the given position.
	 */
	ExternalEmbeddable getEmbeddable(int index);

	/**
	 * Returns the embeddable with the given class name.
	 */
	ExternalEmbeddable getEmbeddable(String className);

	/**
	 * Returns the entity associated with this ORM configuration identified by the given index.
	 */
	ExternalEntity getEntity(int index);

	/**
	 * Returns the entity associated with this ORM configuration identified by the given class name.
	 */
	ExternalEntity getEntity(String className);

	/**
	 * Returns the location on the file system of the ORM configuration file.
	 *
	 * @return The location on the file system of the ORM configuration file
	 */
	URL getLocation();

	/**
	 * Returns the mapped superclass at the given position.
	 */
	ExternalMappedSuperClass getMappedSuperClass(int index);

	/**
	 * Returns the mapped superclass for the given class name.
	 */
	ExternalMappedSuperClass getMappedSuperClass(String className);

	/**
	 * Returns a native named query at the given position.
	 */
	ExternalNativeQuery getNamedNativeQuery(int index);

	/**
	 * Returns the native SQL query with the given name.
	 */
	ExternalNativeQuery getNamedNativeQuery(String queryName);

	/**
	 * Returns the JPQL query at the given position.
	 */
	ExternalBasicNamedQuery getNamedQuery(int index);

	/**
	 * Returns the JPQL query with the given name.
	 */
	ExternalBasicNamedQuery getNamedQuery(String queryName);

	/**
	 * Returns the stored procedure query at the position.
	 */
	ExternalNamedStoredProcedureQuery getNamedStoredProcedureQuery(int index);

	/**
	 * Returns the default package name to be applied to all entities.
	 */
	String getPackageName();

	/**
	 * Returns the {@link TextRange} of the default package name value.
	 */
	TextRange getPackageNameTextRange();

	/**
	 * Returns the persistent unit representing this ORM's persistence unit metadata.
	 */
	ExternalPersistenceUnit getPersistenceUnitMetaData();

	/**
	 * Returns the {@link TextRange} of the root element's name.
	 */
	TextRange getRootElementTextRange();

	/**
	 * Returns the default schema name to be applied to all referenced tables.
	 */
	String getSchemaName();

	/**
	 * Returns the {@link TextRange} of the default schema name value.
	 */
	TextRange getSchemaNameTextRange();

	/**
	 * Returns the sequence generator with the given name.
	 */
	ExternalSequenceGenerator getSequenceGenerator(int index);

	/**
	 * Returns the source root of this ORM.xml.
	 */
	URL getSourceRoot();

	/**
	 * Returns the SQL Result Set Mapping at the given position.
	 */
	ExternalSQLResultSetMapping getSqlResultSetMapping(int index);

	/**
	 * Returns the table generator at the given position.
	 */
	ExternalTableGenerator getTableGenerator(int index);

	/**
	 * Derives and returns the XML representation of this configuration.
	 */
	String getXML();

	/**
	 * Determines whether the persistence unit metadata is defined or not.
	 */
	boolean hasPersistenceUnitMetaData();

	/**
	 * Returns a list of all mapped super class entities associated with this ORM configuration.
	 */
	List<ExternalMappedSuperClass> mappedSuperClasses();

	/**
	 * Returns the count of mapped super class entities associated with this ORM configuration.
	 */
	int mappedSuperClassesSize();

	/**
	 * Returns a list of all native queries declared globally for this ORM configuration.
	 */
	List<ExternalNativeQuery> namedNativeQueries();

	/**
	 * Returns the count of all native queries declared globally for this ORM configuration.
	 */
	int namedNativeQueriesSize();

	/**
	 * Returns a list of all named queries declared globally for this ORM configuration.
	 */
	List<ExternalBasicNamedQuery> namedQueries();

	/**
	 * Returns the count of all named queries declared globally for this ORM configuration.
	 */
	int namedQueriesSize();

	/**
	 * Removes the embedded at the given position.
	 */
	void removeEmbeddable(int index);

	/**
	 * Removes the entity for the given position.
	 */
	void removeEntity(int index);

	/**
	 * Removes the entity with the given class name.
	 */
	void removeEntity(String entityClassName);

	/**
	 * Removes the mapped super class at the given position.
	 */
	void removeMappedSuperClass(int index);

	/**
	 * Removes the JPQL query at the given position.
	 */
	void removeNamedQuery(int index);

	/**
	 * Removes the JPQL query with the given name.
	 */
	void removeNamedQuery(String queryName);

	/**
	 * Removes the native query at the given position.
	 */
	void removeNativeQuery(int index);

	/**
	 * Removes the persistence unit representing this ORM's persistence unit metadata.
	 */
	void removePersistenceUnitMetaData();

	/**
	 * Removes the sequence generator at the given position.
	 */
	void removeSequenceGenerator(int index);

	/**
	 * Removes the SQL result set mapping at the given position.
	 */
	void removeSqlResultSetMapping(int index);

	/**
	 * Removes the stored procedure query at the given position.
	 */
	void removeStoredProcedureQuery(int index);

	/**
	 * Removes the table generator at the given position.
	 */
	void removeTableGenerator(int index);

	/**
	 * Returns a list of all sequence generators defined globally for this ORM configuration.
	 */
	List<ExternalSequenceGenerator> sequenceGenerators();

	/**
	 * Returns the count of all sequence generators defined globally for this ORM configuration.
	 */
	int sequenceGeneratorsSize();

	/**
	 * Set the catalog name.
	 */
	void setCatalogName(String catalog);

	/**
	 * Set the description
	 */
	void setDescription(String description);

	/**
	 * Sets the type of this ORM.
	 */
	void setDocumentType(ORMDocumentType type);

	/**
	 * Set the default package name to be applied to all entities.
	 */
	void setPackageName(String packageName);

	/**
	 * Set the default schema name to be applied to all referenced tables.
	 */
	void setSchemaName(String schema);

	/**
	 * Returns a list of all SQL result set mappings declared globally for this ORM configuration.
	 */
	List<ExternalSQLResultSetMapping> sqlResultSetMappings();

	/**
	 * Returns the count of all SQL result set mappings declared globally for this ORM configuration.
	 */
	int sqlResultSetMappingsSize();

	/**
	 * Returns a list of named stored procedure queries defined for this entity.
	 */
	List<ExternalNamedStoredProcedureQuery> storedProcedureQueries();

	/**
	 * Returns the count of named stored procedure queried defined for this entity
	 */
	int storedProcedureQueriesSize();

	/**
	 * Returns a list of all table generators defined globally for this ORM configuration.
	 */
	List<ExternalTableGenerator> tableGenerators();

	/**
	 * Returns the count of all table generators defined globally for this ORM configuration.
	 */
	int tableGeneratorsSize();
}