/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicColumn;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form of a basic column.
 *
 * @version 2.6
 */
abstract class AbstractColumn extends AbstractExternalForm
                              implements ExternalBasicColumn {

	/**
	 * Creates a new <code>Column</code>.
	 *
	 * @param parent The parent of this external form
	 */
	AbstractColumn(AbstractExternalForm parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(UNIQUE);
		names.add(NULLABLE);
		names.add(INSERTABLE);
		names.add(UPDATABLE);
		names.add(COLUMN_DEFINITION);
		names.add(TABLE);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getColumnDefinition() {
		return getAttribute(COLUMN_DEFINITION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getColumnDefinitionTextRange() {
		return getAttributeTextRange(COLUMN_DEFINITION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getColumnTextRange() {
		return getTextNodeTextRange(getElement());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getInstertableTextRange() {
		return getAttributeTextRange(INSERTABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getNameTextRange() {
		return getAttributeTextRange(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getNullableTextRange() {
		return getAttributeTextRange(NULLABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getTable() {
		return getAttribute(TABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getTableTextRange() {
		return getAttributeTextRange(TABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getUniqueTextRange() {
		return getAttributeTextRange(UNIQUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getUpdatableTextRange() {
		return getAttributeTextRange(UPDATABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Boolean isInstertable() {
		return getBooleanAttribute(INSERTABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Boolean isNullable() {
		return getBooleanAttribute(NULLABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Boolean isUnique() {
		return getBooleanAttribute(UNIQUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Boolean isUpdatable() {
		return getBooleanAttribute(UPDATABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setColumnDefinition(String columnDefinition) {

		setAttribute(COLUMN_DEFINITION, columnDefinition);

		if (shouldRemoveSelf()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setInsertable(Boolean insertable) {

		setAttribute(INSERTABLE, insertable);

		if (shouldRemoveSelf()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setName(String name) {

		setAttribute(NAME, name);

		if (shouldRemoveSelf()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setNullable(Boolean nullable) {

		setAttribute(NULLABLE, nullable);

		if (shouldRemoveSelf()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setTable(String table) {

		setAttribute(TABLE, table);

		if (shouldRemoveSelf()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setUnique(Boolean unique) {

		setAttribute(UNIQUE, unique);

		if (shouldRemoveSelf()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setUpdatable(Boolean updatable) {

		setAttribute(UPDATABLE, updatable);

		if (shouldRemoveSelf()) {
			removeSelf();
		}
	}

	/**
	 * Determines whether this element should be removed when it has no children and no attributes.
	 *
	 * @return <code>true</code> to remove this element; <code>false</code> to not remove it
	 */
	boolean shouldRemoveEmptyElement() {
		return false;
	}

	private boolean shouldRemoveSelf() {
		return shouldRemoveEmptyElement() && !hasAnything();
	}
}