/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalClassConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalConvertibleMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalObjectTypeConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalStructConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalTypeConverter;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form for a convertible mapping, which is a child of an entity.
 *
 * @see Embeddable
 *
 * @version 2.6
 */
abstract class ConvertibleMapping extends NonTransientMapping
                                  implements ExternalConvertibleMapping {

	/**
	 * Creates a new <code>ConvertibleMapping</code>.
	 *
	 * @param parent The parent of this external form
	 */
	ConvertibleMapping(Embeddable parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalClassConverter addConverter() {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		ClassConverter converter = buildConverter();
		converter.addSelf();
		return converter;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalObjectTypeConverter addObjectTypeConverter() {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ClassConverter.CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		ObjectTypeConverter converter = buildObjectTypeConverter();
		converter.addSelf();
		return converter;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalStructConverter addStructConverter() {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(ClassConverter.CONVERTER);

		StructConverter converter = buildStructConverter();
		converter.addSelf();
		return converter;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalTypeConverter addTypeConverter() {

		removeChild(ClassConverter.CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		TypeConverter converter = buildTypeConverter();
		converter.addSelf();
		return converter;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(ACCESS);
		return names;
	}

	private ClassConverter buildConverter() {
		return new ClassConverter(this, -1);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(CONVERT);
		names.add(ClassConverter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private ObjectTypeConverter buildObjectTypeConverter() {
		return new ObjectTypeConverter(this, -1);
	}

	private StructConverter buildStructConverter() {
		return new StructConverter(this, -1);
	}

	private TypeConverter buildTypeConverter() {
		return new TypeConverter(this, -1);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getConvert() {
		return getChildTextNode(CONVERT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalClassConverter getConverter() {

		if (hasChild(ClassConverter.CONVERTER)) {
			return buildConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getConvertTextRange() {
		return getChildTextNodeTextRange(CONVERT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalObjectTypeConverter getObjectTypeConverter() {

		if (hasChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER)) {
			return buildObjectTypeConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalStructConverter getStructConverter() {

		if (hasChild(StructConverter.STRUCT_CONVERTER)) {
			return buildStructConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalTypeConverter getTypeConverter() {

		if (hasChild(TypeConverter.TYPE_CONVERTER)) {
			return buildTypeConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeConverter() {
		removeChild(ClassConverter.CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeObjectTypeConverter() {
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeStructConverter() {
		removeChild(StructConverter.STRUCT_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeTypeConverter() {
		removeChild(TypeConverter.TYPE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setConvert(String value) {
		updateChildTextNode(CONVERT, value);
	}
}