/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.eclipse.persistence.tools.mapping.DefaultFormHelper;
import org.eclipse.persistence.tools.mapping.orm.ExternalORMRepository;
import org.eclipse.persistence.tools.mapping.orm.ORMDocumentType;
import org.eclipse.persistence.tools.utility.XMLTools;
import org.eclipse.persistence.tools.utility.iterable.EmptyIterable;
import org.w3c.dom.Document;

/**
 * Defines the concrete implementation of {@link ExternalORMRepository}. This ORM repository knows
 * how to build an ORM meta data from the scope of a Project.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public final class ORMRepository implements ExternalORMRepository {

	/**
	 * Creates a new <code>ORMRepository</code>.
	 */
	public ORMRepository() {
		super();
	}

	private DocumentBuilder buildDocumentBuilder() throws ParserConfigurationException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setValidating(false);
		factory.setNamespaceAware(false);
		return factory.newDocumentBuilder();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ORMConfiguration buildORMConfiguration(URL location, ORMDocumentType docType) throws IOException {
		DefaultFormHelper helper = new DefaultFormHelper(XMLTools.newDocument(), location);
		ORMConfiguration ormConfiguration = new ORMConfiguration(helper);
		ormConfiguration.setDocumentType(docType);
		return ormConfiguration;
	}

	public ORMConfiguration getORMConfiguration(String ormXML) {

		try {

			Document document = buildDocumentBuilder().parse(
				new ByteArrayInputStream(ormXML.getBytes("utf-8"))
			);

			DefaultFormHelper helper = new DefaultFormHelper(document, null);
			return new ORMConfiguration(helper);
		}
		catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	@SuppressWarnings("resource")
	public ORMConfiguration getORMConfiguration(URL location) {

		InputStream inputStream = null;

		try {

			inputStream = location.openStream();
			Document document = buildDocumentBuilder().parse(inputStream);

			DefaultFormHelper helper = new DefaultFormHelper(document, location);
			return new ORMConfiguration(helper);
		}
		catch (Exception e) {
			e.printStackTrace();
			return null;
		}
		finally {
			try {
				if (inputStream != null) {
					inputStream.close();
				}
			}
			catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<URL> ormConfigurationLocations() {
		return EmptyIterable.instance();
	}
}