/*******************************************************************************
 * Copyright (c) 1998, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation from Oracle TopLink
******************************************************************************/
package org.eclipse.persistence.tools.db.model.spi.jdbc;

import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import org.eclipse.persistence.tools.db.model.spi.ExternalTable;
import org.eclipse.persistence.tools.db.model.spi.ExternalTableDescription;
import org.eclipse.persistence.tools.utility.NameTools;
import org.eclipse.persistence.tools.utility.ObjectTools;

/**
 * @version 2.6
 */
final class JDBCExternalTableDescription implements ExternalTableDescription {
	private final JDBCExternalDatabase database;
	private final String catalogName;
	private final String schemaName;
	private final String name;
	private final String qualifiedName;	// cache the qualified name, since it will not change
	private ExternalTable externalTable;		// pseudo-final


	// ********** constructor/initialization **********

	/**
	 * Construct an external table description from the current row
	 * in the specified result set. The result set corresponds to the
	 * result set returned from DatabaseMetaData#getTables(String, String, String, String[]).
	 * @see java.sql.DatabaseMetaData#getTables(String, String, String, String[])
	 */
	JDBCExternalTableDescription(ResultSet resultSet, JDBCExternalDatabase database) throws SQLException {
		this(
			// * Returns fixed-length strings;
			// these calls *shouldn't* throw any SQL exceptions...
			trim(resultSet.getString(1)),		// TABLE_CAT
			trim(resultSet.getString(2)),		// TABLE_SCHEM
			trim(resultSet.getString(3)),		// TABLE_NAME
			database
		);
	}

	JDBCExternalTableDescription(String catalogName, String schemaName, String name, JDBCExternalDatabase database) {
		super();
		this.catalogName = catalogName;
		this.schemaName = schemaName;
		this.name = name;
		this.qualifiedName = this.buildQualifiedName();
		this.database = database;
	}

	/**
	 * trim down the specified string, to null if necessary
	 */
	private static String trim(String s) {
		if (s == null) {
			return null;
		}
		s = s.trim();
		return (s.length() == 0) ? null : s;
	}

	private String buildQualifiedName() {
		return NameTools.buildQualifiedName(this.catalogName, this.schemaName, this.name);
	}


	// ********** ExternalTableDescription implementation **********

	/**
	 * @see org.eclipse.persistence.tools.db.model.spi.ExternalTableDescription#getCatalogName()
	 */
	@Override
	public String getCatalogName() {
		return this.catalogName;
	}

	/**
	 * @see org.eclipse.persistence.tools.db.model.spi.ExternalTableDescription#getSchemaName()
	 */
	@Override
	public String getSchemaName() {
		return this.schemaName;
	}

	/**
	 * @see org.eclipse.persistence.tools.db.model.spi.ExternalTableDescription#getName()
	 */
	@Override
	public String getName() {
		return this.name;
	}

	/**
	 * @see org.eclipse.persistence.tools.db.model.spi.ExternalTableDescription#getQualifiedName()
	 */
	@Override
	public String getQualifiedName() {
		return this.qualifiedName;
	}

	/**
	 * @see org.eclipse.persistence.tools.db.model.spi.ExternalTableDescription#getTable()
	 */
	@Override
	public ExternalTable getTable() {
		if (this.externalTable == null) {
			this.externalTable = this.buildExternalTable();
		}
		return this.externalTable;
	}

	/**
	 * @see org.eclipse.persistence.tools.db.model.spi.TableDescription#getAdditionalInfo()
	 */
	@Override
	public String getAdditionalInfo() {
		try {
			return this.metaData().getURL();
		} catch (SQLException ex) {
			return null;
		}
	}


	// ********** behavior **********

	private ExternalTable buildExternalTable() {
		try {
			return new JDBCExternalTable(this);
		} catch (SQLException ex) {
			throw new RuntimeException(ex);
		}
	}


	// ********** queries **********

	DatabaseMetaData metaData() {
		return this.database.metaData();
	}

	JDBCExternalDatabase getDatabase() {
		return this.database;
	}

	@Override
	public String toString() {
		return ObjectTools.toString(this, this.qualifiedName);
	}
}