/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.tests;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.persistence.tools.mapping.ExternalForm;
import org.eclipse.persistence.tools.utility.ClassTools;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.junit.Test;
import org.w3c.dom.Node;
import static org.junit.Assert.*;

/**
 * This test defines the behavior to test the manipulation of an XML document through the Mapping SPI.
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public abstract class AbstractExternalFormTests<FORM extends ExternalForm> {

	/**
	 * Populates the given tester with the appropriate {@link NodeTester}.
	 *
	 * @param tester The {@link RootNodeTester} is used to adds the appropriate testers to test the
	 * property of a node; i.e. its attributes and child nodes
	 */
	protected abstract void populate(RootNodeTester tester);

	/**
	 * Performs the actual tests on the XML document for a particular node that is being manipulated
	 * by an {@link ExternalForm}.
	 */
	@Test
	public final void test() throws Exception {
		RootNodeTester tester = new RootNodeTester();
		populate(tester);
		tester.test();
	}

	/**
	 * The runner manages testing a single child node.
	 */
	private abstract class AbstractChildNodeTesterRunner extends NodeTesterRunner {

		/**
		 * Indicates whether the child node being tested is present or not.
		 */
		boolean hasChild;

		/**
		 * Creates a new <code>AbstractChildNodeTesterRunner</code>.
		 *
		 * @param tester This object defines a single child node to test
		 */
		AbstractChildNodeTesterRunner(NodeTester<FORM, ?> tester) {
			super(tester);
		}

		final String displayString() {
			return "<" + getNodeName() + ">";
		}

		final Node getChildNode(Node node) {

			node = node.getFirstChild();

			while (node != null) {

				if (getNodeName().equals(node.getNodeName())) {
					return node;
				}

				node = node.getNextSibling();
			}

			return null;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		boolean isMultipleSupported() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		boolean isNodeDeletedWithNullValue() {
			return false;
		}
	}

	/**
	 * This controller simply asks its runner to test adding the attribute node.
	 */
	private class AttributeNodeRunnerAddingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void log(Logger logger) {
			AttributeNodeTesterRunner runner = (AttributeNodeTesterRunner) this.runner;
			logger.log(Level.FINE, "<" + runner.parentNodeName + " " + runner.getNodeName() + "=\"" + runner.propertyTester().getDefaultValue() + "\"> : adding");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			AttributeNodeTesterRunner runner = (AttributeNodeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testAdding(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This controller simply asks its runner to test reading the attribute node.
	 */
	private class AttributeNodeRunnerReadingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void log(Logger logger) {
			AttributeNodeTesterRunner runner = (AttributeNodeTesterRunner) this.runner;
			logger.log(Level.FINE, "<" + runner.parentNodeName + " " + runner.getNodeName() + "=\"" + runner.propertyTester().getDefaultValue() + "\"> : reading");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			AttributeNodeTesterRunner runner = (AttributeNodeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testReading(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This controller simply asks its runner to test removing the attribute node.
	 */
	private class AttributeNodeRunnerRemovingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void log(Logger logger) {
			AttributeNodeTesterRunner runner = (AttributeNodeTesterRunner) this.runner;
			logger.log(Level.FINE, "<" + runner.parentNodeName + " " + runner.getNodeName() + "=\"" + runner.propertyTester().getDefaultValue() + "\"> : removing");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			AttributeNodeTesterRunner runner = (AttributeNodeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testRemoving(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * An <code>AttributeNodeTester</code> tests manipulating an element's attribute.
	 * <p>
	 * <div nowrap>Form: <code><b>&lt;nodeName attributeName="value"/&gt;</b></div>
	 */
	public interface AttributeNodeTester<FORM extends ExternalForm, VALUE> extends PropertyTester<FORM, VALUE> {
	}

	/**
	 * The runner associated with {@link AttributeNodeTester}.
	 *
	 * <div nowrap>Form: <code><b>&lt;node_name attribute="value"/&gt</b></code>.</div>
	 */
	private class AttributeNodeTesterRunner extends PropertyNodeTesterRunner {

		/**
		 * Creates a new <code>AttributeNodeTesterRunner</code>.
		 *
		 * @param tester This object defines a single node to test an attribute node
		 */
		AttributeNodeTesterRunner(AttributeNodeTester<FORM, ?> tester) {
			super(tester);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		String displayString() {
			return "<" + parentNodeName + " " + getNodeName() + "=\"" + propertyTester().getDefaultValue() + "\">";
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		int getChildrenCount(Node parentNode) {
			return parentNode.getAttributes().getLength();
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		Node getNode(Node parentNode) {
			return parentNode.getAttributes().getNamedItem(getNodeName());
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		String getNodeValue(Node node) {
			return node.getTextContent();
		}
	}

	/**
	 * This controller simply asks its runner to test adding or removing a child node.
	 */
	private class BooleanChildNodeRunnerAddingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		void log(Logger logger) {
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.tester.getNodeName() + "> : addition");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			BooleanChildNodeTesterRunner runner = (BooleanChildNodeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testAdding(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This controller simply asks its runner to test reading if a child node is present or not.
	 */
	private class BooleanChildNodeRunnerReadingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		void log(Logger logger) {
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.tester.getNodeName() + "> : reading");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			BooleanChildNodeTesterRunner runner = (BooleanChildNodeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testReading(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This controller simply asks its runner to test removing the child node.
	 */
	private class BooleanChildNodeRunnerRemovingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		void log(Logger logger) {
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.tester.getNodeName() + "> : removing");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			BooleanChildNodeTesterRunner runner = (BooleanChildNodeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testRemoving(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This <code>BooleanChildNodeTester</code> tests when a form add or remove a single child node
	 * that is represented by a boolean property in the {@link ExternalForm}.
	 */
	public interface BooleanChildNodeTester<FORM extends ExternalForm> extends NodeTester<FORM, Boolean> {

		/**
		 * Retrieves the {@link ExternalForm} representing the single child node.
		 *
		 * @param form The external form being tested
		 * @return The {@link ExternalForm} representing the child node
		 */
		boolean getValue(FORM form);

		/**
		 * Adds the single child node to the node represented by the given form.
		 *
		 * @param form The external form being tested
		 */
		void setValue(FORM form, boolean value);
	}

	/**
	 * The runner associated with {@link BooleanChildNodeTester}.
	 */
	private class BooleanChildNodeTesterRunner extends AbstractChildNodeTesterRunner {

		/**
		 * Creates a new <code>BooleanChildNodeTesterRunner</code>.
		 *
		 * @param tester This object defines a single child node to test
		 */
		BooleanChildNodeTesterRunner(BooleanChildNodeTester<FORM> tester) {
			super(tester);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void testAdding(ExternalFormHolder holder) {

			// Don't add the child if it's present
			if (hasChild) {
				return;
			}

			BooleanChildNodeTester<FORM> tester = (BooleanChildNodeTester<FORM>) ((NodeTester<FORM, ?>) this.tester);

			// Add the child node
			tester.setValue(holder.form, true);
			hasChild = true;
			currentChildrenCount++;

			// Make sure the child node was indeed added
			assertEquals(
				displayString() + " : The number of children is inconsistent",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			// Make sure the child node is correctly retrieved
			assertTrue(
				displayString() + " : The child node should have been found",
				tester.getValue(holder.form)
			);

			// Make sure the right child node was added and is retrieved by the ExternalForm
			assertNotNull(
				displayString() + " : The child node should exist",
				getChildNode(holder.node)
			);

			// Make sure nothing changed
			assertEquals(
				displayString() + " : The child node should exist",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void testInitialState(ExternalFormHolder holder) {

			BooleanChildNodeTester<FORM> tester = (BooleanChildNodeTester<FORM>) ((NodeTester<FORM, ?>) this.tester);

			// Make sure the child node does not exist
			assertEquals(
				displayString() + " : The child node should not exist",
				hasChild,
				tester.getValue(holder.form)
			);

			// Make sure nothing changed
			assertEquals(
				displayString() + " : The child node should not exist",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			// The child node should not exist
			if (hasChild) {
				assertNotNull(
					displayString() + " : The child node should exist",
					getChildNode(holder.node)
				);
			}
			else {
				assertNull(
					displayString() + " : The child node should not exist",
					getChildNode(holder.node)
				);
			}

			// Make sure nothing changed
			assertEquals(
				displayString() + " : The child node should not exist",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testReading(ExternalFormHolder holder) {
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void testRemoving(ExternalFormHolder holder) {

			// Nothing to remove
			if (!hasChild) {
				return;
			}

			BooleanChildNodeTester<FORM> tester = (BooleanChildNodeTester<FORM>) ((NodeTester<FORM, ?>) this.tester);

			// Make sure the child node exists
			assertTrue(
				displayString() + " : The child node should exist",
				tester.getValue(holder.form)
			);

			// Make sure nothing changed with the previous check
			assertEquals(
				displayString() + " : The child node should exist",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			// Remove the child node
			tester.setValue(holder.form, false);
			currentChildrenCount--;
			hasChild = false;

			// Make sure the child node was indeed removed
			assertEquals(
				displayString() + " : The child node should have been removed",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			// Make sure the right child node was removed
			assertNull(
				displayString() + " : The wrong child node was removed",
				getChildNode(holder.node)
			);

			// Test to make sure the child node is not found
			assertFalse(
				displayString() + " : The child node should not exist",
				tester.getValue(holder.form)
			);
		}
	}

	/**
	 * This controller simply asks its runner to add the attribute node.
	 */
	private class ChildAttributeRunnerAddingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void log(Logger logger) {
			ChildAttributeTesterRunner runner = (ChildAttributeTesterRunner) this.runner;
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.getChildNodeName() + " " + runner.tester.getNodeName() + "=\"\">> : adding");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ChildAttributeTesterRunner runner = (ChildAttributeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testAdding(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This controller simply asks its runner to reading the attribute node.
	 */
	private class ChildAttributeRunnerReadingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void log(Logger logger) {
			ChildAttributeTesterRunner runner = (ChildAttributeTesterRunner) this.runner;
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.getChildNodeName() + " " + runner.tester.getNodeName() + "=\"\">> : reading");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ChildAttributeTesterRunner runner = (ChildAttributeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testReading(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This controller simply asks its runner to removing the attribute node.
	 */
	private class ChildAttributeRunnerRemovingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void log(Logger logger) {
			ChildAttributeTesterRunner runner = (ChildAttributeTesterRunner) this.runner;
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.getChildNodeName() + " " + runner.tester.getNodeName() + "=\"\">> : removing");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ChildAttributeTesterRunner runner = (ChildAttributeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testRemoving(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * An <code>AttributeNodeTester</code> tests manipulating an element's attribute.
	 * <p>
	 * <div nowrap>Form: <code><b>&lt;nodeName attributeName="value"/&gt;</b></div>
	 */
	public interface ChildAttributeTester<FORM extends ExternalForm, VALUE> extends PropertyTester<FORM, VALUE> {

		/**
		 *
		 */
		String getChildNodeName();
	}

	private class ChildAttributeTesterRunner extends NodeTesterRunner {

		/**
		 * Keeps track of the status of the node's existence.
		 */
		private boolean nodeExists;

		/**
		 * Creates a new <code>ChildAttributeTesterRunner</code>.
		 *
		 * @param tester This object defines a single node to test a child text node
		 */
		ChildAttributeTesterRunner(ChildAttributeTester<FORM, ?> tester) {
			super(tester);
		}

		private String displayString() {
			return "<" + parentNodeName + "><" + getChildNodeName() + " " + tester.getNodeName() + "=\"\">";
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		String getActualNodeName() {
			return getChildNodeName();
		}

		private Node getChildNode(Node node) {

			node = node.getFirstChild();

			while (node != null) {

				if (getChildNodeName().equals(node.getNodeName())) {
					return node;
				}

				node = node.getNextSibling();
			}

			return null;
		}

		private String getChildNodeName() {
			return ((ChildAttributeTester<FORM, ?>) tester).getChildNodeName();
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		boolean isMultipleSupported() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		boolean isNodeDeletedWithNullValue() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testAdding(ExternalFormHolder holder) {

			ChildAttributeTester<FORM, Object> tester = (ChildAttributeTester<FORM, Object>) this.tester;

			//
			// Test 1
			//
			// Change the node value to null
			tester.setValue(holder.form, null);

			if (nodeExists) {
				currentChildrenCount--;
				nodeExists = false;
			}

			assertEquals(
				displayString() + " : The element should not have any children",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			// Make sure the value retrieved is null
			Object result = tester.getValue(holder.form);

			assertNull(
				displayString() + " : The element's value should be null",
				result
			);

			// Make sure nothing changed
			assertEquals(
				displayString() + " : The element should not have any children",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			//
			// Test 2
			//
			// Change the value to something
			Object expectedValue1 = tester.getValue1();

			assertNotNull(
				displayString() + " : Value 1 cannot be null",
				expectedValue1
			);

			tester.setValue(holder.form, expectedValue1);

			if (!nodeExists) {
				nodeExists = true;
				currentChildrenCount++;
			}

			// The child node should have been added
			assertEquals(
				displayString() + " : The number of children is inconsistent",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			Node childNode = getChildNode(holder.node);

			assertNotNull(
				displayString() + " : The child node was not added correctly",
				childNode
			);

			assertEquals(
				displayString() + " : The number of attributes is inconsistent",
				1,
				childNode.getAttributes().getLength()
			);

			// Test the attribute's value
			Node attribute = childNode.getAttributes().getNamedItem(tester.getNodeName());

			assertNotNull(
				displayString() + " : The attribute was not added correctly",
				attribute
			);

			String stringExpectedValue = tester.toString(expectedValue1);

			assertEquals(
				displayString() + " : The attribute's value was not set correctly",
				stringExpectedValue,
				attribute.getNodeValue()
			);

			// Get the value
			result = tester.getValue(holder.form);

			assertNotNull(
				displayString() + " : The element's value was not set correctly",
				result
			);

			// Get the value directly
			String stringResult = tester.toString(result);

			assertEquals(
				displayString() + " : The value was not set correctly",
				stringExpectedValue,
				stringResult
			);

			//
			// Test 3
			//
			// Change the value to something else
			Object expectedValue2 = tester.getValue2();

			assertNotNull(
				displayString() + " : Value 2 cannot be null",
				expectedValue2
			);

			assertNotSame(
				displayString() + " : Value 1 and value 2 cannot be the same",
				expectedValue1,
				expectedValue2
			);

			tester.setValue(holder.form, expectedValue2);

			// The number of children should not have changed
			assertEquals(
				displayString() + " : The number of children is inconsistent",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			childNode = getChildNode(holder.node);

			assertNotNull(
				displayString() + " : The child node was not added correctly",
				childNode
			);

			assertEquals(
				displayString() + " : The number of attributes is inconsistent",
				1,
				childNode.getAttributes().getLength()
			);

			// Test the attribute's value
			attribute = childNode.getAttributes().getNamedItem(tester.getNodeName());

			assertNotNull(
				displayString() + " : The attribute was not added correctly",
				attribute
			);

			stringExpectedValue = tester.toString(expectedValue2);

			assertEquals(
				displayString() + " : The attribute's value was not set correctly",
				stringExpectedValue,
				attribute.getNodeValue()
			);

			// Get the value
			result = tester.getValue(holder.form);

			assertNotNull(
				displayString() + " : The element's value was not set correctly",
				result
			);

			// Get the value directly
			stringResult = tester.toString(result);

			assertEquals(
				displayString() + " : The value was not set correctly",
				stringExpectedValue,
				stringResult
			);

			//
			// Complete
			//
			// Change the value back to its original value
			Object defaultValue = tester.getDefaultValue();

			assertNotNull(
				displayString() + " : The default value cannot be null",
				defaultValue
			);

			tester.setValue(holder.form, defaultValue);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testInitialState(ExternalFormHolder holder) {

			ChildAttributeTester<FORM, Object> tester = (ChildAttributeTester<FORM, Object>) this.tester;
			nodeExists |= tester.doesNodeAlreadyExist();

			// Node name
			String nodeName = getNodeName();

			assertNotNull(
				displayString() + " : The child node's attribute name cannot be null",
				nodeName
			);

			// Child node name
			String childNodeName = getChildNodeName();

			assertNotNull(
				displayString() + " : The child node name cannot be null",
				childNodeName
			);

			// Test the initial state of the parent node
			assertEquals(
				displayString() + " : The number of children is incorrect",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			// The child node should either not exist or already being present
			Node childNode = getChildNode(holder.node);

			if (!nodeExists) {
				assertNull(
					displayString() + " : The node should be null",
					childNode
				);
			}
			else {
				assertNotNull(
					displayString() + " : The node should not be null",
					childNode
				);

				Node attribute = childNode.getAttributes().getNamedItem(tester.getNodeName());

				assertNotNull(
					displayString() + " : The attribute was not added correctly",
					attribute
				);

				Object defaultValue = tester.getDefaultValue();

				assertNotNull(
					displayString() + " : The default value cannot be null",
					defaultValue
				);

				String stringDefaultValue = tester.toString(defaultValue);

				assertEquals(
					displayString() + " : The attribute's value was not set correctly",
					stringDefaultValue,
					attribute.getNodeValue()
				);
			}
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testReading(ExternalFormHolder holder) {

			ChildAttributeTester<FORM, Object> tester = (ChildAttributeTester<FORM, Object>) this.tester;

			// The node value should either be null or not null
			Object result = tester.getValue(holder.form);

			// Make sure the node's value is not null
			if (nodeExists) {

				assertNotNull(
					displayString() + " : The element's value should not be null",
					result
				);

				assertSame(
					displayString() + " : The element's value was not retrived correctly",
					tester.getDefaultValue(),
					result
				);
			}
			// The node is not present, make sure reading its value returns null
			else {
				assertNull(
					displayString() + " : The element's value should be null",
					result
				);
			}

			// Make sure nothing changed by reading the node's value
			assertEquals(
				displayString() + " : The element should not have any children",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			// Retrieve the actual node
			Node childNode = getChildNode(holder.node);

			if (nodeExists) {

				assertNotNull(
					displayString() + " : The node should not be null",
					childNode
				);

				Node attribute = childNode.getAttributes().getNamedItem(tester.getNodeName());

				assertNotNull(
					displayString() + " : The attribute should not be null",
					attribute
				);
			}
			else {
				assertNull(
					displayString() + " : The node should be null",
					childNode
				);
			}

			// Make sure nothing changed after retrieving the node
			assertEquals(
				displayString() + " : The element should not have any children",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testRemoving(ExternalFormHolder holder) {

			PropertyTester<FORM, Object> tester = (PropertyTester<FORM, Object>) this.tester;

			// Change the node value to null, which will remove it from the parent node
			tester.setValue(holder.form, null);

			if (nodeExists) {
				nodeExists = false;
				currentChildrenCount--;
			}

			assertEquals(
				displayString() + " : The number of children is inconsistent",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);
		}
	}

	/**
	 * This controller tests adding a random portion of child nodes.
	 */
	private class ChildListNodeRunnerAddingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		void log(Logger logger) {
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.tester.getNodeName() + "*> : addition");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ChildListNodeTesterRunner runner = (ChildListNodeTesterRunner) this.runner;

			// Create a random range within the list of values used to create the child nodes
			Random random = new Random();
			int count = runner.getDefaultChildrenCount();

			int startIndex = random.nextInt(count);
			int endIndex = random.nextInt(count);

			// Make sure the start and end indices are not the same
			while (endIndex == startIndex) {
				endIndex = random.nextInt(count);
			}

			runner.currentChildrenCount = currentChildrenCount;
			runner.addRange(startIndex, endIndex);
			runner.rangeIndex = runner.ranges.size() - 1;
			runner.testInitialState(holder);
			runner.testAdding(holder);
			runner.testReading(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This controller tests reading a random portion of child nodes.
	 */
	private class ChildListNodeRunnerReadingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		void log(Logger logger) {
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.tester.getNodeName() + "*> : reading");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ChildListNodeTesterRunner runner = (ChildListNodeTesterRunner) this.runner;
			int rangesCount = runner.ranges.size();

			// Nothing to read
			if (rangesCount == 0) {
				return;
			}

			// Read a range randomly
			Random random = new Random();
			int rangeIndex = random.nextInt(rangesCount);
			runner.rangeIndex = rangeIndex;

			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testReading(holder);
		}
	}

	/**
	 * This controller tests removing a random portion of child nodes.
	 */
	private class ChildListNodeRunnerRemovingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		void log(Logger logger) {
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.tester.getNodeName() + "*> : removal");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ChildListNodeTesterRunner runner = (ChildListNodeTesterRunner) this.runner;
			int rangesCount = runner.ranges.size();

			// Nothing to read
			if (rangesCount == 0) {
				return;
			}

			// Read a range randomly
			Random random = new Random();
			int rangeIndex = random.nextInt(rangesCount);
			runner.rangeIndex = rangeIndex;

			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testRemoving(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This <code>ChildListNodeTester</code> tests a form when the node it represents can have zero
	 * or many child nodes of the same type (i.e. with the same node name).
	 */
	public interface ChildListNodeTester<FORM extends ExternalForm, CHILD_FORM, CHILD_VALUE> extends NodeTester<FORM, CHILD_FORM> {

		/**
		 * Adds a new child node to the node represented by the given form.
		 *
		 * @param form The external form being tested
		 * @param value The value used when creating the child node
		 * @return The external form representing the child node that was added to the node represented
		 * by the given form
		 */
		CHILD_FORM addChild(FORM form, CHILD_VALUE value);

		/**
		 * Returns the child node from the node represented by the given form at the given position.
		 *
		 * @param form The external form being tested
		 * @param index The position of the child node to retrieve
		 * @return The external form representing the child node at the given position
		 */
		CHILD_FORM getChild(FORM form, int index);

		/**
		 * Returns the child form at the given position.
		 *
		 * @param childForms The list of child forms that was collected from the {@link ExternalForm}
		 * being tested
		 * @param index The position of the child form to retrieve
		 * @return The child form at the given position
		 */
		CHILD_FORM getChildForm(List<CHILD_FORM> childForms, int index);

		/**
		 * Returns a list of the children node with the same type (i.e. with the same node name) that
		 * are children of the node represented by the given form.
		 *
		 * @param form The external form being tested
		 * @return An ordered list based on the sequence of children node
		 */
		List<CHILD_FORM> getChildren(FORM form);

		/**
		 * Returns the number of children of the node represented by the given form with the same type
		 * (i.e. with the same node name).
		 *
		 * @param form The external form being tested
		 * @return The count of children of the same type
		 */
		int getChildrenSize(FORM form);

		/**
		 * Returns the value that part of the list ({@link #getChildValues()}) and that was used to
		 * create a child form.
		 *
		 * @param childForm The child form from which the value used to create it should be returned
		 * @return The value retrieved from the child node
		 */
		CHILD_VALUE getChildValue(CHILD_FORM childForm);

		/**
		 * Retrieves the value to be expected at the given position when inserted into the document.
		 *
		 * @param index The position of the value used to insert a child node
		 * @return The value at the given position that is used to insert a child node
		 */
		CHILD_VALUE getExpectedChildValue(int index);

		/**
		 * Returns a list of values that will be used to create children of the node represented by
		 * the form being tested. Each value will be used when calling {@link #addChild(Object, Object)}.
		 *
		 * @return A list of values, which should have more than 1 item
		 */
		List<CHILD_VALUE> getExpectedChildValues();

		/**
		 * Returns the ordered list of names that composes the different children being tested.
		 *
		 * @return A non-<code>null</code> list of at least one item or more
		 */
		List<String> getNodeNames();

		/**
		 * Removes the child node from the node represented by the given form.
		 *
		 * @param form The external form being tested
		 * @param index The position of the child node within the list of children of the same type
		 * (i.e. with the same node name)
		 */
		void removeChild(FORM form, int index);
	}

	/**
	 * The runner associated with {@link ChildListNodeTester}.
	 */
	private class ChildListNodeTesterRunner extends NodeTesterRunner {

		ContainerNodeRetriever containerNodeRetriever;
		int currentChildListCount;
		int rangeIndex;
		List<Integer[]> ranges;

		/**
		 * Creates a new <code>ChildListNodeTesterRunner</code>.
		 *
		 * @param tester This object defines a single child node to test
		 */
		ChildListNodeTesterRunner(ChildListNodeTester<FORM, ?, ?> tester) {
			super(tester);
			initialize(tester);
		}

		/**
		 * Adds the given range of
		 *
		 * @param startIndex The beginning position of the range, inclusive
		 * @param endIndex The end position of the range, exclusive
		 */
		void addRange(int startIndex, int endIndex) {
			ranges.add(new Integer[] { Math.min(startIndex, endIndex), Math.max(startIndex, endIndex) });
		}

		private ContainerNodeRetriever buildContainerNodeRetriever() {
			return new ContainerNodeRetriever() {
				@Override
				public Node getContainerNode(Node parent) {
					return parent;
				}
			};
		}

		private String displayString() {
			return "<" + getNodeName() + ">";
		}

		@SuppressWarnings("unchecked")
		private List<Node> getChildrenNodes(Node node) {

			ChildListNodeTester<FORM, Object, Object> tester = (ChildListNodeTester<FORM, Object, Object>) this.tester;
			List<String> nodeNames = tester.getNodeNames();

			List<Node> children = new ArrayList<Node>();
			node = node.getFirstChild();

			while (node != null) {

				if (nodeNames.contains(node.getNodeName())) {
					children.add(node);
				}

				node = node.getNextSibling();
			}

			return children;
		}

		private Node getContainerNode(Node parent) {
			return containerNodeRetriever.getContainerNode(parent);
		}

		@SuppressWarnings("unchecked")
		int getDefaultChildrenCount() {
			ChildListNodeTester<FORM, Object, Object> tester = (ChildListNodeTester<FORM, Object, Object>) this.tester;
			return tester.getExpectedChildValues().size();
		}

		private int getNodePositionOfInsertion(Node node, List<String> nodeNames) {

			String nodeName = node.getNodeName();

			for (int index = 0, count = nodeNames.size(); index < count; index++) {

				String name = nodeNames.get(index);

				if (name.equals(nodeName)) {
					return index;
				}
			}

			fail("The child node named <" + nodeName + "> is not included into the test");
			return -1;
		}

		private void initialize(ChildListNodeTester<FORM, ?, ?> tester) {

			ranges = new ArrayList<Integer[]>();
			containerNodeRetriever = buildContainerNodeRetriever();

			List<?> childValues = tester.getExpectedChildValues();

			if (!childValues.isEmpty()) {
				assertTrue(
					"The list of child values should contain at least 10 items for proper testing",
					childValues.size() >= 10
				);
			}
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		boolean isMultipleSupported() {
			return true;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		boolean isNodeDeletedWithNullValue() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void setup() {
			rangeIndex = ranges.size();
			ranges.add(new Integer[] { 0, getDefaultChildrenCount() });
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void testAdding(ExternalFormHolder holder) {

			ChildListNodeTester<FORM, Object, Object> tester = (ChildListNodeTester<FORM, Object, Object>) this.tester;
			List<String> nodeNames = tester.getNodeNames();
			List<Object> values = tester.getExpectedChildValues();

			// Retrieve the range within the list of values
			// that will be used to create the children
			Integer[] range = ranges.get(rangeIndex);

			// Create each child node
			for (int index = range[0]; index < range[1]; index++) {

				// Retrieve the child value that will be used to create a child
				Object value = values.get(index);

				assertNotNull(
					displayString() + " : The child value cannot be null",
					value
				);

				// Create the child node
				Object childForm = tester.addChild(holder.form, value);
				currentChildrenCount++;
				currentChildListCount++;

				assertNotNull(
					displayString() + " : The child form should have been created",
					childForm
				);

				// Retrieve the list of child nodes and make sure it matches
				// the number of child nodes that have been created so far
				Node containerNode = getContainerNode(holder.node);
				List<Node> childrenNode = getChildrenNodes(containerNode);

				assertEquals(
					displayString() + " : The child node was not created correctly",
					currentChildListCount,
					childrenNode.size()
				);

				// Make sure nothing else was created
				assertEquals(
					"The child node was not created correctly",
					currentChildrenCount,
					containerNode.getChildNodes().getLength()
				);

				// Now make sure the order is kept
				if (nodeNames.size() > 1) {
					testOrdinalPosition(containerNode, nodeNames);
				}
			}
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void testInitialState(ExternalFormHolder holder) {

			ChildListNodeTester<FORM, Object, Object> tester = (ChildListNodeTester<FORM, Object, Object>) this.tester;

			assertEquals(
				displayString() + " : Incorrect number of children was retrieved",
				currentChildListCount,
				tester.getChildren(holder.form).size()
			);

			// Test to make sure the number of child nodes is correctly retrieved
			assertEquals(
				displayString() + " : The number of children is inconsistent",
				currentChildListCount,
				tester.getChildrenSize(holder.form)
			);

			// Make sure nothing changed
			Node containerNode = getContainerNode(holder.node);

			if (containerNode == null) {
				assertEquals(
					displayString() + " : The number of children is inconsistent",
					0,
					currentChildrenCount
				);
			}
			else {
				assertEquals(
					displayString() + " : The number of children is inconsistent",
					currentChildrenCount,
					containerNode.getChildNodes().getLength()
				);
			}
		}

		private void testOrdinalPosition(Node containerNode, List<String> nodeNames) {

			Node childNode = containerNode.getFirstChild();

			if (childNode != null) {

				List<Integer> nodePositions = new ArrayList<Integer>();

				do {
					int nodePosition = getNodePositionOfInsertion(childNode, nodeNames);
					nodePositions.add(nodePosition);
					childNode = childNode.getNextSibling();
				}
				while (childNode != null);

				// Make sure the ordinal numbers are from the smaller number to the biggest number
				int previousPosition = -1;

				for (int nodePosition : nodePositions) {

					if (previousPosition > nodePosition) {
						fail("The insertion was not performed following the ordering.");
					}

					previousPosition = nodePosition;
				}
			}
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void testReading(ExternalFormHolder holder) {

			ChildListNodeTester<FORM, Object, Object> tester = (ChildListNodeTester<FORM, Object, Object>) this.tester;
			List<Object> childForms = tester.getChildren(holder.form);

			assertEquals(
				displayString() + " : The number of children is inconsistent",
				currentChildListCount,
				childForms.size()
			);

			Integer[] range = ranges.get(rangeIndex);

			// Read each child within the range
			for (int index = range[0]; index < range[1]; index++) {

				// Translate the position to the actual position within the list of children
				int translatedPosition = translatePosition(rangeIndex, index);

				// Retrieve the child form
				Object childForm = tester.getChild(holder.form, translatedPosition);

				assertNotNull(
					displayString() + " : The child form cannot be null",
					childForm
				);

				// Retrieve the child value from the child form
				Object childValue = tester.getChildValue(childForm);

				// Retrieve the value that was used to create the child
				Object expectedChildValue = tester.getExpectedChildValue(index);

				assertEquals(
					displayString() + " : The child value was not retrieved correctly",
					expectedChildValue,
					childValue
				);

				// Retrieve the child from the list of child forms
				childForm = tester.getChildForm(childForms, translatedPosition);

				assertNotNull(
					displayString() + " : The child form cannot be null",
					childForm
				);

				// Retrieve the child value from the child form
				childValue = tester.getChildValue(childForm);

				assertEquals(
					displayString() + " : The child value was not retrieved correctly",
					expectedChildValue,
					childValue
				);
			}

			// Now test reading each child randomly
			Random random = new Random();
			List<Integer> positions = new ArrayList<Integer>();

			for (int index = range[0]; index < range[1]; index++) {
				positions.add(index);
			}

			while (!positions.isEmpty()) {

				// Get a new random position
				int position = (positions.size() == 1) ? 0 : random.nextInt(positions.size());
				int index = positions.remove(position);

				// Translate the position to the actual position within the list of children
				int translatedPosition = translatePosition(rangeIndex, index);

				// Retrieve the child form
				Object childForm = tester.getChild(holder.form, translatedPosition);

				assertNotNull(
					displayString() + " : The child form cannot be null",
					childForm
				);

				// Retrieve the child value from the child form
				Object childValue = tester.getChildValue(childForm);
				Object expectedChildValue = tester.getExpectedChildValue(index);

				// Now retrieve the translated position within the list of values
				assertEquals(
					displayString() + " : The child value was not retrieved correctly",
					expectedChildValue,
					childValue
				);

				// Retrieve the child
				childForm = tester.getChildForm(childForms, translatedPosition);

				assertNotNull(
					displayString() + " : The child form cannot be null",
					childForm
				);

				// Retrieve the child value from the child form
				childValue = tester.getChildValue(childForm);

				assertEquals(
					displayString() + " : The child value was not retrieved correctly",
					expectedChildValue,
					childValue
				);
			}
		}

		private void testReadingAfterRemoval(ChildListNodeTester<FORM, Object, Object> tester,
		                                     ExternalFormHolder holder,
		                                     List<Object> childForms,
		                                     List<Object> values,
		                                     int rangesIndexToSkip,
		                                     int offset) {

			// Iterate through all the ranges to make sure every single child is correct
			for (int rangesIndex = 0; rangesIndex < ranges.size(); rangesIndex++) {

				// Skip the range that is being used to delete children
				if (rangesIndex == rangesIndexToSkip) {
					continue;
				}

				Integer[] range = ranges.get(rangesIndex);

				// Iterate through a single range and check every child is read correctly
				for (int rangeIndex = range[0]; rangeIndex < range[1]; rangeIndex++) {

					// Translate the position to the actual position within the list of children
					// but skip the range that is used to delete children, the offset the number
					// of children contained by that range
					int translatedPosition = translatePosition(rangesIndex, rangeIndex, rangesIndexToSkip);

					// The position is after the index used to delete children, adjust the translated
					// position by adding the count of remaining children that have not being deleted yet
					if (rangesIndex > rangesIndexToSkip) {
						translatedPosition += offset;
					}

					// Retrieve the child form
					Object childForm = tester.getChild(holder.form, translatedPosition);

					assertNotNull(
						displayString() + " : The child form cannot be null",
						childForm
					);

					// Retrieve the child value from the child form
					Object childValue = tester.getChildValue(childForm);
					Object expectedChildValue = tester.getExpectedChildValue(rangeIndex);

					assertEquals(
						displayString() + " : The child value was not retrieved correctly",
						expectedChildValue,
						childValue
					);

					// Retrieve the child
					childForm = tester.getChildForm(childForms, translatedPosition);

					assertNotNull(
						displayString() + " : The child form cannot be null",
						childForm
					);

					// Retrieve the child value from the child form
					childValue = tester.getChildValue(childForm);

					assertEquals(
						displayString() + " : The child value was not retrieved correctly",
						expectedChildValue,
						childValue
					);
				}
			}
		}

		private void testReadingAfterRemoval(ChildListNodeTester<FORM, Object, Object> tester,
		                                     ExternalFormHolder holder,
		                                     List<Object> childForms,
		                                     List<Object> values,
		                                     List<Integer> positions) {

			int childIndex = 0;

			for (int position : positions) {

				// Retrieve the child form
				Object childForm = tester.getChild(holder.form, childIndex++);

				assertNotNull(
					displayString() + " : The child form cannot be null",
					childForm
				);

				// Retrieve the child value from the child form
				Object childValue = tester.getChildValue(childForm);
				Object expectedChildValue = tester.getExpectedChildValue(position);

				assertEquals(
					displayString() + " : The child value was not retrieved correctly",
					expectedChildValue,
					childValue
				);
			}
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void testRemoving(ExternalFormHolder holder) {

			// Nothing to test
			if (ranges.isEmpty()) {
				return;
			}

			ChildListNodeTester<FORM, Object, Object> tester = (ChildListNodeTester<FORM, Object, Object>) this.tester;
			Random random = new Random();
			List<Object> values = tester.getExpectedChildValues();
			List<Object> readOnlyChildForms = tester.getChildren(holder.form);

			// Now remove all the children at random position and
			// test reading the remaining between each removal
			Integer[] range = ranges.get(rangeIndex);
			int rangeCount = ranges.size();

			// Cache the positions
			List<Integer> positions = new ArrayList<Integer>();

			for (int index = range[0]; index < range[1]; index++) {
				positions.add(index);
			}

			List<Integer> readOnlyPositions = new ArrayList<Integer>();
			readOnlyPositions.addAll(positions);

			while (!positions.isEmpty()) {

				// Get a new random position
				int position = (positions.size() == 1) ? 0 : random.nextInt(positions.size());
				positions.remove(position);
				readOnlyPositions.remove(position);

				// Translate the position to the actual position within the list of children
				// plus the position, which the number of items within the range used to delete
				// children before the child being deleted
				int translatedPosition = position;

				if (rangeIndex > 0) {
					int previousRangeIndex = rangeIndex - 1;
					int lastRangeIndex = ranges.get(previousRangeIndex)[1];
					translatedPosition += translatePosition(previousRangeIndex, lastRangeIndex);
				}

				// Retrieve the child form
				tester.removeChild(holder.form, translatedPosition);
				currentChildrenCount--;
				currentChildListCount--;

				// Adjust the cached positions
				for (int positionIndex = position; positionIndex < positions.size(); positionIndex++) {
					positions.set(positionIndex, positions.get(positionIndex) - 1);
				}

				//
				// Test 1: Make sure only one node was removed
				//
				// Make sure the parent node has the right amount of children left
				assertEquals(
					displayString() + " : The number of children is inconsistent",
					currentChildListCount,
					tester.getChildrenSize(holder.form)
				);

				Node containerNode = getContainerNode(holder.node);

				if (containerNode == null) {
					assertEquals(
						displayString() + " : The number of children is inconsistent",
						0,
						currentChildrenCount
					);
				}
				else {
					assertEquals(
						displayString() + " : The number of children is inconsistent",
						currentChildrenCount,
						containerNode.getChildNodes().getLength()
					);
				}

				// Retrieve the values that will be used to create the child nodes
				List<Object> childForms = tester.getChildren(holder.form);

				assertEquals(
					displayString() + " : The number of children is inconsistent",
					currentChildListCount,
					childForms.size()
				);

				//
				// Test 2: Read the other children to make sure the right one was removed
				//
				// Retrieve the child
				if (rangeCount == 1) {
					testReadingAfterRemoval(tester, holder, readOnlyChildForms, values, readOnlyPositions);
				}
				else {
					testReadingAfterRemoval(tester, holder, childForms, values, rangeIndex, positions.size());
				}
			}

			// Completed removing the children from the selected range
			ranges.remove(rangeIndex);
		}

		private int translatePosition(int endRangesIndex, int rangeIndexToAdjust) {
			return translatePosition(endRangesIndex, rangeIndexToAdjust, -1);
		}

		private int translatePosition(final int endRangeIndex,
		                              final int rangeIndexToAdjust,
		                              final int rangeIndexToSkip) {

			int translatedPosition = 0;

			for (int rangesIndex = 0; rangesIndex <= endRangeIndex; rangesIndex++) {

				// Skip the specified range since work is being done with it
				if (rangesIndex == rangeIndexToSkip) {
					continue;
				}

				Integer[] range = ranges.get(rangesIndex);

				// Quick calculation
				if (rangesIndex != endRangeIndex) {
					translatedPosition += (range[1] - range[0]);
				}
				// Adjust the index within the range that ends the translation
				else {
					translatedPosition += (rangeIndexToAdjust - range[0]);
				}
			}

			return translatedPosition;
		}
	}

	/**
	 * This controller simply asks its runner to add the child node.
	 */
	private class ChildNodeRunnerAddingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		void log(Logger logger) {
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.tester.getNodeName() + "> : addition");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ChildNodeTesterRunner runner = (ChildNodeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testAdding(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This controller simply asks its runner to read the child node.
	 */
	private class ChildNodeRunnerReadingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		void log(Logger logger) {
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.tester.getNodeName() + "> : addition");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ChildNodeTesterRunner runner = (ChildNodeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testReading(holder);
		}
	}

	/**
	 * This controller simply asks its runner to remove the child node.
	 */
	private class ChildNodeRunnerRemovingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		void log(Logger logger) {
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.tester.getNodeName() + "> : addition");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ChildNodeTesterRunner runner = (ChildNodeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testRemoving(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This <code>ChildNodeTester</code> tests when a form add or remove a single child node.
	 */
	public interface ChildNodeTester<FORM extends ExternalForm, VALUE extends ExternalForm> extends NodeTester<FORM, VALUE> {

		/**
		 * Adds the single child node to the node represented by the given form.
		 *
		 * @param form The external form being tested
		 * @return The {@link ExternalForm} of the child node that was added
		 */
		VALUE addChild(FORM form);

		/**
		 * Retrieves the {@link ExternalForm} representing the single child node.
		 *
		 * @param form The external form being tested
		 * @return The {@link ExternalForm} representing the child node
		 */
		VALUE getChild(FORM form);

		/**
		 * Determines whether the given form has a child node with the specific node name.
		 *
		 * @param form The external form being tested
		 * @return <code>true</code> if the document has the single node as a child node of the node
		 * for which the given form represents; <code>false</code> if the node does not exist
		 */
		boolean hasChild(FORM form);

		/**
		 * Removes the single child node from the node represented by the given form.
		 *
		 * @param form The external form being tested
		 */
		void removeChild(FORM form);
	}

	/**
	 * The runner associated with {@link ChildNodeTester}.
	 */
	private class ChildNodeTesterRunner extends AbstractChildNodeTesterRunner {

		/**
		 * Creates a new <code>ChildNodeTesterRunner</code>.
		 *
		 * @param tester This object defines a single child node to test
		 */
		ChildNodeTesterRunner(ChildNodeTester<FORM, ?> tester) {
			super(tester);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testAdding(ExternalFormHolder holder) {

			// Don't add the child if it's present
			if (hasChild) {
				return;
			}

			ChildNodeTester<FORM, ? extends ExternalForm> tester = (ChildNodeTester<FORM, ?>) this.tester;

			// Add the child node
			ExternalForm childForm = tester.addChild(holder.form);
			hasChild = true;
			currentChildrenCount++;

			assertNotNull(
				displayString() + " : The child form should exist",
				childForm
			);

			// Make sure the child node was indeed added
			assertEquals(
				displayString() + " : The number of children is inconsistent",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			// Make sure the child node is correctly retrieved
			assertTrue(
				displayString() + " : The child node should have been found",
				tester.hasChild(holder.form)
			);

			// Make sure the right child node was added and is retrieved by the ExternalForm
			assertNotNull(
				displayString() + " : The child node should exist",
				getChildNode(holder.node)
			);

			// Make sure nothing changed
			assertEquals(
				displayString() + " : The child node should exist",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testInitialState(ExternalFormHolder holder) {

			ChildNodeTester<FORM, ?> tester = (ChildNodeTester<FORM, ?>) this.tester;

			// Make sure the child node does not exist
			assertEquals(
				displayString() + " : The child node should not exist",
				hasChild,
				tester.hasChild(holder.form)
			);

			// Make sure nothing changed
			assertEquals(
				displayString() + " : The child node should not exist",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			// The child node should not exist
			if (hasChild) {
				assertNotNull(
					displayString() + " : The child node should exist",
					getChildNode(holder.node)
				);
			}
			else {
				assertNull(
					displayString() + " : The child node should not exist",
					getChildNode(holder.node)
				);
			}

			// Make sure nothing changed
			assertEquals(
				displayString() + " : The child node should not exist",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testReading(ExternalFormHolder holder) {
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testRemoving(ExternalFormHolder holder) {

			// Nothing to remove
			if (!hasChild) {
				return;
			}

			ChildNodeTester<FORM, ?> tester = (ChildNodeTester<FORM, ?>) this.tester;

			// Make sure the child node exists
			assertTrue(
				displayString() + " : The child node should exist",
				tester.hasChild(holder.form)
			);

			// Make sure nothing changed with the previous check
			assertEquals(
				displayString() + " : The child node should exist",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			// Remove the child node
			tester.removeChild(holder.form);
			currentChildrenCount--;
			hasChild = false;

			// Make sure the child node was indeed removed
			assertEquals(
				displayString() + " : The child node should have been removed",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);

			// Make sure the right child node was removed
			assertNull(
				displayString() + " : The wrong child node was removed",
				getChildNode(holder.node)
			);

			// Test to make sure the child node is not found
			assertFalse(
				displayString() + " : The child node should not exist",
				tester.hasChild(holder.form)
			);
		}
	}

	/**
	 * This controller tests adding one of the possible child nodes.
	 */
	private class ChoiceChildNodeRunnerAddingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void log(Logger logger) {
			ChoiceChildNodeTesterRunner runner = (ChoiceChildNodeTesterRunner) this.runner;
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.getNodeName() + "> : addition");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ChoiceChildNodeTesterRunner runner = (ChoiceChildNodeTesterRunner) this.runner;

			// Create a random range within the list of values used to create the child nodes
			Random random = new Random();
			int index = random.nextInt(runner.testers.length);

			runner.currentChildrenCount = currentChildrenCount;
			runner.index = index;
			runner.testInitialState(holder);
			runner.testAdding(holder);
			runner.testReading(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This controller tests reading the child node that was added.
	 */
	private class ChoiceChildNodeRunnerReadingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void log(Logger logger) {
			ChoiceChildNodeTesterRunner runner = (ChoiceChildNodeTesterRunner) this.runner;
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.getNodeName() + "> : reading");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ChoiceChildNodeTesterRunner runner = (ChoiceChildNodeTesterRunner) this.runner;

			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testReading(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This controller tests removing the child node that was added.
	 */
	private class ChoiceChildNodeRunnerRemovingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void log(Logger logger) {
			ChoiceChildNodeTesterRunner runner = (ChoiceChildNodeTesterRunner) this.runner;
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.getNodeName() + "> : removing");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ChoiceChildNodeTesterRunner runner = (ChoiceChildNodeTesterRunner) this.runner;

			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testAdding(holder);
			runner.testReading(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	private class ChoiceChildNodeTesterRunner extends NodeTesterRunner {

		/**
		 * Specifies which child to add or remove.
		 */
		public int index;

		/**
		 * The testers for each of the child nodes when only one can be added.
		 */
		private ChildNodeTester<?, ?>[] testers;

		/**
		 * Creates a new <code>ChoiceChildNodeTesterRunner</code>.
		 *
		 * @param testers The testers for each of the child nodes when only one can be added
		 */
		ChoiceChildNodeTesterRunner(ChildNodeTester<?, ?>[] testers) {
			super(null);
			this.index   = -1;
			this.testers = testers;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		String getNodeName() {
			return testers[index].getNodeName();
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		boolean isMultipleSupported() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		boolean isNodeDeletedWithNullValue() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testAdding(ExternalFormHolder holder) {
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testInitialState(ExternalFormHolder holder) {
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testReading(ExternalFormHolder holder) {
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testRemoving(ExternalFormHolder holder) {
		}
	}

	/**
	 * This controller tests adding a random portion of child nodes (which are owned by a container node).
	 */
	private class ContainerChildListNodeRunnerAddingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void log(Logger logger) {
			ContainerChildListNodeTesterRunner runner = (ContainerChildListNodeTesterRunner) this.runner;
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.getContainerNodeName() + "> : adding");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ContainerChildListNodeTesterRunner runner = (ContainerChildListNodeTesterRunner) this.runner;

			// Create a random range within the list of values used to create the child nodes
			Random random = new Random();
			int count = runner.runner.getDefaultChildrenCount();

			int startIndex = random.nextInt(count);
			int endIndex = random.nextInt(count);

			// Make sure the start and end indices are not the same
			while (endIndex == startIndex) {
				endIndex = random.nextInt(count);
			}

			runner.currentChildrenCount = currentChildrenCount;
			runner.runner.addRange(startIndex, endIndex);
			runner.runner.rangeIndex = runner.runner.ranges.size() - 1;
			runner.testInitialState(holder);
			runner.testAdding(holder);
			runner.testReading(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This controller tests reading a random portion of child nodes (which are owned by a container node).
	 */
	private class ContainerChildListNodeRunnerReadingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void log(Logger logger) {
			ContainerChildListNodeTesterRunner runner = (ContainerChildListNodeTesterRunner) this.runner;
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.getContainerNodeName() + "> : reading");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ContainerChildListNodeTesterRunner runner = (ContainerChildListNodeTesterRunner) this.runner;
			int rangesCount = runner.runner.ranges.size();

			// Nothing to read
			if (rangesCount == 0) {
				return;
			}

			// Read a range randomly
			Random random = new Random();
			int rangeIndex = random.nextInt(rangesCount);
			runner.runner.rangeIndex = rangeIndex;

			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testReading(holder);
		}
	}

	/**
	 * This controller tests removing a random portion of child nodes (which are owned by a container node).
	 */
	private class ContainerChildListNodeRunnerRemovingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void log(Logger logger) {
			ContainerChildListNodeTesterRunner runner = (ContainerChildListNodeTesterRunner) this.runner;
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.getContainerNodeName() + "> : removing");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			ContainerChildListNodeTesterRunner runner = (ContainerChildListNodeTesterRunner) this.runner;
			int rangesCount = runner.runner.ranges.size();

			// Nothing to read
			if (rangesCount == 0) {
				return;
			}

			// Read a range randomly
			Random random = new Random();
			int rangeIndex = random.nextInt(rangesCount);
			runner.runner.rangeIndex = rangeIndex;

			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testRemoving(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This <code>ContainerChildListNodeTester</code> tests a form when the node it represents can
	 * have zero or many child nodes of the same type (i.e. with the same node name) that are
	 * "wrapped" by a container node.
	 */
	public interface ContainerChildListNodeTester<FORM extends ExternalForm, CHILD_FORM, CHILD_VALUE> extends ChildListNodeTester<FORM, CHILD_FORM, CHILD_VALUE> {

		/**
		 * Returns the name of the container node which can contain zero or many nodes.
		 *
		 * @param containerNodeName The node that contains the child nodes added by the given tester
		 */
		String getContainerNodeName();
	}

	/**
	 * This runner tests adding child nodes not directly to the parent node but to a container node.
	 * <p>
	 * Example:
	 * <pre><code>&lt;parentNode&gt;
	 *   &lt;containerNode&gt;
	 *      &lt;child_1&gt;
	 *      &lt;child_2&gt;
	 *      ...
	 *      &lt;child_n&gt;
	 *   &lt;/containerNode&gt;
	 *&lt;/parentNode&gt;</code></pre>
	 */
	private class ContainerChildListNodeTesterRunner extends NodeTesterRunner {

		/**
		 * This runner adds child nodes to a container node and not directly to the parent.
		 */
		private ChildListNodeTesterRunner runner;

		/**
		 * Creates a new <code>ContainerChildListNodeTesterRunner</code>.
		 *
		 * @param tester This runner adds child nodes to a container node and not directly to the parent
		 */
		ContainerChildListNodeTesterRunner(ContainerChildListNodeTester<FORM, ?, ?> tester) {
			super(tester);
			this.runner = new ChildListNodeTesterRunner(tester);
			this.runner.containerNodeRetriever = buildContainerNodeRetreiver();
		}

		private ContainerNodeRetriever buildContainerNodeRetreiver() {
			return new ContainerNodeRetriever() {
				@Override
				public Node getContainerNode(Node parent) {
					return ContainerChildListNodeTesterRunner.this.getNode(parent);
				}
			};
		}

		private String displayString() {
			return "<" + getContainerNodeName() + ">";
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		String getActualNodeName() {
			return getContainerNodeName();
		}

		@SuppressWarnings("unchecked")
		String getContainerNodeName() {
			ContainerChildListNodeTester<FORM, ?, ?> tester = (ContainerChildListNodeTester<FORM, ?, ?>) this.tester;
			return tester.getContainerNodeName();
		}

		private Node getNode(Node node) {

			node = node.getFirstChild();

			while (node != null) {

				if (getContainerNodeName().equals(node.getNodeName())) {
					return node;
				}

				node = node.getNextSibling();
			}

			return null;
		}

		@Override
		boolean isMultipleSupported() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		boolean isNodeDeletedWithNullValue() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void test(ExternalFormHolder holder) {
			runner.setup();
			super.test(holder);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testAdding(ExternalFormHolder holder) {

			// Check to see if the container node will be added during addition
			Node childNode = getNode(holder.node);

			if (childNode == null) {
				currentChildrenCount++;
			}

			// Delegate the call
			runner.testAdding(holder);

			// Test to make sure the children were added to the container node
			assertEquals(
				runner.displayString() + " : The child nodes were not created correctly",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testInitialState(ExternalFormHolder holder) {
			runner.testInitialState(holder);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testReading(ExternalFormHolder holder) {
			runner.testReading(holder);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testRemoving(ExternalFormHolder holder) {

			// Delegate the call
			runner.testRemoving(holder);

			// Make sure the deletion happened correctly
			Node childNode = getNode(holder.node);

			if (runner.currentChildListCount == 0) {

				currentChildrenCount--;

				assertNull(
					displayString() + " : The child nodes were not removed correctly",
					childNode
				);
			}
			else {
				assertNotNull(
					displayString() + " : The child nodes were not removed correctly",
					childNode
				);
			}

			assertEquals(
				displayString() + " : The child nodes were not removed correctly",
				currentChildrenCount,
				holder.node.getChildNodes().getLength()
			);
		}
	}

	/**
	 * This is used to retrieve the actual container of the node being tested.
	 */
	interface ContainerNodeRetriever {

		/**
		 * Retrieves the node that contains the child nodes, which may differ from the given node that
		 * is been manipulated by its {@link ExternalForm}.
		 *
		 * @param parent The parent node manipulated by the {@link ExternalForm} being tested
		 * @return Either the given node or a child node acting like a container
		 */
		Node getContainerNode(Node parent);
	}

	/**
	 * A <code>Controller</code> wraps a {@link NodeTesterRunner} and executes one of the possible
	 * tests, which is either add, read or remove the property from the node being manipulated by
	 * the {@link ExternalForm}.
	 */
	private abstract class Controller {

		/**
		 * The number of children the node currently has, which is required to make sure the runner
		 * tests the accuracy of the {@link ExternalForm} when it manipulates the node.
		 */
		int currentChildrenCount;

		/**
		 * The runner to executed but with some tweaks applied.
		 */
		NodeTesterRunner runner;

		/**
		 * Outputs what is being executed, used for debug purposes.
		 *
		 * @param logger Used to log messages during execution
		 */
		abstract void log(Logger logger);

		/**
		 * Executes one of the operations that can be performed on an node.
		 *
		 * @param holder The holder of The {@link ExternalForm} to test and the node that is manipulated
		 */
		abstract void test(ExternalFormHolder holder);

		/**
		 * {@inheritDoc}
		 */
		@Override
		public String toString() {
			return getClass().getSimpleName();
		}
	}

	/**
	 * A <code>ExternalFormBuilder</code> is responsible to create the {@link ExternalForm} to be tested.
	 */
	public interface ExternalFormBuilder<FORM extends ExternalForm> {

		/**
		 * Creates the {@link ExternalForm} to test.
		 *
		 * @return The {@link ExternalForm} to test
		 * @throws IOException If an error occurred during the creation process
		 */
		FORM buildExternalForm() throws IOException;

		/**
		 * Adds
		 *
		 * @param form
		 * @return
		 */
		FORM buildExternalForm(ExternalForm parentForm);

		/**
		 * Returns the number of attributes the {@link Node} contains before any manipulation has been performed.
		 *
		 * @return The count of children before any testing has been done
		 */
		int getDefaultAttributeCount();

		/**
		 * Returns the node represented by the {@link ExternalForm}.
		 *
		 * @param form The {@link ExternalForm} for which to return its node
		 * @return The node from the document that is been manipulated
		 */
		Node getNode(FORM form);

		/**
		 * Returns
		 *
		 * @return
		 */
		String getNodeName();

		/**
		 * Returns
		 *
		 * @return
		 */
		List<String> getTreeNodeNames();
	}

	private abstract class ExternalFormHolder {

		/**
		 *
		 */
		FORM form;

		/**
		 * @param form The {@link ExternalForm} to test
		 * @param node The node that is manipulated by the given {@link ExternalForm}
		 */
		Node node;

		/**
		 *
		 */
		abstract void rebuild();
	}

	private interface NodeCountRetriever {
		int getCount(Node parent);
	}

	/**
	 * The root interface of the testers defined to test the manipulation of a node's property (either
	 * an attribute, a single child node, a list of child nodes or a text node).
	 */

	private interface NodeTester<FORM extends ExternalForm, Value> {
		/**
		 * Retrieves the name of the node for which retrieving and setting its value is tested.
		 *
		 * @return The name of the node to test
		 */
		String getNodeName();
	}

	/**
	 * This is the root of the runner class, which is associated with a {@link NodeTester}.
	 */
	private abstract class NodeTesterRunner {

		/**
		 * Keeps track of the current count of child nodes owned by the node being manipulated.
		 */
		int currentChildrenCount;

		/**
		 *
		 */
		@SuppressWarnings("unused")
		ExternalFormHolder holder;

		/**
		 * For debug purposes, this is the name of the parent node.
		 */
		String parentNodeName;

		/**
		 * This object defines a single node to test (which is either a child element or an attribute).
		 */
		final NodeTester<FORM, Object> tester;

		/**
		 * Creates a new <code>AbstractNodeTester</code>.
		 *
		 * @param tester The bridge between this tester and the document's node being tested
		 */
		@SuppressWarnings("unchecked")
		NodeTesterRunner(NodeTester<FORM, ?> tester) {
			super();
			this.tester = (NodeTester<FORM, Object>) tester;
			assertNotNull("The tester cannot be null: "    + tester, tester);
			assertNotNull("The node name cannot be null: " + tester, tester.getNodeName());
		}

		/**
		 * Returns
		 *
		 * @return
		 */
		String getActualNodeName() {
			return getNodeName();
		}

		/**
		 * Returns the name of the node for which this runner tests.
		 *
		 * @return The node name
		 */
		String getNodeName() {
			return tester.getNodeName();
		}

		/**
		 * Determines whether
		 *
		 * @return
		 */
		abstract boolean isMultipleSupported();

		/**
		 * Determines whether the node being tested is removed from its parent element when the value
		 * set is <code>null</code>.
		 *
		 * @return <code>true</code> if the node is removed from its parents after the property is
		 * nullified; <code>false</code> if it is not removed
		 */
		abstract boolean isNodeDeletedWithNullValue();

		/**
		 * Notification before a test is performed.
		 */
		void setup() {
		}

		/**
		 * Tests the given {@link ExternalForm} by manipulating the node for a single property.
		 *
		 * @param holder The holder of The {@link ExternalForm} to test and the node that is manipulated
		 */
		void test(ExternalFormHolder holder) {
			setup();
			testInitialState(holder);
			testAdding(holder);
			testReading(holder);
			testRemoving(holder);
		}

		/**
		 * Tests the given {@link ExternalForm} by adding the property to the node.
		 *
		 * @param holder The holder of The {@link ExternalForm} to test and the node that is manipulated
		 */
		abstract void testAdding(ExternalFormHolder holder);

		/**
		 * Tests the given {@link ExternalForm} by making sure the given node is in its original state.
		 *
		 * @param holder The holder of The {@link ExternalForm} to test and the node that is manipulated
		 */
		abstract void testInitialState(ExternalFormHolder holder);

		/**
		 * Tests the given {@link ExternalForm} by reading the property from the node.
		 *
		 * @param holder The holder of The {@link ExternalForm} to test and the node that is manipulated
		 */
		abstract void testReading(ExternalFormHolder holder);

		/**
		 * Tests the given {@link ExternalForm} by removing the property from the node.
		 *
		 * @param holder The holder of The {@link ExternalForm} to test and the node that is manipulated
		 */
		abstract void testRemoving(ExternalFormHolder holder);

		/**
		 * {@inheritDoc}
		 */
		@Override
		public String toString() {
			return getClass().getSimpleName() + " : " + tester.getNodeName();
		}
	}

	/**
	 * This <code>AttributeNodeTester</code> does nothing, it is used when an attribute is not
	 * currently by the {@link ExternalForm}.
	 */
	private class NotSupportedAttributeTester implements AttributeNodeTester<FORM, Object> {

		/**
		 * The name of the attribute that is not supported by the {@link ExternalForm}.
		 */
		private String attributeName;

		/**
		 * Creates a new <code>NotSupportedAttributeTester</code>.
		 *
		 * @param attributeName The name of the attribute that is not supported by the {@link ExternalForm}
		 */
		NotSupportedAttributeTester(String attributeName) {
			super();
			this.attributeName = attributeName;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public boolean doesNodeAlreadyExist() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public Object getDefaultValue() {
			return null;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public String getNodeName() {
			return attributeName;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public Object getValue(FORM form) {
			return null;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public Object getValue1() {
			return null;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public Object getValue2() {
			return null;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public boolean isNodeDeletedWithNullValue() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public boolean isNullAllowed() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public void setValue(FORM form, Object value) {
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public String toString(Object value) {
			return null;
		}
	}

	/**
	 * This runner simply does nothing but adds support for adding an attribute name into the list of
	 * attributes so that the ordering defined by the XML schema is properly tested.
	 */
	private class NotSupportedAttributeTesterRunner extends AttributeNodeTesterRunner {

		/**
		 * Creates a new <code>NotSupportedAttributeTesterRunner</code>.
		 *
		 * @param attributeName The name of the attribute that is currently not supported by the
		 * {@link ExternalForm}
		 */
		NotSupportedAttributeTesterRunner(String attributeName) {
			super(new NotSupportedAttributeTester(attributeName));
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testAdding(ExternalFormHolder holder) {
			// Nothing to do
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testInitialState(ExternalFormHolder holder) {
			// Nothing to do
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testReading(ExternalFormHolder holder) {
			// Nothing to do
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testRemoving(ExternalFormHolder holder) {
			// Nothing to do
		}
	}

	/**
	 * This wraps a node name that is currently not supported by the Mapping SPI.
	 */
	private class NotSupportedNodeTester implements NodeTester<FORM, Object> {

		/**
		 * The name of the node that is currently not supported by the {@link ExternalForm}.
		 */
		private String nodeName;

		/**
		 * Creates a new <code>NotSupportedNodeTester</code>.
		 *
		 * @param nodeName The name of the node that is currently not supported by the {@link ExternalForm}
		 */
		NotSupportedNodeTester(String nodeName) {
			super();
			this.nodeName = nodeName;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public String getNodeName() {
			return nodeName;
		}
	}

	/**
	 * This runner simply does nothing but adds support for adding a node name into the list of child
	 * nodes so that the ordering defined by the XML schema is properly tested.
	 */
	private class NotSupportedNodeTesterRunner extends NodeTesterRunner {

		/**
		 * Creates a new <code>NotSupportedNodeTesterRunner</code>.
		 *
		 * @param nodeName The name of the node that is currently not supported by the {@link ExternalForm}
		 */
		NotSupportedNodeTesterRunner(String nodeName) {
			super(new NotSupportedNodeTester(nodeName));
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		boolean isMultipleSupported() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		boolean isNodeDeletedWithNullValue() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testAdding(ExternalFormHolder holder) {
			// Nothing to do
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testInitialState(ExternalFormHolder holder) {
			// Nothing to do
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testReading(ExternalFormHolder holder) {
			// Nothing to do
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testRemoving(ExternalFormHolder holder) {
			// Nothing to do
		}
	}

	private abstract class PropertyNodeTesterRunner extends NodeTesterRunner {

		/**
		 * Keeps track of the status of the node's existence.
		 */
		private boolean nodeExists;

		/**
		 * Creates a new <code>PropertyNodeTesterRunner</code>.
		 *
		 * @param tester This object defines a single node to test a child text node
		 */
		PropertyNodeTesterRunner(PropertyTester<FORM, ?> tester) {
			super(tester);
		}

		/**
		 * Returns the display string representing this runner.
		 *
		 * @return A description of the node being tested
		 */
		abstract String displayString();

		/**
		 * Returns the number of children owned by the given parent {@link Node}.
		 *
		 * @param parentNode The owner of the child node being tested
		 * @return The count of children, which could be either attributes or child nodes
		 */
		abstract int getChildrenCount(Node parentNode);

		/**
		 * Returns the node being tested.
		 *
		 * @param parentNode The owner of the node being tested
		 * @return The child node of the given node that is being tested
		 */
		abstract Node getNode(Node parentNode);

		/**
		 * Returns the value of the given node.
		 *
		 * @param node The node being tested
		 * @return The value being tested
		 */
		abstract String getNodeValue(Node node);

		/**
		 * {@inheritDoc}
		 */
		@Override
		final boolean isMultipleSupported() {
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		boolean isNodeDeletedWithNullValue() {
			return propertyTester().isNodeDeletedWithNullValue();
		}

		PropertyTester<FORM, Object> propertyTester() {
			return (PropertyTester<FORM, Object>) tester;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testAdding(ExternalFormHolder holder) {

			PropertyTester<FORM, Object> tester = propertyTester();

			// Change the node value to null
			if (tester.isNullAllowed()) {

				tester.setValue(holder.form, null);

				if (nodeExists) {
					nodeExists = false;
					currentChildrenCount--;
				}

				assertEquals(
					displayString() + " : The element should not have any children",
					currentChildrenCount,
					getChildrenCount(holder.node)
				);

				// Make sure nothing changed
				Object result = tester.getValue(holder.form);

				assertNull(
					displayString() + " : The element's value should be null",
					result
				);

				// Make sure nothing changed
				assertEquals(
					displayString() + " : The element should not have any children",
					currentChildrenCount,
					getChildrenCount(holder.node)
				);

				if (tester.isNodeDeletedWithNullValue()) {
					holder.rebuild();
				}
			}

			// Change the value to something
			Object expectedValue1 = tester.getValue1();

			assertNotNull(
				displayString() + " : Value 1 cannot be null",
				expectedValue1
			);

			tester.setValue(holder.form, expectedValue1);

			if (!nodeExists) {
				nodeExists = true;
				currentChildrenCount++;
			}

			// The child node should have been added
			Node childNode = getNode(holder.node);

			assertNotNull(
				displayString() + " : The node cannot be null",
				childNode
			);

			assertEquals(
				displayString() + " : The element should have a " + currentChildrenCount + " children",
				currentChildrenCount,
				getChildrenCount(holder.node)
			);

			// Get the value
			Object result = tester.getValue(holder.form);

			assertEquals(
				displayString() + " : The element's value was not set correctly",
				expectedValue1,
				result
			);

			// Get the value directly
			String stringResult = getNodeValue(childNode);
			String stringNodeValue = tester.toString(expectedValue1);

			assertEquals(
				displayString() + " : The value was not set correctly",
				stringNodeValue,
				stringResult
			);

			// Change the value to something else
			Object expectedValue2 = tester.getValue2();
			assertNotNull(displayString() + " : Value 2 cannot be null", expectedValue2);

			assertNotSame(
				displayString() + " : Value 1 and value 2 cannot be the same",
				expectedValue1,
				expectedValue2
			);

			tester.setValue(holder.form, expectedValue2);

			// Get the value
			result = tester.getValue(holder.form);

			assertEquals(
				displayString() + " The element's value was not set correctly",
				expectedValue2,
				result
			);

			// Get the value directly
			stringResult = getNodeValue(childNode);
			stringNodeValue = tester.toString(expectedValue2);

			assertEquals(
				displayString() + " : The element's value was not set correctly",
				stringNodeValue,
				stringResult
			);

			// Change the value back to its original value
			Object defaultValue = tester.getDefaultValue();

			assertNotNull(
				displayString() + " : The default value cannot be null",
				defaultValue
			);

			tester.setValue(holder.form, defaultValue);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testInitialState(ExternalFormHolder holder) {

			PropertyTester<FORM, Object> tester = (PropertyTester<FORM, Object>) this.tester;
			nodeExists |= tester.doesNodeAlreadyExist();

			// Node name
			String nodeName = getNodeName();
			assertNotNull("The node name cannot be null", nodeName);

			// Test the initial state of the parent node
			assertEquals(
				displayString() + " : The child count is incorrect",
				currentChildrenCount,
				getChildrenCount(holder.node)
			);

			// The child node should either not exist or already being present
			Node childNode = getNode(holder.node);

			if (nodeExists) {
				assertNotNull(
					displayString() + " : The node should not be null",
					childNode
				);
			}
			else {
				assertNull(
					displayString() + " : The node should be null",
					childNode
				);
			}
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testReading(ExternalFormHolder holder) {

			PropertyTester<FORM, Object> tester = (PropertyTester<FORM, Object>) this.tester;

			// The node value should either be null or not null
			Object result = tester.getValue(holder.form);

			// Make sure the node's value is not null
			if (nodeExists) {
				assertNotNull(
					displayString() + " : The element's value should not be null",
					result
				);
				assertSame(
					displayString() + " : The element's value was not retrived correctly",
					tester.getDefaultValue(),
					result
				);
			}
			// The node is not present, make sure reading its value returns null
			else {
				assertNull(
					displayString() + " : The element's value should be null",
					result
				);
			}

			// Make sure nothing changed by reading the node's value
			assertEquals(
				displayString() + " : The element should not have any children",
				currentChildrenCount,
				getChildrenCount(holder.node)
			);

			// Retrieve the actual node
			Node childNode = getNode(holder.node);

			if (nodeExists) {
				assertNotNull(
					displayString() + " : The node should not be null",
					childNode
				);
			}
			else {
				assertNull(
					displayString() + " : The node should be null",
					childNode
				);
			}

			// Make sure nothing changed after retrieving the node
			assertEquals(
				displayString() + " : The element should not have any children",
				currentChildrenCount,
				getChildrenCount(holder.node)
			);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		void testRemoving(ExternalFormHolder holder) {

			PropertyTester<FORM, Object> tester = (PropertyTester<FORM, Object>) this.tester;

			// Change the node value to null, which will remove it from the parent node
			if (tester.isNullAllowed()) {

				tester.setValue(holder.form, null);

				if (nodeExists) {
					nodeExists = false;
					currentChildrenCount--;
				}

				assertEquals(
					displayString() + " : The child count does not match the current state",
					currentChildrenCount,
					getChildrenCount(holder.node)
				);
			}
		}
	}

	/**
	 * A <code>PropertyTester</code> handles testing either an attribute or a single child node of
	 * the node being tested.
	 */
	private interface PropertyTester<FORM extends ExternalForm, VALUE> extends NodeTester<FORM, VALUE> {

		/**
		 * Determines whether the property is by default present in the document.
		 *
		 * @return <code>true</code> if the property exists before any changes is done to the document;
		 * <code>false</code> otherwise
		 */
		boolean doesNodeAlreadyExist();

		/**
		 * If {@link #doesNodeAlreadyExist()} returns <code>true</code>, then this should return the
		 * node value that is already present in the document, which is done before the document is
		 * being changed.
		 *
		 * @return Either a non-<code>null</code> value if the node is present in the document or
		 * <code>null</code> if the node is not by default present in the document
		 */
		VALUE getDefaultValue();

		/**
		 * Retrieves the value of the node.
		 *
		 * @param form The external form that will retrieve the value
		 * @return The value, which can be <code>null</code>
		 */
		VALUE getValue(FORM form);

		/**
		 * Retrieves a non-<code>null</code> value that will be used for testing that is different
		 * than the one returned by {@link #getValue2()}.
		 *
		 * @return A non-<code>null</code> value used for testing
		 */
		VALUE getValue1();

		/**
		 * Retrieves a non-<code>null</code> value that will be used for testing that is different
		 * than the one returned by {@link #getValue1()}.
		 *
		 * @return A non-<code>null</code> value used for testing
		 */
		VALUE getValue2();

		/**
		 * Determines whether the node being tested is removed from its parent element when the value
		 * set is <code>null</code>.
		 *
		 * @return <code>true</code> if the node is removed from its parents after the property is
		 * nullified; <code>false</code> if it is not removed
		 */
		boolean isNodeDeletedWithNullValue();

		/**
		 * Determines whether the node can receive a <code>null</code> value or not.
		 *
		 * @return <code>true</code> if <code>null</code> is a valid value; <code>false</code> otherwise
		 */
		boolean isNullAllowed();

		/**
		 * Sets the value of the node.
		 *
		 * @param form The external form that will set the value
		 * @return The value, which can be <code>null</code>
		 */
		void setValue(FORM form, VALUE value);

		/**
		 * Converts the value to its string representation.
		 *
		 * @param value A non-<code>null</code> value to convert into a string
		 * @return The given value represented as a string
		 */
		String toString(VALUE value);
	}

	/**
	 * A <code>RootNodeTester</code> is the container of all the testers that will test every single
	 * property of the node to test, i.e. its attributes and child nodes.
	 */
	public final class RootNodeTester {

		/**
		 * The list of testers that used to test every single attribute
		 * nodes of the one represented by this tester.
		 */
		private List<AttributeNodeTesterRunner> attributes;

		/**
		 * The builder of the {@link ExternalForm} to test.
		 */
		private ExternalFormBuilder<FORM> builder;

		/**
		 * The list of testers that used to test every single child nodes of the one represented by
		 * this tester.
		 */
		private List<NodeTesterRunner> children;

		/**
		 * Creates a new <code>RootNodeTester</code>.
		 */
		private RootNodeTester() {
			super();
			attributes = new ArrayList<AttributeNodeTesterRunner>();
			children   = new ArrayList<NodeTesterRunner>();
		}

		/**
		 * Adds the given tester when the form has an attribute.
		 *
		 * @param tester The tester for a single attribute
		 */
		public void addAttribute(AttributeNodeTester<FORM, ?> tester) {
			attributes.add(new AttributeNodeTesterRunner(tester));
		}

		/**
		 * Adds the given tester when the form representing the node it represents can have one child
		 * node of a certain type and its property is represented by a boolean value.
		 *
		 * @param tester The tester for a single child node
		 */
		public void addBooleanNode(BooleanChildNodeTester<FORM> tester) {
			children.add(new BooleanChildNodeTesterRunner(tester));
		}

		/**
		 * Adds the given tester when the form has a child node with a single attribute.
		 *
		 * @param tester The tester for a single attribute owned by a single child node
		 */
		public void addChildAttribute(ChildAttributeTester<FORM,?> tester) {
			children.add(new ChildAttributeTesterRunner(tester));
		}

		/**
		 * Adds the given list of testers when the form has a single child node from a list of choices.
		 *
		 * @param testers The testers for each of the child nodes when only one can be added
		 */
		public void addChoiceNodes(ChildNodeTester<?, ?>[] testers) {
			children.add(new ChoiceChildNodeTesterRunner(testers));
		}

		/**
		 * Adds the given tester when the form representing the node it represents can have zero or
		 * many child nodes of the same type (i.e. with the same node name).
		 *
		 * @param tester The tester for a list of child nodes with the same type
		 */
		public void addContainerListNodes(ContainerChildListNodeTester<FORM, ?, ?> tester) {
			children.add(new ContainerChildListNodeTesterRunner(tester));
		}

		/**
		 * Adds the given tester when the form representing the node it represents can have zero or
		 * many child nodes of the same type (i.e. with the same node name).
		 *
		 * @param tester The tester for a list of child nodes with the same type
		 */
		public void addListNodes(ChildListNodeTester<FORM, ?, ?> tester) {
			children.add(new ChildListNodeTesterRunner(tester));
		}

		/**
		 * Adds the given tester when the form representing the node it represents can have one child
		 * node of a certain type.
		 *
		 * @param tester The tester for a single child node
		 */
		public void addNode(ChildNodeTester<FORM, ?> tester) {
			children.add(new ChildNodeTesterRunner(tester));
		}

		/**
		 * Adds the given tester when the form has a single child node that is a text node.
		 *
		 * @param tester The tester for a single child text node
		 */
		public void addTextNode(TextNodeTester<FORM, ?> tester) {
			children.add(new TextNodeTesterRunner(tester));
		}

		/**
		 * Adds the given attribute name to indicate it is part of the "root" node but it is not
		 * supported by the {@link ExternalForm} yet.
		 *
		 * @param attributeName The name of the attribute not currently supported
		 */
		public void addUnsupportedAttribute(String attributeName) {
			attributes.add(new NotSupportedAttributeTesterRunner(attributeName));
		}

		/**
		 * Adds the given node name to indicate it is part of the "root" node but it is not supported
		 * by the {@link ExternalForm} yet.
		 *
		 * @param nodeName The name of the node not currently supported
		 */
		public void addUnsupportedNode(String nodeName) {
			children.add(new NotSupportedNodeTesterRunner(nodeName));
		}

		@SuppressWarnings({"unchecked", "rawtypes"})
		private Map<Class<NodeTesterRunner>, Class<Controller>[]> buildAttributeControllerTypes() {

			Map classes = new HashMap();

			classes.put(
				AttributeNodeTesterRunner.class,
				new Class<?>[] {
					AttributeNodeRunnerAddingController.class,
					AttributeNodeRunnerReadingController.class,
					AttributeNodeRunnerRemovingController.class
				}
			);

			return classes;
		}

		private NodeCountRetriever buildAttributeCountRetriever() {
			return new NodeCountRetriever() {
				@Override
				public int getCount(Node node) {
					return node.getAttributes().getLength();
				}
			};
		}

		@SuppressWarnings({"unchecked", "rawtypes"})
		private Map<Class<NodeTesterRunner>, Class<Controller>[]> buildChildrenControllerTypes() {

			Map classes = new HashMap();

			classes.put(
				ChildAttributeTesterRunner.class,
				new Class<?>[] {
					ChildAttributeRunnerAddingController.class,
					ChildAttributeRunnerReadingController.class,
					ChildAttributeRunnerRemovingController.class
				}
			);

			classes.put(
				BooleanChildNodeTesterRunner.class,
				new Class<?>[] {
					BooleanChildNodeRunnerAddingController.class,
					BooleanChildNodeRunnerReadingController.class,
					BooleanChildNodeRunnerRemovingController.class
				}
			);

			classes.put(
				ChildNodeTesterRunner.class,
				new Class<?>[] {
					ChildNodeRunnerAddingController.class,
					ChildNodeRunnerReadingController.class,
					ChildNodeRunnerRemovingController.class
				}
			);

			classes.put(
				TextNodeTesterRunner.class,
				new Class<?>[] {
					TextNodeRunnerAddingController.class,
					TextNodeRunnerReadingController.class,
					TextNodeRunnerRemovingController.class
				}
			);

			classes.put(
				ChildListNodeTesterRunner.class,
				new Class<?>[] {
					ChildListNodeRunnerAddingController.class,
					ChildListNodeRunnerReadingController.class,
					ChildListNodeRunnerRemovingController.class
				}
			);

			classes.put(
				ContainerChildListNodeTesterRunner.class,
				new Class<?>[] {
					ContainerChildListNodeRunnerAddingController.class,
					ContainerChildListNodeRunnerReadingController.class,
					ContainerChildListNodeRunnerRemovingController.class
				}
			);

			classes.put(
				ChoiceChildNodeTesterRunner.class,
				new Class<?>[] {
					ChoiceChildNodeRunnerAddingController.class,
					ChoiceChildNodeRunnerReadingController.class,
					ChoiceChildNodeRunnerRemovingController.class
				}
			);

			return classes;
		}

		private NodeCountRetriever buildChildrenCountRetriever() {
			return new NodeCountRetriever() {
				@Override
				public int getCount(Node node) {
					return node.getChildNodes().getLength();
				}
			};
		}

		private ExternalFormHolder buildExternalFormHolder() {
			return new ExternalFormHolder() {
				@Override
				void rebuild() {
					form = ObjectTools.execute(form, "getParent");
					form = builder.buildExternalForm(form);
					node = ObjectTools.execute(form, "getElement");
				}
			};
		}

		private String displayString() {
			return "<" + builder.getNodeName() + ">";
		}

		private Node getNode(Node node) {

			// The builder is for the root node
			if (node.getNodeName().equals(builder.getNodeName())) {
				return node;
			}

			// Dig into the DOM tree to find the child node
			return getNode(node, builder.getTreeNodeNames());
		}

		private Node getNode(Node node, List<String> nodeNames) {

			String nodeName = nodeNames.get(0);
			node = node.getFirstChild();

			while (node != null) {

				if (node.getNodeName().equals(nodeName)) {

					if (nodeNames.size() == 1) {
						return node;
					}

					return getNode(node, nodeNames.subList(1, nodeNames.size()));
				}

				node = node.getNextSibling();
			}

			return null;
		}

		private int getNodePositionOfInsertion(Node node) {

			String nodeName = node.getNodeName();

			for (int index = 0, count = children.size(); index < count; index++) {

				NodeTesterRunner runner = children.get(index);

				if (runner.getActualNodeName().equals(nodeName)) {
					return index;
				}
			}

			fail("The child node named <" + nodeName + "> is not included into the test");
			return -1;
		}

		/**
		 * Sets the builder that will create the document and tree node representation down to node to test.
		 *
		 * @param builder The builder of the node to test
		 */
		@SuppressWarnings("unchecked")
		public void setBuilder(ExternalFormBuilder<? extends FORM> builder) {
			this.builder = (ExternalFormBuilder<FORM>) builder;
		}

		/**
		 * Tests the {@link ExternalForm}, which might have attributes and child nodes.
		 */
		void test() throws Exception {

			// Creates the ExternalForm
			FORM form = builder.buildExternalForm();

			assertNotNull(
				displayString() + " : The external form cannot be null",
				form
			);

			// Retrieve the node represented by the ExternalForm
			Node node = builder.getNode(form);

			assertNotNull(
				displayString() + " : The node cannot be null",
				node
			);

			// Retrieve the node directly from the document
			Node expectedNode = getNode(node.getOwnerDocument());

			assertSame(
				displayString() + " : The node was not retrieved correctly",
				expectedNode,
				node
			);

			ExternalFormHolder holder = buildExternalFormHolder();
			holder.form = form;
			holder.node = node;

			// Test each attribute
			testAttributes(holder);

			// Test each child node
			testChildren(holder);

			// Now test manipulating the attributes in random order
			testAttributesRandomly(holder);

			// Now test manipulating the child nodes in random order
			testChildNodesRandomly(holder);
		}

		private void testAttributes(ExternalFormHolder holder) {

			int defaultAttributeCount = builder.getDefaultAttributeCount();

			for (AttributeNodeTesterRunner runner : attributes) {

				runner.currentChildrenCount = defaultAttributeCount;
				runner.parentNodeName = builder.getNodeName();
				runner.test(holder);

				// The node was deleted when testing with a null value,
				// make sure the node is rebuilt before runner the next tester
				if (runner.isNodeDeletedWithNullValue()) {
					holder.rebuild();
				}

				// Make sure the number of attributes didn't change after the test
				assertEquals(
					displayString() + " : The number of attributes is inconsistent",
					defaultAttributeCount,
					holder.node.getAttributes().getLength()
				);

				// Make sure the number of children didn't change after the test
				assertEquals(
					displayString() + " : The number of children is inconsistent",
					0,
					holder.node.getChildNodes().getLength()
				);
			}
		}

		private void testAttributesRandomly(ExternalFormHolder holder) throws Exception {
			testNodesRandomly(
				holder,
				attributes,
				buildAttributeCountRetriever(),
				buildAttributeControllerTypes()
			);
		}

		private void testChildNodesRandomly(ExternalFormHolder holder) throws Exception {
			testNodesRandomly(
				holder,
				children,
				buildChildrenCountRetriever(),
				buildChildrenControllerTypes()
			);
		}

		private void testChildren(ExternalFormHolder holder) {

			for (NodeTesterRunner runner : children) {

				runner.parentNodeName = builder.getNodeName();
				runner.test(holder);

				assertEquals(
					displayString() + " : No child nodes should have been left after a test",
					0,
					holder.node.getChildNodes().getLength()
				);

				if (runner.isNodeDeletedWithNullValue()) {
					holder.rebuild();
				}
			}
		}

		private void testNodesRandomly(ExternalFormHolder holder,
		                               List<? extends NodeTesterRunner> runners,
		                               NodeCountRetriever countRetriever,
		                               Map<Class<NodeTesterRunner>, Class<Controller>[]> controllerClasses) throws Exception {

			// The list of controllers is used to execute a single test for a particular runner.
			// A list will be generated randomly to test the order
			List<Controller> controllers = new ArrayList<Controller>();
			Random random = new Random();

			int childrenCount = runners.size();
			int controllerTypeCount = controllerClasses.size();
			int executionCount = childrenCount * controllerTypeCount * 20;

			// Let's run each Runner something like 20 times
			for (int index = 0; index < executionCount; index++) {

				// Get the runner randomly
				int childIndex = random.nextInt(childrenCount);
				NodeTesterRunner runner = runners.get(childIndex);

				// Get the controllers associated with the runner
				Class<Controller>[] controllerTypes = controllerClasses.get(runner.getClass());

				// Happen for unsupported node name
				if (controllerTypes == null) {
					continue;
				}

				// Get the controller type randomly
				int controllerIndex = random.nextInt(controllerTypes.length);
				Class<Controller> controllerType = controllerTypes[controllerIndex];

				// Create the controller
				Controller controller = ClassTools.newInstance(
					controllerType,
					AbstractExternalFormTests.class,
					AbstractExternalFormTests.this
				);

				controller.runner = runner;
				controllers.add(controller);
			}

			// Run each controller and keep the child nodes count up to date
			for (Controller controller : controllers) {

				controller.currentChildrenCount = countRetriever.getCount(holder.node);
				//controller.log();
				controller.test(holder);

				// Test the ordinal of each child node
				testOrdinalPosition(holder.node);
			}

			// Note: The runners don't need to be disposed since there are not used anymore
		}

		private void testOrdinalPosition(Node parent) {

			Node childNode = parent.getFirstChild();

			if (childNode != null) {

				List<Integer> nodePositions = new ArrayList<Integer>();

				do {
					int nodePosition = getNodePositionOfInsertion(childNode);
					nodePositions.add(nodePosition);
					childNode = childNode.getNextSibling();
				}
				while (childNode != null);

				// Make sure the ordinal numbers are from the smaller number to the biggest number
				int previousPosition = -1;

				for (int nodePosition : nodePositions) {

					if (previousPosition > nodePosition) {
						fail("The insertion was not performed following the ordering.");
					}

					previousPosition = nodePosition;
				}
			}
		}
	}

	/**
	 * This controller simply asks its runner to add the child text node.
	 */
	private class TextNodeRunnerAddingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		void log(Logger logger) {
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.tester.getNodeName() + "> : addition");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			TextNodeTesterRunner runner = (TextNodeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testAdding(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * This controller simply asks its runner to read the child node.
	 */
	private class TextNodeRunnerReadingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		void log(Logger logger) {
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.tester.getNodeName() + "> addition");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			TextNodeTesterRunner runner = (TextNodeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testReading(holder);
		}
	}

	/**
	 * This controller simply asks its runner to remove the child node.
	 */
	private class TextNodeRunnerRemovingController extends Controller {

		/**
		 * {@inheritDoc}
		 */
		@Override
		void log(Logger logger) {
			logger.log(Level.FINE, "<" + runner.parentNodeName + "><" + runner.tester.getNodeName() + "> : addition");
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		@SuppressWarnings("unchecked")
		void test(ExternalFormHolder holder) {

			TextNodeTesterRunner runner = (TextNodeTesterRunner) this.runner;
			runner.currentChildrenCount = currentChildrenCount;
			runner.testInitialState(holder);
			runner.testRemoving(holder);

			currentChildrenCount = runner.currentChildrenCount;
		}
	}

	/**
	 * A <code>TextNodeTester</code> tests setting and retrieving the value associated with a text node.
	 *
	 * <div nowrap>Form: <code><b>&lt;node_name&gt;text&lt;/node_name&gt;</b></code>.</div>
	 */
	public interface TextNodeTester<FORM extends ExternalForm, VALUE> extends PropertyTester<FORM, VALUE> {
	}

	/**
	 * The runner associated with {@link TextNodeTester}.
	 */
	private class TextNodeTesterRunner extends PropertyNodeTesterRunner {

		/**
		 * Creates a new <code>TextNodeTesterRunner</code>.
		 *
		 * @param tester This object defines a single node to test a child text node
		 */
		TextNodeTesterRunner(TextNodeTester<FORM, ?> tester) {
			super(tester);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		String displayString() {
			return "<" + parentNodeName + "><" + getNodeName() + "></>";
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		int getChildrenCount(Node parentNode) {
			return parentNode.getChildNodes().getLength();
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		Node getNode(Node node) {

			node = node.getFirstChild();

			while (node != null) {

				if (getNodeName().equals(node.getNodeName())) {
					return node;
				}

				node = node.getNextSibling();
			}

			return null;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		String getNodeValue(Node node) {
			return node.getTextContent();
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		boolean isNodeDeletedWithNullValue() {
			return ((TextNodeTester<?, ?>) tester).isNodeDeletedWithNullValue();
		}
	}
}