/*
 * Copyright (c) 1998, 2021 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */

// Contributors:
// dmccann - October 15/2009 - 2.0 - Initial implementation
package org.eclipse.persistence.testing.jaxb.externalizedmetadata.xmllist;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;

import org.eclipse.persistence.jaxb.JAXBContext;
import org.eclipse.persistence.jaxb.JAXBContextFactory;
import org.eclipse.persistence.jaxb.JAXBContextProperties;
import org.eclipse.persistence.testing.jaxb.JAXBWithJSONTestCases;

/**
 * Tests XmlList via eclipselink-oxm.xml
 *
 */
public class XmlListTestCases extends JAXBWithJSONTestCases {
    private static final String CONTEXT_PATH = "org.eclipse.persistence.testing.jaxb.externalizedmetadata.xmllist";
    private static final String XML_RESOURCE = "org/eclipse/persistence/testing/jaxb/externalizedmetadata/xmllist/employee.xml";
    private static final String JSON_RESOURCE = "org/eclipse/persistence/testing/jaxb/externalizedmetadata/xmllist/employee.json";

    /**
     * This is the preferred (and only) constructor.
     *
     */
    public XmlListTestCases(String name) throws Exception{
        super(name);
        setControlDocument(XML_RESOURCE);
        setControlJSON(JSON_RESOURCE);
        setContextPath(CONTEXT_PATH);
    }

    @Override
    public Map getProperties(){
        InputStream inputStream = ClassLoader.getSystemResourceAsStream("org/eclipse/persistence/testing/jaxb/externalizedmetadata/xmllist/eclipselink-oxm.xml");
        HashMap<String, Source> metadataSourceMap = new HashMap<String, Source>();
        metadataSourceMap.put("org.eclipse.persistence.testing.jaxb.externalizedmetadata.xmllist", new StreamSource(inputStream));
        Map<String, Map<String, Source>> properties = new HashMap<String, Map<String, Source>>();
        properties.put(JAXBContextProperties.OXM_METADATA_SOURCE, metadataSourceMap);

        return properties;
    }

    public void testSchemaGen() throws Exception{
        List controlSchemas = new ArrayList();
        InputStream is = ClassLoader.getSystemResourceAsStream("org/eclipse/persistence/testing/jaxb/externalizedmetadata/xmllist/schema.xsd");
        controlSchemas.add(is);

        super.testSchemaGen(controlSchemas);

        InputStream schemaInputStream = ClassLoader.getSystemResourceAsStream("org/eclipse/persistence/testing/jaxb/externalizedmetadata/xmllist/schema.xsd");
        InputStream controlDocStream = ClassLoader.getSystemResourceAsStream(XML_RESOURCE);
        validateAgainstSchema(controlDocStream, new StreamSource(schemaInputStream));


    }

    /**
     * Tests exception handling for @XmlList via eclipselink-oxm.xml.
     * This test should cause an exception as xml-list can only be used
     * with a collection or array, and in this case it is being set on
     * a String property.
     *
     * Negative test.
     */
    public void testXmlListInvalid() {

        InputStream inputStream = ClassLoader.getSystemResourceAsStream("org/eclipse/persistence/testing/jaxb/externalizedmetadata/xmllist/eclipselink-oxm-invalid.xml");
        HashMap<String, Source> metadataSourceMap = new HashMap<>();
        metadataSourceMap.put("org.eclipse.persistence.testing.jaxb.externalizedmetadata.xmllist", new StreamSource(inputStream));
        Map<String, Object> invalidProperties = new HashMap<>();
        invalidProperties.put(JAXBContextProperties.OXM_METADATA_SOURCE, metadataSourceMap);

        boolean exception = false;

        try {
            JAXBContext jaxbContext = (JAXBContext) JAXBContextFactory.createContext(CONTEXT_PATH, getClass().getClassLoader(), invalidProperties);
            MySchemaOutputResolver msor = new MySchemaOutputResolver();
            jaxbContext.generateSchema(msor);
        } catch (Exception ex) {
            exception = true;
        }
        assertTrue("The expected exception was not thrown", exception);

    }

    @Override
    protected Object getControlObject() {

        Employee emp = new Employee();
        java.util.List<String> data = new ArrayList<String>();
        data.add("xxx");
        data.add("yyy");
        data.add("zzz");
        emp.data = data;
        return emp;

    }
}
