/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.linuxtools.tmf.core.statesystem.backend.historytree;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.util.Vector;
import org.eclipse.linuxtools.tmf.core.statesystem.TimeRangeException;
import org.eclipse.linuxtools.tmf.core.statesystem.backend.historytree.CoreNode;
import org.eclipse.linuxtools.tmf.core.statesystem.backend.historytree.HTConfig;
import org.eclipse.linuxtools.tmf.core.statesystem.backend.historytree.HTInterval;
import org.eclipse.linuxtools.tmf.core.statesystem.backend.historytree.HTNode;
import org.eclipse.linuxtools.tmf.core.statesystem.backend.historytree.HT_IO;

class HistoryTree {
    private static final int HISTORY_FILE_MAGIC_NUMBER = 100641024;
    private static final int MAJOR_VERSION = 3;
    private static final byte MINOR_VERSION = 0;
    protected final HTConfig config;
    private final HT_IO treeIO;
    private long treeEnd;
    private int nodeCount;
    protected Vector<CoreNode> latestBranch;
    private int curDepth;

    private HistoryTree(HTConfig conf) throws IOException {
        assert (conf.blockSize >= HistoryTree.getTreeHeaderSize());
        this.config = conf;
        this.treeEnd = conf.treeStart;
        this.nodeCount = 0;
        this.latestBranch = new Vector();
        this.treeIO = new HT_IO(this, true);
        CoreNode firstNode = this.initNewCoreNode(-1, conf.treeStart);
        this.latestBranch.add(firstNode);
    }

    HistoryTree(File newStateFile, int blockSize, int maxChildren, long startTime) throws IOException {
        this(new HTConfig(newStateFile, blockSize, maxChildren, startTime));
    }

    HistoryTree(File existingStateFile) throws IOException {
        if (!existingStateFile.exists() || existingStateFile.length() <= 0L) {
            throw new IOException("Invalid state file selected");
        }
        FileInputStream fis = new FileInputStream(existingStateFile);
        ByteBuffer buffer = ByteBuffer.allocate(HistoryTree.getTreeHeaderSize());
        FileChannel fc = fis.getChannel();
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        buffer.clear();
        fc.read(buffer);
        buffer.flip();
        int res = buffer.getInt();
        if (res != 100641024) {
            throw new IOException("Selected file does not look like a History Tree file");
        }
        res = buffer.getInt();
        if (res != 3) {
            throw new IOException("Select History Tree file is of an older format. Please use a previous version of the parser to open it.");
        }
        res = buffer.getInt();
        int bs = buffer.getInt();
        int maxc = buffer.getInt();
        this.nodeCount = buffer.getInt();
        int rootNodeSeqNb = buffer.getInt();
        fc.position((long)HistoryTree.getTreeHeaderSize() + (long)rootNodeSeqNb * (long)bs);
        buffer = ByteBuffer.allocate(8);
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        buffer.clear();
        res = fc.read(buffer);
        assert (res == 8);
        buffer.flip();
        long ts = buffer.getLong();
        this.config = new HTConfig(existingStateFile, bs, maxc, ts);
        fis.close();
        this.treeIO = new HT_IO(this, false);
        this.rebuildLatestBranch(rootNodeSeqNb);
        ts = this.latestBranch.firstElement().getNodeStart();
        this.treeEnd = this.latestBranch.firstElement().getNodeEnd();
    }

    void closeTree() {
        int i = 0;
        while (i < this.latestBranch.size()) {
            this.latestBranch.get(i).closeThisNode(this.treeEnd);
            this.treeIO.writeNode(this.latestBranch.get(i));
            ++i;
        }
        FileChannel fc = this.treeIO.getFcOut();
        ByteBuffer buffer = ByteBuffer.allocate(HistoryTree.getTreeHeaderSize());
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        buffer.clear();
        try {
            fc.position(0L);
            buffer.putInt(100641024);
            buffer.putInt(3);
            buffer.putInt(0);
            buffer.putInt(this.config.blockSize);
            buffer.putInt(this.config.maxChildren);
            buffer.putInt(this.nodeCount);
            buffer.putInt(this.latestBranch.firstElement().getSequenceNumber());
            buffer.flip();
            int res = fc.write(buffer);
            assert (res <= HistoryTree.getTreeHeaderSize());
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    long getTreeStart() {
        return this.config.treeStart;
    }

    long getTreeEnd() {
        return this.treeEnd;
    }

    int getNodeCount() {
        return this.nodeCount;
    }

    HT_IO getTreeIO() {
        return this.treeIO;
    }

    private void rebuildLatestBranch(int rootNodeSeqNb) {
        this.latestBranch = new Vector();
        HTNode nextChildNode = this.treeIO.readNodeFromDisk(rootNodeSeqNb);
        this.latestBranch.add((CoreNode)nextChildNode);
        while (this.latestBranch.lastElement().getNbChildren() > 0) {
            nextChildNode = this.treeIO.readNodeFromDisk(this.latestBranch.lastElement().getLatestChild());
            this.latestBranch.add((CoreNode)nextChildNode);
        }
    }

    void insertInterval(HTInterval interval) throws TimeRangeException {
        if (interval.getStartTime() < this.config.treeStart) {
            throw new TimeRangeException();
        }
        this.tryInsertAtNode(interval, this.latestBranch.size() - 1);
    }

    private void tryInsertAtNode(HTInterval interval, int indexOfNode) {
        HTNode targetNode = this.latestBranch.get(indexOfNode);
        if (interval.getIntervalSize() > targetNode.getNodeFreeSpace()) {
            this.addSiblingNode(indexOfNode);
            this.tryInsertAtNode(interval, this.latestBranch.size() - 1);
            return;
        }
        if (interval.getStartTime() < targetNode.getNodeStart()) {
            assert (indexOfNode >= 1);
            this.tryInsertAtNode(interval, indexOfNode - 1);
            return;
        }
        targetNode.addInterval(interval);
        if (interval.getEndTime() > this.treeEnd) {
            this.treeEnd = interval.getEndTime();
        }
    }

    private void addSiblingNode(int indexOfNode) {
        long splitTime = this.treeEnd;
        assert (indexOfNode < this.latestBranch.size());
        if (indexOfNode == 0) {
            this.addNewRootNode();
            return;
        }
        if (this.latestBranch.get(indexOfNode - 1).getNbChildren() == this.config.maxChildren) {
            this.addSiblingNode(indexOfNode - 1);
            return;
        }
        int i = indexOfNode;
        while (i < this.latestBranch.size()) {
            this.latestBranch.get(i).closeThisNode(splitTime);
            this.treeIO.writeNode(this.latestBranch.get(i));
            CoreNode prevNode = this.latestBranch.get(i - 1);
            CoreNode newNode = this.initNewCoreNode(prevNode.getSequenceNumber(), splitTime + 1L);
            prevNode.linkNewChild(newNode);
            this.latestBranch.set(i, newNode);
            ++i;
        }
    }

    private void addNewRootNode() {
        long splitTime = this.treeEnd;
        CoreNode oldRootNode = this.latestBranch.firstElement();
        CoreNode newRootNode = this.initNewCoreNode(-1, this.config.treeStart);
        oldRootNode.setParentSequenceNumber(newRootNode.getSequenceNumber());
        int i = 0;
        while (i < this.latestBranch.size()) {
            this.latestBranch.get(i).closeThisNode(splitTime);
            this.treeIO.writeNode(this.latestBranch.get(i));
            ++i;
        }
        newRootNode.linkNewChild(oldRootNode);
        int depth = this.latestBranch.size();
        this.latestBranch = new Vector();
        this.latestBranch.add(newRootNode);
        i = 1;
        while (i < depth + 1) {
            CoreNode prevNode = this.latestBranch.get(i - 1);
            CoreNode newNode = this.initNewCoreNode(prevNode.getParentSequenceNumber(), splitTime + 1L);
            prevNode.linkNewChild(newNode);
            this.latestBranch.add(newNode);
            ++i;
        }
    }

    private CoreNode initNewCoreNode(int parentSeqNumber, long startTime) {
        CoreNode newNode = new CoreNode(this, this.nodeCount, parentSeqNumber, startTime);
        ++this.nodeCount;
        if (startTime >= this.treeEnd) {
            this.treeEnd = startTime + 1L;
        }
        return newNode;
    }

    HTNode selectNextChild(CoreNode currentNode, long t) {
        assert (currentNode.getNbChildren() > 0);
        int potentialNextSeqNb = currentNode.getSequenceNumber();
        int i = 0;
        while (i < currentNode.getNbChildren()) {
            if (t < currentNode.getChildStart(i)) break;
            potentialNextSeqNb = currentNode.getChild(i);
            ++i;
        }
        assert (potentialNextSeqNb != currentNode.getSequenceNumber());
        if (currentNode.isDone()) {
            return this.treeIO.readNodeFromDisk(potentialNextSeqNb);
        }
        return this.treeIO.readNode(potentialNextSeqNb);
    }

    static int getTreeHeaderSize() {
        return 4096;
    }

    long getFileSize() {
        return this.config.stateFile.length();
    }

    boolean checkNodeIntegrity(HTNode zenode) {
        HTNode otherNode;
        String message = "";
        boolean ret = true;
        if (!(zenode instanceof CoreNode)) {
            return true;
        }
        CoreNode node = (CoreNode)zenode;
        if (node.getNbChildren() > 0) {
            otherNode = this.treeIO.readNode(node.getChild(0));
            if (node.getNodeStart() != otherNode.getNodeStart()) {
                message = String.valueOf(message) + "Start time of node (" + node.getNodeStart() + ") " + "does not match start time of first child " + "(" + otherNode.getNodeStart() + "), " + "node #" + otherNode.getSequenceNumber() + ")\n";
                ret = false;
            }
            if (node.isDone()) {
                otherNode = this.treeIO.readNode(node.getLatestChild());
                if (node.getNodeEnd() != otherNode.getNodeEnd()) {
                    message = String.valueOf(message) + "End time of node (" + node.getNodeEnd() + ") does not match end time of last child (" + otherNode.getNodeEnd() + ", node #" + otherNode.getSequenceNumber() + ")\n";
                    ret = false;
                }
            }
        }
        int i = 0;
        while (i < node.getNbChildren()) {
            otherNode = this.treeIO.readNode(node.getChild(i));
            if (otherNode.getNodeStart() != node.getChildStart(i)) {
                message = String.valueOf(message) + "  Expected start time of child node #" + node.getChild(i) + ": " + node.getChildStart(i) + "\n" + "  Actual start time of node #" + otherNode.getSequenceNumber() + ": " + otherNode.getNodeStart() + "\n";
                ret = false;
            }
            ++i;
        }
        if (!ret) {
            System.out.println("");
            System.out.println("SHT: Integrity check failed for node #" + node.getSequenceNumber() + ":");
            System.out.println(message);
        }
        return ret;
    }

    void checkIntegrity() {
        int i = 0;
        while (i < this.nodeCount) {
            this.checkNodeIntegrity(this.treeIO.readNode(i));
            ++i;
        }
    }

    public String toString() {
        return "Information on the current tree:\n\nBlocksize: " + this.config.blockSize + "\n" + "Max nb. of children per node: " + this.config.maxChildren + "\n" + "Number of nodes: " + this.nodeCount + "\n" + "Depth of the tree: " + this.latestBranch.size() + "\n" + "Size of the treefile: " + this.getFileSize() + "\n" + "Root node has sequence number: " + this.latestBranch.firstElement().getSequenceNumber() + "\n" + "'Latest leaf' has sequence number: " + this.latestBranch.lastElement().getSequenceNumber();
    }

    private void preOrderPrint(PrintWriter writer, boolean printIntervals, CoreNode currentNode) {
        writer.println(currentNode.toString());
        if (printIntervals) {
            currentNode.debugPrintIntervals(writer);
        }
        ++this.curDepth;
        int i = 0;
        while (i < currentNode.getNbChildren()) {
            HTNode nextNode = this.treeIO.readNode(currentNode.getChild(i));
            assert (nextNode instanceof CoreNode);
            int j = 0;
            while (j < this.curDepth - 1) {
                writer.print("  ");
                ++j;
            }
            writer.print("+-");
            this.preOrderPrint(writer, printIntervals, (CoreNode)nextNode);
            ++i;
        }
        --this.curDepth;
    }

    void debugPrintFullTree(PrintWriter writer, boolean printIntervals) {
        this.curDepth = 0;
        this.preOrderPrint(writer, false, this.latestBranch.firstElement());
        if (printIntervals) {
            writer.println("\nDetails of intervals:");
            this.curDepth = 0;
            this.preOrderPrint(writer, true, this.latestBranch.firstElement());
        }
        writer.println('\n');
    }
}

