/********************************************************************************
 * Copyright (c) 2007 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the terms
 * of the Eclipse Public License v1.0 which accompanies this distribution, and is 
 * available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Initial Contributors:
 * The following IBM employees contributed to the Remote System Explorer
 * component that contains this file: David McKnight.
 * 
 * Contributors:
 * Martin Oberhuber (Wind River) - [168975] Move RSE Events API to Core
 * Martin Oberhuber (Wind River) - [186773] split ISystemRegistryUI from ISystemRegistry
 * Martin Oberhuber (Wind River) - [207100] Events for after a resource is downloaded and uploaded
 ********************************************************************************/
package org.eclipsecon.tmtutorial.eventconsole;

import java.io.OutputStream;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.rse.core.RSECorePlugin;
import org.eclipse.rse.core.events.ISystemRemoteChangeEvent;
import org.eclipse.rse.core.events.ISystemRemoteChangeEvents;
import org.eclipse.rse.core.events.ISystemRemoteChangeListener;
import org.eclipse.rse.core.events.ISystemResourceChangeEvent;
import org.eclipse.rse.core.events.ISystemResourceChangeEvents;
import org.eclipse.rse.core.events.ISystemResourceChangeListener;
import org.eclipse.rse.core.model.ISystemRegistry;
import org.eclipse.rse.ui.model.ISystemShellProvider;
import org.eclipse.rse.ui.view.ISystemViewElementAdapter;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IStartup;
import org.eclipse.ui.console.ConsolePlugin;
import org.eclipse.ui.console.IConsole;
import org.eclipse.ui.console.MessageConsole;
import org.eclipse.ui.plugin.AbstractUIPlugin;

public class RSEEventLogging implements ISystemRemoteChangeListener, ISystemResourceChangeListener,
	ISystemShellProvider, IStartup
{
	private OutputStream _loggingStream;

	public RSEEventLogging()
	{		
	}
	
	/**
	 * implementation of IStartup.earlyStartup().  This starts the RSE event logger.
	 */
	public void earlyStartup() 
	{
		startLogging();
	}
	/**
	 * This gets called to start listening to RSE events.
	 */
	public void startLogging()
	{
		_loggingStream = getLoggingStream();

		ISystemRegistry registry = RSECorePlugin.getTheSystemRegistry();
		registry.addSystemResourceChangeListener(this);
		registry.addSystemRemoteChangeListener(this);				
	}
	
	
	/**
	 * To get the console logging stream.  We will use this to write RSE events to the console.
	 * @return an output stream to the console
	 */
	private OutputStream getLoggingStream()
	{
		MessageConsole messageConsole=null;
		
		IConsole[] consoles = ConsolePlugin.getDefault().getConsoleManager().getConsoles();
		for (int i = 0; i < consoles.length; i++) {
			if(consoles[i].getName().equals("RSE Event Log")) { //$NON-NLS-1$ 
				messageConsole = (MessageConsole)consoles[i];
				break;
			}	
		}
		
		if(messageConsole==null)
		{
			ImageDescriptor imageDescriptor = AbstractUIPlugin.imageDescriptorFromPlugin("org.eclipsecon.tmtutorial", "icons/view16/system_view.gif"); //$NON-NLS-1$ //$NON-NLS-2$
			messageConsole = new MessageConsole("RSE Event Log", imageDescriptor); //$NON-NLS-1$
			ConsolePlugin.getDefault().getConsoleManager().addConsoles(new IConsole[]{ messageConsole });
		}
		
		ConsolePlugin.getDefault().getConsoleManager().showConsoleView(messageConsole);
		
		return messageConsole.newOutputStream();
	}
	
	
	public Shell getShell() 
	{
		return null;
	}

	/**
	 * Writes a string-represented event to the console
	 * @param event a string representing an event
	 */
	private void logEvent(String event)
	{		
		try
		{
			_loggingStream.write(event.getBytes());
			_loggingStream.write("\n".getBytes()); //$NON-NLS-1$
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
	}
	
	
	/**
	 * Gets called when a remote resource change event occurs in RSE
	 */
	public void systemRemoteResourceChanged(ISystemRemoteChangeEvent event) 
	{
		int type = event.getEventType();
		String eventStr = getRemoteResourceEventType(type);
		
		String resource = getResourceName(event.getResource());
		String message = eventStr + ":\t" + resource; //$NON-NLS-1$
		logEvent(message);		
	}
	
	/**
	 * Gets called when a resource change event occurs in RSE (not necessarily remote)
	 */
	public void systemResourceChanged(ISystemResourceChangeEvent event) 
	{
		int type = event.getType();
		String eventStr = getResourceChangeEventType(type); //int to String
		String resource = getResourceName(event.getSource());
		String message = eventStr + ":\t" + resource; //$NON-NLS-1$
		logEvent(message); //print into Console; could also do stdout
	}


	/**
	 * Gets the name of an RSE resource for display purposes
	 * @param resource the RSE resource
	 * @return the name of the RSE resource
	 */
	private String getResourceName(Object resource)
	{
		if (resource instanceof IAdaptable)
		{ 
			ISystemViewElementAdapter adapter = (ISystemViewElementAdapter)((IAdaptable)resource).getAdapter(ISystemViewElementAdapter.class);
			if (adapter != null)
			{   
				String type = adapter.getType(resource);
				String name = adapter.getName(resource);
				return "(" + type + ")\t" + name; //$NON-NLS-1$ //$NON-NLS-2$
			}
		}
		return resource.toString();
	}
	
	/**
	 * Gets the string that describes a particular remote resource event type
	 * @param type an integer indicating the event type
	 * @return the string that describes the event
	 */
	private String getRemoteResourceEventType(int type)
	{	
		String message;
		switch (type)
		{
		case ISystemRemoteChangeEvents.SYSTEM_REMOTE_RESOURCE_CREATED:
			message = "SYSTEM_REMOTE_RESOURCE_CREATED"; //$NON-NLS-1$
			break;
		case ISystemRemoteChangeEvents.SYSTEM_REMOTE_RESOURCE_CHANGED:
			message = "SYSTEM_REMOTE_RESOURCE_CHANGED"; //$NON-NLS-1$
			break;
		case ISystemRemoteChangeEvents.SYSTEM_REMOTE_RESOURCE_DELETED:
			message = "SYSTEM_REMOTE_RESOURCE_DELETED"; //$NON-NLS-1$
			break;
		case ISystemRemoteChangeEvents.SYSTEM_REMOTE_RESOURCE_RENAMED:
			message = "SYSTEM_REMOTE_RESOURCE_RENAMED"; //$NON-NLS-1$
			break;
		case ISystemRemoteChangeEvents.SYSTEM_REMOTE_RESOURCE_DOWNLOADED:
			message = "SYSTEM_REMOTE_RESOURCE_DOWNLOADED"; //$NON-NLS-1$
			break;
		case ISystemRemoteChangeEvents.SYSTEM_REMOTE_RESOURCE_UPLOADED:
			message = "SYSTEM_REMOTE_RESOURCE_UPLOADED"; //$NON-NLS-1$
			break;
		default:
			message = "(Unknown_Remote_event_type:"+type+")";
			break;
		}
		return message;
	}
	
	/**
	 * Gets the string that describes a particular resource change event type
	 * @param type an integer indicating the event type
	 * @return the string that describes the event
	 */
	private String getResourceChangeEventType(int type)
	{	
		String message;
		switch (type)
		{
		case ISystemResourceChangeEvents.EVENT_ADD_FILTER_REFERENCE:
			message = "EVENT_ADD_FILTER_REFERENCE"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_RENAME_FILTER_REFERENCE:
			message = "EVENT_RENAME_FILTER_REFERENCE"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_DELETE_FILTER_REFERENCE:		
			message = "EVENT_DELETE_FILTER_REFERENCE"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_CHANGE_FILTER_REFERENCE:
			message = "EVENT_CHANGE_FILTER_REFERENCE"; //$NON-NLS-1$
			break;
	
		case ISystemResourceChangeEvents.EVENT_MOVE_FILTER_REFERENCES:
			message = "EVENT_MOVE_FILTER_REFERENCES"; //$NON-NLS-1$
			break;
	
		case ISystemResourceChangeEvents.EVENT_ADD_FILTERSTRING_REFERENCE:
			message = "EVENT_ADD_FILTERSTRING_REFERENCE"; //$NON-NLS-1$
			break;
	
		case ISystemResourceChangeEvents.EVENT_DELETE_FILTERSTRING_REFERENCE:	
			message = "EVENT_DELETE_FILTERSTRING_REFERENCE"; //$NON-NLS-1$
			break;
	
		case ISystemResourceChangeEvents.EVENT_CHANGE_FILTERSTRING_REFERENCE:
			message = "EVENT_CHANGE_FILTERSTRING_REFERENCE"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_MOVE_FILTERSTRING_REFERENCES :
			message = "EVENT_MOVE_FILTERSTRING_REFERENCES"; //$NON-NLS-1$
			break;
			
		case ISystemResourceChangeEvents.EVENT_ADD:
			message = "EVENT_ADD"; //$NON-NLS-1$
			break;
		
		case ISystemResourceChangeEvents.EVENT_ADD_MANY:
			message = "EVENT_ADD_MANY"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_ADD_RELATIVE :
			message = "EVENT_ADD_RELATIVE"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_REVEAL_AND_SELECT:
			message = "EVENT_REVEAL_AND_SELECT"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_DELETE:
			message = "EVENT_DELETE"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_DELETE_MANY:
			message = "EVENT_DELETE_MANY"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_RENAME:
			message = "EVENT_RENAME"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_MOVE_MANY:
			message = "EVENT_MOVE_MANY"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_ICON_CHANGE:
			message = "EVENT_ICON_CHANGE"; //$NON-NLS-1$
			break;
	
		case ISystemResourceChangeEvents.EVENT_REFRESH:
			message = "EVENT_REFRESH"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_REFRESH_SELECTED:
			message = "(@deprecated) EVENT_REFRESH_SELECTED"; //$NON-NLS-1$
			break;
		
		case ISystemResourceChangeEvents.EVENT_REFRESH_SELECTED_PARENT:
			message = "(@deprecated) EVENT_REFRESH_SELECTED_PARENT"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_REFRESH_SELECTED_FILTER:
			message = "(@deprecated) EVENT_REFRESH_SELECTED_FILTER"; //$NON-NLS-1$
			break;
			
		case ISystemResourceChangeEvents.EVENT_REFRESH_REMOTE:
			message = "EVENT_REFRESH_REMOTE"; //$NON-NLS-1$
			break;
	
		case ISystemResourceChangeEvents.EVENT_PROPERTY_CHANGE:
			message = "EVENT_PROPERTY_CHANGE"; //$NON-NLS-1$
			break;
			
		case ISystemResourceChangeEvents.EVENT_PROPERTYSHEET_UPDATE:
			message = "EVENT_PROPERTYSHEET_UPDATE"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_MUST_COLLAPSE:
			message = "EVENT_MUST_COLLAPSE"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_COLLAPSE_ALL:	
			message = "EVENT_COLLAPSE_ALL"; //$NON-NLS-1$
			break;
		
		case ISystemResourceChangeEvents.EVENT_COLLAPSE_SELECTED:
			message = "EVENT_COLLAPSE_SELECTED"; //$NON-NLS-1$
			break;
		
		case ISystemResourceChangeEvents.EVENT_EXPAND_SELECTED:
			message = "EVENT_EXPAND_SELECTED"; //$NON-NLS-1$
			break;
		
		case ISystemResourceChangeEvents.EVENT_CHANGE_CHILDREN:
			message = "EVENT_CHANGE_CHILDREN"; //$NON-NLS-1$
			break;
		
		case ISystemResourceChangeEvents.EVENT_SELECT:
			message = "EVENT_SELECT"; //$NON-NLS-1$
			break;
		
		case ISystemResourceChangeEvents.EVENT_SELECT_REMOTE:
			message = "EVENT_SELECT_REMOTE"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_SELECT_EXPAND:
			message = "EVENT_SELECT_EXPAND"; //$NON-NLS-1$
			break;
		
		case ISystemResourceChangeEvents.EVENT_COMMAND_RUN:
			message = "EVENT_COMMAND_RUN"; //$NON-NLS-1$
			break;
		
		case ISystemResourceChangeEvents.EVENT_COMMAND_MESSAGE:
			message = "EVENT_COMMAND_MESSAGE"; //$NON-NLS-1$
			break;
		
		case ISystemResourceChangeEvents.EVENT_REPLACE_CHILDREN:
			message = "EVENT_REPLACE_CHILDREN"; //$NON-NLS-1$
			break;
		
		case ISystemResourceChangeEvents.EVENT_COMPILE_COMMAND_RUN:
			message = "(@deprecated) EVENT_COMPILE_COMMAND_RUN"; //$NON-NLS-1$
			break;
		
		case ISystemResourceChangeEvents.EVENT_COMMAND_HISTORY_UPDATE:
			message = "EVENT_COMMAND_HISTORY_UPDATE"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_COMMAND_SHELL_FINISHED:
			message = "EVENT_COMMAND_SHELL_FINISHED"; //$NON-NLS-1$
			break;
		
		case ISystemResourceChangeEvents.EVENT_COMMAND_SHELL_REMOVED:
			message = "EVENT_SEARCH_FINISHED"; //$NON-NLS-1$
			break;

		case ISystemResourceChangeEvents.EVENT_SEARCH_FINISHED:
			message = "EVENT_SEARCH_FINISHED"; //$NON-NLS-1$
			break;
		default:
			message = "(Unknown_Resource_event_type:"+type+")";
			break;
		}
		return message;
	}

}
