/*
 * Decompiled with CFR 0.152.
 */
package com.google.javascript.refactoring;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.javascript.jscomp.AbstractCompiler;
import com.google.javascript.jscomp.JSError;
import com.google.javascript.jscomp.NodeTraversal;
import com.google.javascript.jscomp.NodeUtil;
import com.google.javascript.refactoring.Match;
import com.google.javascript.refactoring.NodeMetadata;
import com.google.javascript.refactoring.SuggestedFix;
import com.google.javascript.rhino.JSDocInfo;
import com.google.javascript.rhino.Node;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class ErrorToFixMapper {
    private static final Pattern DID_YOU_MEAN = Pattern.compile(".*Did you mean (.*)\\?");
    private static final Pattern MISSING_REQUIRE = Pattern.compile("'([^']+)' used but not required");
    private static final Pattern EXTRA_REQUIRE = Pattern.compile("'([^']+)' required but not used");

    private ErrorToFixMapper() {
    }

    public static List<SuggestedFix> getFixesForJsError(JSError error, AbstractCompiler compiler) {
        SuggestedFix fix = ErrorToFixMapper.getFixForJsError(error, compiler);
        if (fix != null) {
            return ImmutableList.of((Object)fix);
        }
        switch (error.getType().key) {
            case "JSC_IMPLICITLY_NULLABLE_JSDOC": {
                return ErrorToFixMapper.getFixesForImplicitlyNullableJsDoc(error);
            }
        }
        return ImmutableList.of();
    }

    public static SuggestedFix getFixForJsError(JSError error, AbstractCompiler compiler) {
        switch (error.getType().key) {
            case "JSC_REQUIRES_NOT_SORTED": {
                return ErrorToFixMapper.getFixForUnsortedRequiresOrProvides("goog.require", error, compiler);
            }
            case "JSC_PROVIDES_NOT_SORTED": {
                return ErrorToFixMapper.getFixForUnsortedRequiresOrProvides("goog.provide", error, compiler);
            }
            case "JSC_DEBUGGER_STATEMENT_PRESENT": 
            case "JSC_USELESS_EMPTY_STATEMENT": {
                return ErrorToFixMapper.removeNode(error);
            }
            case "JSC_INEXISTENT_PROPERTY": {
                return ErrorToFixMapper.getFixForInexistentProperty(error);
            }
            case "JSC_MISSING_REQUIRE_WARNING": {
                return ErrorToFixMapper.getFixForMissingRequire(error, compiler);
            }
            case "JSC_DUPLICATE_REQUIRE_WARNING": 
            case "JSC_EXTRA_REQUIRE_WARNING": {
                return ErrorToFixMapper.getFixForExtraRequire(error, compiler);
            }
            case "JSC_UNNECESSARY_CAST": {
                return ErrorToFixMapper.getFixForUnnecessaryCast(error, compiler);
            }
        }
        return null;
    }

    private static List<SuggestedFix> getFixesForImplicitlyNullableJsDoc(JSError error) {
        SuggestedFix qmark = new SuggestedFix.Builder().setOriginalMatchedNode(error.node).insertBefore(error.node, "?").setDescription("Make nullability explicit").build();
        SuggestedFix bang = new SuggestedFix.Builder().setOriginalMatchedNode(error.node).insertBefore(error.node, "!").setDescription("Make type non-nullable").build();
        return ImmutableList.of((Object)qmark, (Object)bang);
    }

    private static SuggestedFix removeNode(JSError error) {
        return new SuggestedFix.Builder().setOriginalMatchedNode(error.node).delete(error.node).build();
    }

    private static SuggestedFix getFixForInexistentProperty(JSError error) {
        Matcher m = DID_YOU_MEAN.matcher(error.description);
        if (m.matches()) {
            String suggestedPropName = m.group(1);
            return new SuggestedFix.Builder().setOriginalMatchedNode(error.node).rename(error.node, suggestedPropName).build();
        }
        return null;
    }

    private static SuggestedFix getFixForMissingRequire(JSError error, AbstractCompiler compiler) {
        Matcher regexMatcher = MISSING_REQUIRE.matcher(error.description);
        Preconditions.checkState((boolean)regexMatcher.matches(), (String)"Unexpected error description: %s", (Object[])new Object[]{error.description});
        String namespaceToRequire = regexMatcher.group(1);
        NodeMetadata metadata = new NodeMetadata(compiler);
        Match match = new Match(error.node, metadata);
        return new SuggestedFix.Builder().setOriginalMatchedNode(error.node).addGoogRequire(match, namespaceToRequire).build();
    }

    private static SuggestedFix getFixForExtraRequire(JSError error, AbstractCompiler compiler) {
        Matcher regexMatcher = EXTRA_REQUIRE.matcher(error.description);
        Preconditions.checkState((boolean)regexMatcher.matches(), (String)"Unexpected error description: %s", (Object[])new Object[]{error.description});
        String namespace = regexMatcher.group(1);
        NodeMetadata metadata = new NodeMetadata(compiler);
        Match match = new Match(error.node, metadata);
        return new SuggestedFix.Builder().setOriginalMatchedNode(error.node).removeGoogRequire(match, namespace).build();
    }

    private static SuggestedFix getFixForUnnecessaryCast(JSError error, AbstractCompiler compiler) {
        return new SuggestedFix.Builder().setOriginalMatchedNode(error.node).removeCast(error.node, compiler).build();
    }

    private static SuggestedFix getFixForUnsortedRequiresOrProvides(String closureFunction, JSError error, AbstractCompiler compiler) {
        SuggestedFix.Builder fix = new SuggestedFix.Builder();
        fix.setOriginalMatchedNode(error.node);
        Node script = NodeUtil.getEnclosingScript(error.node);
        RequireProvideSorter cb = new RequireProvideSorter(closureFunction);
        NodeTraversal.traverseEs6(compiler, script, cb);
        Node first = (Node)cb.calls.get(0);
        Node last = (Node)cb.calls.get(cb.calls.size() - 1);
        cb.sortCallsAlphabetically();
        StringBuilder sb = new StringBuilder();
        for (Node n : cb.calls) {
            String statement = fix.generateCode(compiler, n);
            JSDocInfo jsDoc = NodeUtil.getBestJSDocInfo(n);
            if (jsDoc != null) {
                statement = jsDoc.getOriginalCommentString() + "\n" + statement;
            }
            sb.append(statement);
        }
        String newContent = sb.toString().trim();
        return fix.replaceRange(first, last, newContent).build();
    }

    private static String getNamespaceFromClosureNode(Node exprResult) {
        Preconditions.checkState((boolean)exprResult.isExprResult());
        return exprResult.getFirstChild().getLastChild().getString();
    }

    private static class RequireProvideSorter
    extends NodeTraversal.AbstractShallowCallback
    implements Comparator<Node> {
        private final String closureFunction;
        private final List<Node> calls = new ArrayList<Node>();

        RequireProvideSorter(String closureFunction) {
            this.closureFunction = closureFunction;
        }

        @Override
        public final void visit(NodeTraversal nodeTraversal, Node n, Node parent) {
            if (n.isCall() && parent.isExprResult() && n.getFirstChild().matchesQualifiedName(this.closureFunction)) {
                this.calls.add(parent);
            }
        }

        public void sortCallsAlphabetically() {
            Collections.sort(this.calls, this);
        }

        @Override
        public int compare(Node n1, Node n2) {
            String namespace1 = ErrorToFixMapper.getNamespaceFromClosureNode(n1);
            String namespace2 = ErrorToFixMapper.getNamespaceFromClosureNode(n2);
            return namespace1.compareTo(namespace2);
        }
    }
}

