// kshutdown.h - KShutdown base library
// Copyright (C) 2007  Konrad Twardowski
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

#pragma once

#include "infowidget.h"
#include "plugins.h"

#include <QAction>
#include <QCheckBox>
#include <QDateTimeEdit>
#include <QHBoxLayout>

#ifdef QT_DBUS_LIB
	#include <QDBusInterface>
	#include <QDBusReply>
#endif // QT_DBUS_LIB

#ifndef Q_OS_WIN32
	// HACK: fixes some compilation error (?)
	#include <sys/types.h>
#endif // !Q_OS_WIN32

class BootEntryComboBox;
class MainWindow;

typedef int UShutdownType;

class PowerAction: public Action {
public:
	explicit PowerAction(const QString &text, const QString &iconName, const QString &id);
	#ifdef QT_DBUS_LIB
	static QDBusInterface *getHalDeviceInterface();
	static QDBusInterface *getHalDeviceSystemPMInterface();
	static QDBusInterface *getUPowerInterface();
	#endif // QT_DBUS_LIB
	virtual void initContainerWidget() final override { }
	virtual bool onAction() final override;
	enum class PowerActionType { Suspend, Hibernate };
protected:
	QString m_methodName = QString();
	static bool isAvailable(const PowerActionType feature);
private:
	Q_DISABLE_COPY(PowerAction)
	#ifdef QT_DBUS_LIB
	inline static QDBusInterface *m_halDeviceInterface = nullptr;
	inline static QDBusInterface *m_halDeviceSystemPMInterface = nullptr;
	inline static QDBusInterface *m_upowerInterface = nullptr;
	#endif // QT_DBUS_LIB
};

class HibernateAction final: public PowerAction {
public:
	explicit HibernateAction();
private:
	Q_DISABLE_COPY(HibernateAction)
};

class SuspendAction final: public PowerAction {
public:
	explicit SuspendAction();
private:
	Q_DISABLE_COPY(SuspendAction)
};

class ExitAction: public Action {
public:
	// NOTE: This is mapped to the values in kworkspace.h
	inline static const UShutdownType U_SHUTDOWN_TYPE_LOGOUT = 0; // KDE: ShutdownTypeNone
	inline static const UShutdownType U_SHUTDOWN_TYPE_REBOOT = 1; // KDE: ShutdownTypeReboot
	inline static const UShutdownType U_SHUTDOWN_TYPE_HALT   = 2; // KDE: ShutdownTypeHalt

	explicit ExitAction(const QString &text, const QString &iconName, const QString &id, const UShutdownType type);
	virtual void initContainerWidget() override;
	virtual bool onAction() override;
protected:
	QCheckBox *m_forceCheckBox;
	#ifdef QT_DBUS_LIB
	inline static QDBusInterface *m_consoleKitInterface = nullptr;
	inline static QDBusInterface *m_kdeSessionInterface = nullptr;
	inline static QDBusInterface *m_kdeShutdownInterface = nullptr;
	void checkAvailable(const QString &consoleKitName);
	#endif // QT_DBUS_LIB
private:
	Q_DISABLE_COPY(ExitAction)
	#ifdef QT_DBUS_LIB
	inline static bool m_kdeShutDownAvailable = false;
	#endif // QT_DBUS_LIB
	UShutdownType m_type;
};

class LogoutAction final: public ExitAction {
public:
	explicit LogoutAction();
private:
	Q_DISABLE_COPY(LogoutAction)
};

class RebootAction final: public ExitAction {
public:
	explicit RebootAction();
	virtual void initContainerWidget() override;
	virtual bool onAction() override;
	virtual bool onCommandLineOption() override;
	static RebootAction *self() { return static_cast<RebootAction *>(PluginManager::action("reboot")); }
	void updateBootEntryView();
private:
	Q_DISABLE_COPY(RebootAction)
	BootEntryComboBox *m_bootEntryComboBox = nullptr;
	#ifdef Q_OS_LINUX
	QCommandLineOption systemOption;
	#endif // Q_OS_LINUX
};

class ShutDownAction final: public ExitAction {
public:
	explicit ShutDownAction();
private:
	Q_DISABLE_COPY(ShutDownAction)
};
